package org.maachang.crypto ;

import java.security.MessageDigest;

/**
 * RSA暗号、復号処理.
 *
 * @version 2012/08/09
 * @author  masahito suzuki
 * @since   Crypto 1.00
 */
public abstract class CryptoRsa {
    private static final CryptoAse ase = new CryptoAse() ;
    private static final String HATENA = "?" ;
    private static final String SIG_SM = "::52cee64bb3a38f6403386519a39ac91c::" ;
    private static final String CHARSET = "UTF8" ;
    private static final String GENERATE_CODE = "03" ;
    
    protected static final byte[] blockXOR( byte[] a,byte[] b,byte[] ret ) {
        if( ret == null ) {
            ret = new byte[ 16 ] ;
        }
        for( int i = 0; i < 16; i++ ) {
            ret[i] = (byte)( (a[i]&0x000000ff) ^ (b[i]&0x000000ff) );
        }
        return ret ;
    }
    
    protected static final byte[] blockIV( byte[] ret )
        throws Exception {
        if( ret == null ) {
            ret = new byte[ 16 ] ;
        }
        CryptoRsaKey.getRand().nextBytes( ret ) ;
        return ret ;
    }
    
    protected static final byte[] pad16( byte[] block ) {
        int len = block.length ;
        int padding = ( 16 - ( len & 0x0000000f ) ) & 0x0000000f ;
        int lenLen = len + padding ;
        byte[] newBytes = new byte[ lenLen ] ;
        System.arraycopy( block,0,newBytes,0,len ) ;
        return newBytes ;
    }
    
    protected static final byte[] depad( byte[] bytes ) {
        int len = bytes.length ;
        int p = -1 ;
        for( int i = len-1 ; i >= 0 ; i -- ) {
            if( bytes[ i ] != 0 ) {
                p = i ;
                byte[] ret = new byte[ p+1 ] ;
                System.arraycopy( bytes,0,ret,0,p+1 ) ;
                return ret ;
            }
        }
        return bytes ;
    }
    
    protected static final String bytes2string( byte[] bin ) {
        int len = bin.length ;
        StringBuilder buf = new StringBuilder( bin.length ) ;
        for( int i = 0 ; i < len ; i ++ ) {
            buf.append( ( char )( bin[ i ] & 0x000000ff ) ) ;
        }
        return buf.toString() ;
    }
    
    protected static final byte[] string2bytes( String string ) {
        int len = string.length() ;
        byte[] ret = new byte[ len ] ;
        for( int i = 0 ; i < len ; i ++ ) {
            ret[ i ] = (byte)( string.charAt( i ) & 0x000000ff ) ;
        }
        return ret ;
    }
    
    protected static final byte[] cloneBytes( byte[] bin ) {
        byte[] ret = new byte[ bin.length ] ;
        System.arraycopy( bin,0,ret,0,bin.length ) ;
        return ret ;
    }
    
    protected static final byte[] cutBytes( byte[] src,int p,int e,byte[] ret ) {
        int len = e-p ;
        if( ret == null ) {
            ret = new byte[ len ] ;
        }
        System.arraycopy( src,p,ret,0,len ) ;
        return ret ;
    }
    
    protected static final byte[] addBytes( byte[] src,byte[] addBin ) {
        int len = src.length + addBin.length ;
        byte[] ret = new byte[ len ] ;
        System.arraycopy( src,0,ret,0,src.length ) ;
        System.arraycopy( addBin,0,ret,src.length,addBin.length ) ;
        return ret ;
    }
    
    protected static final byte[] nextAddBytes( byte[] src,int pos,byte[] addBin ) {
        System.arraycopy( addBin,0,src,pos,addBin.length ) ;
        return src ;
    }
    
    protected static final String encryptAESCBC( byte[] blocks,byte[] key )
        throws Exception {
        byte[] exKey = cloneBytes( key ) ;
        exKey = ase.expandKey( exKey ) ;
        blocks = pad16( blocks ) ;
        byte[] encryptedBlocks = blockIV( null ) ;
        int len = blocks.length ;
        int loopLen = len / 16 ;
        
        byte[] tmp = new byte[ (loopLen * 16)+16 ] ;
        System.arraycopy( encryptedBlocks,0,tmp,0,encryptedBlocks.length ) ;
        encryptedBlocks = tmp ;
        tmp = null ;
        
        byte[] tmpBlock = new byte[ 16 ] ;
        byte[] previewBlock = new byte[ 16 ] ;
        int p = 16 ;
        
        for( int i = 0 ; i < loopLen ; i ++ ) {
            cutBytes( blocks,i*16,(i*16)+16,tmpBlock ) ;
            cutBytes( encryptedBlocks,i*16,(i*16)+16,previewBlock ) ;
            blockXOR( previewBlock,tmpBlock,tmpBlock ) ;
            ase.encrypt( tmpBlock,exKey ) ;
            nextAddBytes( encryptedBlocks,p,tmpBlock ) ;
            p += 16 ;
        }
        return CryptoUtils_Base64.encode( encryptedBlocks ) ;
    }
    
    protected static final byte[] decryptAESCBC( String encryptedText,byte[] key )
        throws Exception {
        byte[] exKey = cloneBytes( key ) ;
        exKey = ase.expandKey( exKey ) ;
        
        byte[] encryptedBlocks = CryptoUtils_Base64.decode( encryptedText ) ;
        int len = encryptedBlocks.length ;
        int loopLen = len / 16 ;
        
        byte[] tmpBlock = new byte[ 16 ] ;
        byte[] previewBlock = new byte[ 16 ] ;
        CryptoUtils_BinaryBuffer decryptedBlocks = new CryptoUtils_BinaryBuffer() ;
        for( int i = 1 ; i < loopLen ; i ++ ) {
            cutBytes( encryptedBlocks,i*16,(i*16)+16,tmpBlock ) ;
            cutBytes( encryptedBlocks,(i-1)*16,((i-1)*16)+16,previewBlock ) ;
            ase.decrypt( tmpBlock,exKey ) ;
            blockXOR( previewBlock,tmpBlock,tmpBlock ) ;
            decryptedBlocks.write( tmpBlock ) ;
        }
        return depad( decryptedBlocks.getBinary() ) ;
    }
    
    protected static final byte[] generateAESKey( byte[] ret )
        throws Exception {
        if( ret == null ) {
            ret = new byte[ 32 ] ;
        }
        CryptoRsaKey.getRand().nextBytes( ret ) ;
        return ret ;
    }
    
    /**
     * RSAKeyを生成.
     * @param passphrase 対象のパスフレーズを設定します.
     * @param bits キーの長さ(Bit長)を指定します.
     * @return CryptoRsaKey 生成されたRSAKeyが返されます.
     * @exception Exception 例外.
     */
    public static final CryptoRsaKey generateRSAKey( String passphrase,int bits )
        throws Exception {
        if( passphrase == null || passphrase.length() <= 0 ) {
            throw new IllegalArgumentException( "passphraseが設定されていません" ) ;
        }
        if( bits <= 0 ) {
            bits = 512 ;
        }
        MessageDigest md = MessageDigest.getInstance( "SHA-256" ) ;
        md.reset();
        md.update( passphrase.getBytes( CHARSET ) ) ;
        CryptoRsaKey.getRand().setSeed( md.digest() ) ;
        CryptoRsaKey ret = new CryptoRsaKey() ;
        ret.generate( bits,GENERATE_CODE ) ;
        return ret ;
    }
    
    /**
     * 指定されたRSAKeyからパブリックキーを取得.
     * @param rsa 対象のRSAKeyを設定します.
     * @return String 対象のパブリックキーが返されます.
     * @exception Exception 例外.
     */
    public static final String publicKeyString( CryptoRsaKey rsa )
        throws Exception {
        if( rsa == null || rsa.n == null ) {
            throw new IllegalArgumentException( "RSAKeyが設定されていません" ) ;
        }
        return CryptoUtils_Base64.encode( rsa.n.toByteArray() ) ;
    }
    
    /**
     * パブリックキーをRSAKeyに変換.
     * @param string 対象のパブリックキーを文字列で設定します.
     * @return CryptoRsaKey 変換されたRSAKeyが返されます.
     * @exception Exception 例外.
     */
    public static final CryptoRsaKey publicKeyFromString( String string )
        throws Exception {
        if( string == null || string.length() <= 0 ) {
            throw new IllegalArgumentException( "publicKeyが設定されていません" ) ;
        }
        int p = string.indexOf( "|" ) ;
        if( p != -1 ) {
            string = string.substring( 0,p ) ;
        }
        CryptoRsaKey ret = new CryptoRsaKey() ;
        ret.setPublic( CryptoUtils_Base64.decode( string ),GENERATE_CODE ) ;
        return ret ;
    }
    
    /**
     * 暗号処理.
     * @param string 暗号化対象の文字列を設定します.
     * @param publicKey 対象のパブリックキーを設定します.
     * @param signingKey 署名キーを設定します. ※署名を設定しない場合はnullをセット.
     * @return String 暗号化された文字列が返されます.
     * @exception Exception 例外.
     */
    public static final String encrypt( String string,String publicKey,CryptoRsaKey signingKey )
        throws Exception {
        if( string == null || string.length() <= 0 ) {
            throw new IllegalArgumentException( "対象の文字列が設定されていません" ) ;
        }
        if( publicKey == null || publicKey.length() <= 0 ) {
            throw new IllegalArgumentException( "publicKeyが設定されていません" ) ;
        }
        byte[] aesKey = generateAESKey( null ) ;
        StringBuilder ret = new StringBuilder( string.length()*4 ) ;
        
        CryptoRsaKey publickey = publicKeyFromString( publicKey ) ;
        ret.append( CryptoUtils_Base64.encode( publickey.encrypt( bytes2string( aesKey ) ) ) ).append( HATENA ) ;
        
        byte[] binary ;
        if( signingKey != null ) {
            // signingKey認証(署名).
            String signString = CryptoUtils_Base64.encode( signingKey.sign( string.getBytes( CHARSET ),"sha256" ).toByteArray() ) ;
            string = new StringBuilder().append( string ).
                append( SIG_SM ).
                append( publicKeyString( signingKey ) ).
                append( SIG_SM ).
                append( signString ).toString() ;
            binary = string.getBytes( CHARSET ) ;
        }
        else {
            binary = string.getBytes( CHARSET ) ;
        }
        
        ret.append( encryptAESCBC( binary,aesKey ) ) ;
        return ret.toString() ;
    }
    
    /**
     * 復号化処理.
     * @param string 暗号化された情報を設定します.
     * @param privateKey プライベートキーを設定します.
     * @param signingKey 署名キーを設定します. ※署名を設定しない場合はnullをセット.
     * @return String 復号化された情報が返却されます.
     * @exception Exception 例外.
     */
    public static final String decrypt( String string,CryptoRsaKey privateKey,CryptoRsaKey signingKey )
        throws Exception {
        if( string == null || string.length() <= 0 ) {
            throw new IllegalArgumentException( "対象の文字列が設定されていません" ) ;
        }
        if( privateKey == null ) {
            throw new IllegalArgumentException( "privateKeyが設定されていません" ) ;
        }
        String[] cipherBlock = string.split( "\\?" ) ;
        String tmpKey = privateKey.decrypt( CryptoUtils_Base64.decode( cipherBlock[ 0 ] ) ) ;
        if( tmpKey == null ) {
            throw new CryptoRsaFailureException( "復号処理に失敗" ) ;
        }
        byte[] aesKey = string2bytes( tmpKey ) ;
        tmpKey = null ;
        
        String[] plainText = ( new String( decryptAESCBC( cipherBlock[ 1 ],aesKey ),CHARSET ) ).split( SIG_SM ) ;
        if( plainText.length == 3 ) {
            // signingKey認証(署名).
            if( signingKey != null ) {
                CryptoRsaKey publickey = publicKeyFromString( plainText[ 1 ] ) ;
                byte[] signature = CryptoUtils_Base64.decode( plainText[ 2 ] ) ;
                // 署名チェック.
                if( publickey.verify( plainText[ 0 ].getBytes( CHARSET ),signature ) ) {
                    // 署名内容が一致するかチェック.
                    if( publicKeyString( signingKey ).equals( publicKeyString( publickey ) ) ) {
                        // 一致する場合のみ、結果情報を返却.
                        return plainText[ 0 ] ;
                    }
                }
                throw new CryptoRsaNotSigningException( "署名処理に失敗" ) ;
            }
            else {
                throw new CryptoRsaNotSigningException( "署名処理に失敗" ) ;
            }
        }
        return plainText[ 0 ] ;
    }
}
