package org.seasar.framework.aop.proxy;

import java.lang.reflect.Method;

import net.sf.cglib.proxy.Callback;
import net.sf.cglib.proxy.CallbackFilter;
import net.sf.cglib.proxy.Enhancer;
import net.sf.cglib.proxy.MethodInterceptor;
import net.sf.cglib.proxy.MethodProxy;

import org.seasar.framework.aop.Aspect;
import org.seasar.framework.aop.Joinpoint;
import org.seasar.framework.aop.Pointcut;
import org.seasar.framework.aop.impl.JoinpointImpl;
import org.seasar.framework.aop.impl.PointcutImpl;
import org.seasar.framework.exception.EmptyRuntimeException;

/**
 * @author higa
 *
 * AspectKpProxy쐬܂B
 */
public final class AopProxy implements MethodInterceptor {

	private static final int AOP_PROXY_INDEX = 0;
	private static final int EQUALS_INTERCEPTOR_INDEX = 1;
	private Class targetClass_;
	private Pointcut defaultPointcut_;
	private Aspect[] aspects_;

	public AopProxy(Class targetClass, Aspect[] aspects)
		throws EmptyRuntimeException {

		if (targetClass == null) {
			throw new EmptyRuntimeException("targetClass");
		}

		setTargetClass(targetClass);
		setAspects(aspects);
	}
	
	private void setTargetClass(Class targetClass) {
		targetClass_ = targetClass;
		defaultPointcut_ = new PointcutImpl(targetClass);
	}

	private void setAspects(Aspect[] aspects) throws EmptyRuntimeException {
		if (aspects == null || aspects.length == 0) {
			throw new EmptyRuntimeException("aspects");
		}
		aspects_ = aspects;
		for (int i = 0; i < aspects.length; ++i) {
			Aspect aspect = aspects[i];
			if (aspect.getPointcut() == null) {
				aspect.setPointcut(defaultPointcut_);
			}
		}
	}

	public Object create() {
		Enhancer e = setupEnhancer();
		return e.create();
	}
	
	public Object create(Class[] argTypes, Object[] args) {
		Enhancer e = setupEnhancer();
		return e.create(argTypes, args);
	}
	
	private Enhancer setupEnhancer() {
		Enhancer e = new Enhancer();
		e.setSuperclass(targetClass_);
		e.setCallbacks(new Callback[] { this, new EqualsInterceptor()});
		e.setCallbackFilter(MyCallbackFilter.INSTANCE);
		return e;
	}

	/**
	 * @see net.sf.cglib.proxy.MethodInterceptor#intercept(java.lang.Object, java.lang.reflect.Method, java.lang.Object[], net.sf.cglib.proxy.MethodProxy)
	 */
	public Object intercept(
		Object obj,
		Method method,
		Object[] args,
		MethodProxy proxy)
		throws Throwable {

		Joinpoint joinpoint =
			new JoinpointImpl(
				obj,
				targetClass_,
				method,
				args,
				proxy,
				aspects_);
		return joinpoint.proceed();
	}

	private class EqualsInterceptor implements MethodInterceptor {

		/**
		 * @see net.sf.cglib.proxy.MethodInterceptor#intercept(java.lang.Object, java.lang.reflect.Method, java.lang.Object[], net.sf.cglib.proxy.MethodProxy)
		 */
		public Object intercept(
			Object obj,
			Method method,
			Object[] args,
			MethodProxy proxy)
			throws Throwable {

			if (args[0] == obj) {
				return Boolean.TRUE;
			} else {
				return Boolean.FALSE;
			}

		}

	}

	private static class MyCallbackFilter implements CallbackFilter {

		private static MyCallbackFilter INSTANCE = new MyCallbackFilter();

		private MyCallbackFilter() {
		}

		/**
		 * @see net.sf.cglib.proxy.CallbackFilter#accept(java.lang.reflect.Method)
		 */
		public int accept(Method method) {
			if (isEqualsMethod(method)) {
				return EQUALS_INTERCEPTOR_INDEX;
			} else {
				return AOP_PROXY_INDEX;
			}
		}

		private boolean isEqualsMethod(Method method) {
			return "equals".equals(method.getName())
				&& method.getParameterTypes().length == 1
				&& method.getParameterTypes()[0] == Object.class;
		}
	}
}