package org.seasar.framework.sel.tokenizer;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.Map;

import org.seasar.framework.sel.*;

public final class SelTokenizer {

	public final static int INVALID_CHAR = -1;
	public final static int OR = 1;
	public final static int AND = 2;
	public final static int NOT = 3;
	public final static int WORD = 4;
	public final static int QUOTED_STRING = 5;
	public final static int INTEGER = 6;
	public final static int LONG = 7;
	public final static int BIGDECIMAL = 8;
	public final static int LEFT_PAREN = 9;
	public final static int RIGHT_PAREN = 10;
	public final static int ADD = 11;
	public final static int SUBTRACT = 12;
	public final static int MULTIPLY = 13;
	public final static int DIVIDE = 14;
	public final static int MOD = 15;
	public final static int EQUAL = 16;
	public final static int NOT_EQUAL = 17;
	public final static int GREATER_THAN = 18;
	public final static int GREATER_EQUAL = 19;
	public final static int LESS_THAN = 20;
	public final static int LESS_EQUAL = 21;
	public final static int COMMA = 22;
	public final static int NEW = 23;
	public final static int FROM = 24;
	public final static int IN = 25;
	public final static int BETWEEN = 26;
	public final static int TO_TIMESTAMP = 27;
	public final static int TO_STRING = 28;
	public final static int FOR = 32;
	public final static int NOW = 34;
	public final static int TRUE = 35;
	public final static int FALSE = 36;
	public final static int IS = 38;
	public final static int NULL = 39;
	public final static int LIKE = 40;
	public final static int CONCATENATE = 42;
	public final static int LEFT_BRACKET = 49;
	public final static int RIGHT_BRACKET = 50;
	public final static int LEFT_BRACE = 51;
	public final static int RIGHT_BRACE = 52;
	public final static int WHEN = 58;
	public final static int THEN = 59;
	public final static int ELSE = 60;
	public final static int END = 61;
	public final static int ASC = 65;
	public final static int DESC = 66;
	public final static int COUNT = 67;
	public final static int SUM = 68;
	public final static int MAX = 69;
	public final static int MIN = 70;
	public final static int AVG = 71;
	public final static int AS = 72;
	public final static int DOT = 73;
	public final static int EOF = 99;

	private static Map _wordMap = new HashMap();
	private CoreTokenizer tokenizer_;

	static {
		setupWord();
	}

	public SelTokenizer(String str) {
		tokenizer_ = new CoreTokenizer(str);
	}

	public void expect(int expected, int real)
		throws UnexpectedTokenRuntimeException {

		if (expected != real) {
			throw new UnexpectedTokenRuntimeException(
				tokenizer_.getReadString(),
				getTokenName(expected),
				getTokenName(real));
		}
	}

	public int nextToken() throws UnexpectedTokenRuntimeException {
		switch (tokenizer_.nextToken()) {
			case CoreTokenizer.TT_EOF :
				return EOF;
			case CoreTokenizer.TT_WORD :
				Integer type =
					(Integer) _wordMap.get(tokenizer_.getStringValue().toLowerCase());
				if (type != null) {
					return type.intValue();
				} else {
					return WORD;
				}
			case CoreTokenizer.TT_INTEGER :
				return INTEGER;
			case CoreTokenizer.TT_LONG :
				return LONG;
			case CoreTokenizer.TT_DECIMAL :
				return BIGDECIMAL;
			case '\'' :
				return QUOTED_STRING;
			case '(' :
				return LEFT_PAREN;
			case ')' :
				return RIGHT_PAREN;
			case '+' :
				return ADD;
			case '-' :
				return SUBTRACT;
			case '/' :
				return DIVIDE;
			case '%' :
				return MOD;
			case ',' :
				return COMMA;
			case '.' :
				return DOT;
			case '[' :
				return LEFT_BRACKET;
			case ']' :
				return RIGHT_BRACKET;
			case '{' :
				return LEFT_BRACE;
			case '}' :
				return RIGHT_BRACE;
			default :
				throw new UnexpectedTokenRuntimeException(
					tokenizer_.getReadString(),
					getTokenName(tokenizer_.getTokenType()));
		}
	}

	public int nextToken(int t) throws UnexpectedTokenRuntimeException {
			
		int next = nextToken();
		expect(t, next);
		return next;
	}
	
	public String getReadString() {
		return tokenizer_.getReadString();
	}
	
	public Integer getIntegerValue() {
		return tokenizer_.getIntegerValue();
	}
	
	public Long getLongValue() {
		return tokenizer_.getLongValue();
	}
	
	public BigDecimal getBigDecimalValue() {
		return tokenizer_.getBigDecimalValue();
	}
	
	public String getStringValue() {
		return tokenizer_.getStringValue();
	}

	public static String getTokenName(int t) {
		switch (t) {
			case ADD :
				return "ADD";
			case SUBTRACT :
				return "SUBTRACT";
			case MULTIPLY :
				return "MULTIPLY";
			case DIVIDE :
				return "DIVIDE";
			case MOD :
				return "MOD";
			case LEFT_PAREN :
				return "LEFT_PAREN";
			case RIGHT_PAREN :
				return "RIGHT_PAREN";
			case LEFT_BRACKET :
				return "LEFT_BRACKET";
			case RIGHT_BRACKET :
				return "RIGHT_BRACKET";
			case LEFT_BRACE :
				return "LEFT_BRACE";
			case RIGHT_BRACE :
				return "RIGHT_BRACE";
			case AND :
				return "AND";
			case OR :
				return "OR";
			case NOT :
				return "NOT";
			case QUOTED_STRING :
				return "QUOTED_STRING";
			case EOF :
				return "EOF";
			case WORD :
				return "WORD";
			case INTEGER :
				return "INTEGER";
			case LONG :
				return "LONG";
			case BIGDECIMAL :
				return "DIGDECIMAL";
			case TO_TIMESTAMP :
				return "TO_TIMESTAMP";
			case TO_STRING :
				return "TO_STRING";
			case NOW :
				return "NOW";
			case EQUAL :
				return "EQUAL";
			case NOT_EQUAL :
				return "NOT_EQUAL";
			case GREATER_THAN :
				return "GREATER_THAN";
			case GREATER_EQUAL :
				return "GREATER_EQUAL";
			case LESS_THAN :
				return "LESS_THAN";
			case LESS_EQUAL :
				return "LESS_EQUAL";
			case COMMA :
				return "COMMA";
			case IN :
				return "IN";
			case BETWEEN :
				return "BETWEEN";
			case TRUE :
				return "TRUE";
			case FALSE :
				return "FALSE";
			case IS :
				return "IS";
			case NULL :
				return "NULL";
			case LIKE :
				return "LIKE";
			case FROM :
				return "FROM";
			case FOR :
				return "FOR";
			case CONCATENATE :
				return "CONCATENATE";
			case WHEN :
				return "WHEN";
			case THEN :
				return "THEN";
			case ELSE :
				return "ELSE";
			case END :
				return "END";
			case NEW :
				return "NEW";
			case ASC :
				return "ASC";
			case DESC :
				return "DESC";
			case COUNT :
				return "COUNT";
			case SUM :
				return "SUM";
			case MAX :
				return "MAX";
			case MIN :
				return "MIN";
			case AVG :
				return "AVG";
			case AS :
				return "AS";
			case DOT :
				return "DOT";
			default :
				return "Unknown(" + t + ")";
		}
	}

	private static void setupWord() {
		_wordMap.put("or", new Integer(OR));
		_wordMap.put("and", new Integer(AND));
		_wordMap.put("not", new Integer(NOT));
		_wordMap.put("=", new Integer(EQUAL));
		_wordMap.put("eq", new Integer(EQUAL));
		_wordMap.put("!=", new Integer(NOT_EQUAL));
		_wordMap.put("<>", new Integer(NOT_EQUAL));
		_wordMap.put("ne", new Integer(NOT_EQUAL));
		_wordMap.put(">", new Integer(GREATER_THAN));
		_wordMap.put("gt", new Integer(GREATER_THAN));
		_wordMap.put(">=", new Integer(GREATER_EQUAL));
		_wordMap.put("ge", new Integer(GREATER_EQUAL));
		_wordMap.put("<", new Integer(LESS_THAN));
		_wordMap.put("lt", new Integer(LESS_THAN));
		_wordMap.put("<=", new Integer(LESS_EQUAL));
		_wordMap.put("le", new Integer(LESS_EQUAL));
		_wordMap.put("||", new Integer(CONCATENATE));
		_wordMap.put("to_timestamp", new Integer(TO_TIMESTAMP));
		_wordMap.put("to_string", new Integer(TO_STRING));
		_wordMap.put("in", new Integer(IN));
		_wordMap.put("between", new Integer(BETWEEN));
		_wordMap.put("true", new Integer(TRUE));
		_wordMap.put("false", new Integer(FALSE));
		_wordMap.put("is", new Integer(IS));
		_wordMap.put("null", new Integer(NULL));
		_wordMap.put("like", new Integer(LIKE));
		_wordMap.put("from", new Integer(FROM));
		_wordMap.put("for", new Integer(FOR));
		_wordMap.put("now", new Integer(NOW));
		_wordMap.put("when", new Integer(WHEN));
		_wordMap.put("then", new Integer(THEN));
		_wordMap.put("else", new Integer(ELSE));
		_wordMap.put("end", new Integer(END));
		_wordMap.put("new", new Integer(NEW));
		_wordMap.put("*", new Integer(MULTIPLY));
		_wordMap.put("asc", new Integer(ASC));
		_wordMap.put("desc", new Integer(DESC));
		_wordMap.put("count", new Integer(COUNT));
		_wordMap.put("sum", new Integer(SUM));
		_wordMap.put("max", new Integer(MAX));
		_wordMap.put("min", new Integer(MIN));
		_wordMap.put("avg", new Integer(AVG));
		_wordMap.put("as", new Integer(AS));
	}
}