#!/usr/bin/perl
#
# Rest Interface of LISM
#
# This code was developped by SECIOSS (http://www.secioss.co.jp/).
#
#              Copyright (C) 2011 SECIOSS CORPORATION
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation.
#

use strict;
use LISM;
use Net::LDAP::Util qw(ldap_error_text);
use CGI qw(:standard);
use JSON;
use Config::General;
use MIME::Base64;
use Encode;
use Data::Dumper;

our $CONF = 'lismapi.conf';
my $SESSION_DIR = '/tmp';

sub config
{
    my $config = Config::General->new($CONF);
    my %conf = $config->getall;

    if (!defined($conf{'basedn'})) {
        $conf{'basedn'} = 'o=lism,o=cgi';
    }

    return %conf;
}

sub factory
{
    my (%conf) = @_;

    my $lism = new LISM;
    $lism->config('basedn', $conf{'basedn'});
    if (defined($conf{'admin'})) {
        $lism->config('admindn', "cn=$conf{'admin'},$conf{'basedn'}");
    }
    foreach my $key (qw(syncdir conf adminpw)) {
        $lism->config($key, $conf{$key});
    }

    if ($lism->init()) {
        return undef;
    }

    return $lism;
}

sub search
{
    my ($base, $scope, $sizeLim, $timeLim, $filter, $attrs) = @_;
    my @entries;
    my %conf = config();

    my $user = $ENV{REMOTE_USER};
    if (!$user) {
        return {code => -1, message => "Not authenticated"};
    }

    my $lism = factory(%conf);
    if (!defined($lism)) {
        return {-1, "Can't load LISM"};
    }
    $lism->{bind}{dn} = "uid=$user";

    $base =~ s/o=lism$/$conf{'basedn'}/i;
    $scope = defined($scope) ? $scope : 2;
    $filter = $filter ? $filter : '(objectClass=*)';
    $filter =~ s/o=lism\)/$conf{'basedn'})/gi;
    $filter =~ s/&amp;/&/g;

    my @objects;
    my ($rc, @entries) = $lism->search($base, $scope, 0, $sizeLim, $timeLim, $filter, 0, split(/,/, $attrs));
    for (my $i = 0; $i < @entries; $i++) {
        $entries[$i] =~ s/$conf{'basedn'}$/o=lism/gmi;
        $objects[$i] = {};
        foreach my $line (split(/\n/, $entries[$i])) {
            my ($attr, $value) = split(/: */, $line);
            if ($attr eq 'dn') {
                $objects[$i]{$attr} = $value;
            } else {
                if (!defined($objects[$i]{$attr})) {
                    $objects[$i]{$attr} = [];
                }
                push(@{$objects[$i]{$attr}}, $value);
            }
        }
    }

    return {code => $rc, message => ldap_error_text($rc), entries => \@objects};
}

sub add
{
    my ($dn, $json_req) = @_;
    my $entryStr;
    my %conf = config();

    my $user = $ENV{REMOTE_USER};
    if (!$user) {
        return {code => -1, message => "Not authenticated"};
    }

    $entryStr = "dn: $dn\n";
    my $req = decode_json($json_req);
    foreach my $attr (keys %{$req}) {
        my @values;
        if (ref($req->{$attr}) eq 'ARRAY') {
            @values = @{$req->{$attr}};
        } else {
            $values[0] = $req->{$attr};
        }
        for (my $i = 0; $i < @values; $i++) {
            $entryStr = "$entryStr$attr: $values[$i]\n";
        }
    }

    $entryStr =~ s/o=lism$/$conf{'basedn'}/gmi;

    my $lism = factory(%conf);
    if (!defined($lism)) {
        return {code => -1, message => "Can't load LISM"};
    }
    $lism->{bind}{dn} = "uid=$user";

    my $rc = $lism->add($entryStr);

    return {code => $rc, message => ldap_error_text($rc)};
}

sub modify
{
    my ($dn, $json_req) = @_;
    my @changes = ();
    my %conf = config();

    my $user = $ENV{REMOTE_USER};
    if (!$user) {
        return {code => -1, message => "Not authenticated"};
    }

    my $req = decode_json($json_req);
    foreach my $action (keys %{$req}) {
        if (ref($req->{$action}) eq 'ARRAY') {
            foreach my $info (@{$req->{$action}}) {
                foreach my $attr (keys %{$info}) {
                    my @values;
                    if (ref($info->{$attr}) eq 'ARRAY') {
                        foreach my $value (@{$info->{$attr}}) {
                            $value =~ s/o=lism$/$conf{'basedn'}/i;
                            push(@values, $value);
                        }
                    } else {
                        $values[0] = $info->{$attr};
                        $values[0] =~ s/o=lism$/$conf{'basedn'}/i;
                    }
                    push(@changes, uc($action), $attr, @values);
                }
            }
        } else {
            foreach my $attr (keys %{$req->{$action}}) {
                my @values;
                if (ref($req->{$action}->{$attr}) eq 'ARRAY') {
                    foreach my $value (@{$req->{$action}->{$attr}}) {
                        $value =~ s/o=lism$/$conf{'basedn'}/i;
                        push(@values, $value);
                    }
                } else {
                        $values[0] = $req->{$action}->{$attr};
                        $values[0] =~ s/o=lism$/$conf{'basedn'}/i;
                }
                push(@changes, uc($action), $attr, @values);
            }
        }
    }

    my $lism = factory(%conf);
    if (!defined($lism)) {
        return {code => -1, message => "Can't load LISM"};
    }
    $lism->{bind}{dn} = "uid=$user";

    $dn =~ s/o=lism$/$conf{'basedn'}/i;

    my $rc = $lism->modify($dn, @changes);

    return {code => $rc, message => ldap_error_text($rc)};
}

sub delete
{
    my ($dn) = @_;
    my %conf = config();

    my $user = $ENV{REMOTE_USER};
    if (!$user) {
        return {code => -1, message => "Not authenticated"};
    }

    my $lism = factory(%conf);
    if (!defined($lism)) {
        return {code => -1, message => "Can't load LISM"};
    }
    $lism->{bind}{dn} = "uid=$user";

    $dn =~ s/o=lism$/$conf{'basedn'}/i;

    my $rc = $lism->delete($dn);

    return {code => $rc, message => ldap_error_text($rc)};
}

my $q = CGI->new;
$q->charset('utf-8');
print $q->header();

my ($action) = ($ENV{QUERY_STRING} =~ /action=([^&]+)/);

my $res;
if ($action eq 'search') {
    $res = &search($q->param('base'), $q->param('scope'), $q->param('sizelimit'), $q->param('timelimit'), $q->param('filter'), $q->param('attrs'));
} elsif ($action eq 'add') {
    my ($dn) = ($ENV{QUERY_STRING} =~ /dn=([^&]+)/);
    $res = &add($dn, $q->param('POSTDATA'));
} elsif ($action eq 'modify') {
    my ($dn) = ($ENV{QUERY_STRING} =~ /dn=([^&]+)/);
    $res = &modify($dn, $q->param('POSTDATA'));
} elsif ($action eq 'delete') {
    my ($dn) = ($ENV{QUERY_STRING} =~ /dn=([^&]+)/);
    $res = &delete($dn);
} else {
    $res = {code => -1, message => "Invalid action"};
}

print encode_json($res);

1;
