/*
SRInfoCenterController.m

Author: Makoto Kinoshita

Copyright 2004 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRInfoCenterController.h"

#pragma mark -
//--------------------------------------------------------------//
// _SRHalfTransparentTextField class
//--------------------------------------------------------------//

static NSDictionary*    _defaultAttr = nil;
static NSDictionary*    _halfTransparentAttr = nil;

@interface _SRHalfTransparentTextField : NSTextField
{
    BOOL    _halfTransparent;
}

- (BOOL)halfTransparent;
- (void)setHalfTransparent:(BOOL)flag;

@end

@implementation _SRHalfTransparentTextField

- (id)initWithFrame:(NSRect)frame
{
    self = [super initWithFrame:frame];
    if (!self) {
        return nil;
    }
    
    // Create attribute
    if (!_defaultAttr) {
        _defaultAttr = [[NSDictionary dictionaryWithObjectsAndKeys:
                [NSFont systemFontOfSize:[NSFont smallSystemFontSize]], NSFontAttributeName, 
                [NSColor blackColor], NSForegroundColorAttributeName, 
                nil] retain];
        
        NSShadow*   shadow;
        shadow = [[NSShadow alloc] init];
        [shadow setShadowColor:[NSColor blackColor]];
        [shadow setShadowOffset:NSMakeSize(2.0, -2.0)];
        [shadow setShadowBlurRadius:2.0f];
        
        _halfTransparentAttr = [[NSDictionary dictionaryWithObjectsAndKeys:
                [NSFont systemFontOfSize:[NSFont smallSystemFontSize]], NSFontAttributeName, 
                [NSColor whiteColor], NSForegroundColorAttributeName, 
                shadow, NSShadowAttributeName, 
                nil] retain];
    }
    
    // Initialize member variables
    _halfTransparent = NO;
    
    return self;
}

- (BOOL)halfTransparent
{
    return _halfTransparent;
}

- (void)setHalfTransparent:(BOOL)flag
{
    if (_halfTransparent != flag) {
        _halfTransparent = flag;
        
        // Update attributed string value
        [self setStringValue:[self stringValue]];
        
        [self setNeedsDisplay:YES];
    }
}

- (void)setStringValue:(NSString*)string
{
    // Create attributed string
    NSDictionary*   attr = _defaultAttr;
    if (_halfTransparent) {
        attr = _halfTransparentAttr;
    }
    
    NSAttributedString* attrStr;
    attrStr = [[NSAttributedString alloc] initWithString:string attributes:attr];
    
    // Set attributed string
    [self setAttributedStringValue:attrStr];
}

@end

#pragma mark -
//--------------------------------------------------------------//
// SRInfoCenterItem class
//--------------------------------------------------------------//

static NSTimeInterval   _SRInfoCenterIntervalAutoRemove = 3; // seconds

@implementation SRInfoCenterItem

- (id)init
{
    self = [super init];
    if (!self) {
        return nil;
    }
    
    // Initialize member variables
    _autoRemoveFlag = NO;
    
    return self;
}

- (void)dealloc
{
    [_text release];
    [_buttons release];
    [_box release];
    [_textField release];
}

- (void)setInfoCenter:(SRInfoCenterController*)infoCenter
{
    _infoCenter = infoCenter;
}

- (SRInfoCenterController*)infoCenter
{
    return _infoCenter;
}

- (void)setText:(NSString*)text
{
    [_text release];
    _text = [text retain];
    
    // Update text field
    if (_textField) {
        [_textField setStringValue:text];
    }
}

- (NSString*)text
{
    return _text;
}

- (void)_updateButtons
{
    if (!_box) {
        return;
    }
    
    NSView* contentView;
    contentView = [_box contentView];
    
    // Remove buttons from box
    NSEnumerator*   enumerator;
    NSView*         subview;
    enumerator = [[contentView subviews] objectEnumerator];
    while (subview = [enumerator nextObject]) {
        if ([subview isKindOfClass:[SRHalfTransparentButton class]]) {
            [subview removeFromSuperview];
        }
    }
    
    // Add buttons from right to left
    SRHalfTransparentButton*    button;
    NSRect                      contentFrame;
    NSPoint                     origin;
    contentFrame = [contentView frame];
    origin.x = contentFrame.size.width;
    origin.y = 0;
    enumerator = [_buttons objectEnumerator];
    while (button = [enumerator nextObject]) {
        NSRect  buttonFrame;
        buttonFrame = [button frame];
        
        origin.x -= buttonFrame.size.width;
        [button setFrameOrigin:origin];
        [_box addSubview:button];
        
        // Margin between buttons
        origin.x -= 1;
    }
}

- (void)setButtons:(NSArray*)buttons
{
    [_buttons release];
    _buttons = [buttons retain];
    
    // Update button layout
    [self _updateButtons];
}

- (NSArray*)buttons
{
    return _buttons;
}

- (void)setBox:(NSBox*)box
{
    [_box release];
    _box = [box retain];
    
    // Update button layout
    [self _updateButtons];
}

- (NSBox*)box
{
    return _box;
}

- (void)setTextField:(NSTextField*)textField
{
    [_textField release];
    _textField = [textField retain];
}

- (NSTextField*)textField
{
    return _textField;
}

- (BOOL)autoRemove
{
    return _autoRemoveFlag;
}

- (void)setAutoRemove:(BOOL)flag
{
    _autoRemoveFlag = flag;
}

- (void)startAutoRemove
{
    // Start timer for auto remove
    _autoRemoveTimer = [NSTimer scheduledTimerWithTimeInterval:_SRInfoCenterIntervalAutoRemove 
            target:self 
            selector:@selector(_timerAutoRemoveExpired:) 
            userInfo:nil 
            repeats:NO];
}

- (void)_timerAutoRemoveExpired:(id)userInfo
{
    _autoRemoveTimer = nil;
    
    // Remove itself
    [_infoCenter removeInfoCenterItem:self];
}

@end

#pragma mark -
//--------------------------------------------------------------//
// SRDownloadInfoCenterItem class
//--------------------------------------------------------------//

@implementation SRDownloadInfoCenterItem

- (SRHalfTransparentButton*)_createRemoveButton
{
    // Get images
    static NSImage* _closeFrontImage = nil;
    static NSImage* _closePressedImage = nil;
    static NSImage* _closeRolloverImage = nil;
    if (!_closeFrontImage) {
        _closeFrontImage = [[NSImage imageNamed:@"close_trFront"] retain];
        _closePressedImage = [[NSImage imageNamed:@"close_trPressed"] retain];
        _closeRolloverImage = [[NSImage imageNamed:@"close_trRollover"] retain];
    }
    
    // Create button
    SRHalfTransparentButton*    button;
    NSRect                      buttonFrame;
    buttonFrame.origin = NSZeroPoint;
    buttonFrame.size = [_closeFrontImage size];
    button = [[SRHalfTransparentButton alloc] initWithFrame:buttonFrame];
    [button autorelease];
    [button setImage:_closeFrontImage];
    [button setAlternateImage:_closePressedImage];
    [button setRolloverImage:_closeRolloverImage];
    [button setToolTip:NSLocalizedStringFromTable(@"SRRemoveSmallControl", @"SmallControl", nil)];
    [button setTarget:self];
    [button setAction:@selector(removeActiveDowloadAction:)];
    
    return button;
}

- (SRHalfTransparentButton*)_createResumeButton
{
    // Get images
    static NSImage* _resumeFrontImage = nil;
    static NSImage* _resumePressedImage = nil;
    static NSImage* _resumeRolloverImage = nil;
    if (!_resumeFrontImage) {
        _resumeFrontImage = [[NSImage imageNamed:@"resume_trFront"] retain];
        _resumePressedImage = [[NSImage imageNamed:@"resume_trPressed"] retain];
        _resumeRolloverImage = [[NSImage imageNamed:@"resume_trRollover"] retain];
    }
    
    // Create button
    SRHalfTransparentButton*    button;
    NSRect                      buttonFrame;
    buttonFrame.origin = NSZeroPoint;
    buttonFrame.size = [_resumeFrontImage size];
    button = [[SRHalfTransparentButton alloc] initWithFrame:buttonFrame];
    [button autorelease];
    [button setImage:_resumeFrontImage];
    [button setAlternateImage:_resumePressedImage];
    [button setRolloverImage:_resumeRolloverImage];
    [button setToolTip:NSLocalizedStringFromTable(@"SRResumeSmallControl", @"SmallControl", nil)];
    [button setTarget:self];
    [button setAction:@selector(resumeActiveDowloadAction:)];
    
    return button;
}

- (SRHalfTransparentButton*)_createFindButton
{
    // Get images
    static NSImage* _findFrontImage = nil;
    static NSImage* _findPressedImage = nil;
    static NSImage* _findRolloverImage = nil;
    if (!_findFrontImage) {
        _findFrontImage = [[NSImage imageNamed:@"find_trFront"] retain];
        _findPressedImage = [[NSImage imageNamed:@"find_trPressed"] retain];
        _findRolloverImage = [[NSImage imageNamed:@"find_trRollover"] retain];
    }
    
    // Create button
    SRHalfTransparentButton*    button;
    NSRect                      buttonFrame;
    buttonFrame.origin = NSZeroPoint;
    buttonFrame.size = [_findFrontImage size];
    button = [[SRHalfTransparentButton alloc] initWithFrame:buttonFrame];
    [button autorelease];
    [button setImage:_findFrontImage];
    [button setAlternateImage:_findPressedImage];
    [button setRolloverImage:_findRolloverImage];
    [button setToolTip:NSLocalizedStringFromTable(@"SRFindSmallControl", @"SmallControl", nil)];
    [button setTarget:self];
    [button setAction:@selector(findActiveDowloadAction:)];
    
    return button;
}

- (SRHalfTransparentButton*)_createPauseButton
{
    // Get images
    static NSImage* _pauseFrontImage = nil;
    static NSImage* _pausePressedImage = nil;
    static NSImage* _pauseRolloverImage = nil;
    if (!_pauseFrontImage) {
        _pauseFrontImage = [[NSImage imageNamed:@"pause_trFront"] retain];
        _pausePressedImage = [[NSImage imageNamed:@"pause_trPressed"] retain];
        _pauseRolloverImage = [[NSImage imageNamed:@"pause_trRollover"] retain];
    }
    
    // Create button
    SRHalfTransparentButton*    button;
    NSRect                      buttonFrame;
    buttonFrame.origin = NSZeroPoint;
    buttonFrame.size = [_pauseFrontImage size];
    button = [[SRHalfTransparentButton alloc] initWithFrame:buttonFrame];
    [button autorelease];
    [button setImage:_pauseFrontImage];
    [button setAlternateImage:_pausePressedImage];
    [button setRolloverImage:_pauseRolloverImage];
    [button setToolTip:NSLocalizedStringFromTable(@"SRPauseSmallControl", @"SmallControl", nil)];
    [button setTarget:self];
    [button setAction:@selector(pauseActiveDowloadAction:)];
    
    return button;
}

- (void)_updateDownloadControlButtons
{
    NSMutableArray* buttons;
    buttons = [NSMutableArray array];
    
    if (_currentButtons & SRDownloadInfoCenterItemRemoveButton) {
        [buttons addObject:_removeButton];
    }
    if (_currentButtons & SRDownloadInfoCenterItemPauseButton) {
        [buttons addObject:_pauseButton];
    }
    if (_currentButtons & SRDownloadInfoCenterItemResumeButton) {
        [buttons addObject:_resumeButton];
    }
    if (_currentButtons & SRDownloadInfoCenterItemFindButton) {
        [buttons addObject:_findButton];
    }
    
    [self setButtons:buttons];
}

- (id)init
{
    self = [super init];
    if (!self) {
        return nil;
    }
    
    // Set buttons
    _removeButton = [[self _createRemoveButton] retain];
    _pauseButton = [[self _createPauseButton] retain];
    _resumeButton = [[self _createResumeButton] retain];
    _findButton = [[self _createFindButton] retain];
    
    _currentButtons = SRDownloadInfoCenterItemRemoveButton | 
                      SRDownloadInfoCenterItemPauseButton | 
                      SRDownloadInfoCenterItemFindButton;
    [self _updateDownloadControlButtons];
    
    return self;
}

- (void)dealloc
{
    [_removeButton release];
    [_pauseButton release];
    [_resumeButton release];
    [_findButton release];
    
    [super dealloc];
}

- (SRDownloadHistoryItem*)downloadHistoryItem
{
    return _historyItem;
}

- (void)setDownloadHistoryItem:(SRDownloadHistoryItem*)historyItem
{
    [_historyItem release];
    _historyItem = [historyItem retain];
}

- (int)downloadControlButtons
{
    return _currentButtons;
}

- (void)setDwonloadControlButtons:(int)buttons
{
    if (_currentButtons != buttons) {
        _currentButtons = buttons;
        [self _updateDownloadControlButtons];
    }
}

#pragma mark -
//--------------------------------------------------------------//
// Actions
//--------------------------------------------------------------//

- (void)pauseActiveDowloadAction:(id)sender
{
}

- (void)resumeActiveDowloadAction:(id)sender
{
}

- (void)removeActiveDowloadAction:(id)sender
{
    if (!_historyItem) {
        return;
    }
    
    // Remove active download item
    [[SRDownloadCenter sharedInstance] removeDownloadForItem:_historyItem];
}

- (void)findActiveDowloadAction:(id)sender
{
    if (!_historyItem) {
        return;
    }
    
    // Find download item
    [[SRDownloadCenter sharedInstance] findItem:_historyItem];
}

@end

#pragma mark -
//--------------------------------------------------------------//
// SRInfoCenterController class
//--------------------------------------------------------------//

@implementation SRInfoCenterController

#pragma mark -
//--------------------------------------------------------------//
// Initialize
//--------------------------------------------------------------//

- (void)_setHidesOnDeactivate
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    [[self window] setHidesOnDeactivate:![defaults boolForKey:SRInfoCenterAlwaysDisply]];
}

- (id)initWithWindowNibName:(NSString*)windowNibName
{
    self = [super initWithWindowNibName:windowNibName];
    if (!self) {
        return self;
    }
    
    // Initialize member variables
    _infoCenterItems = [[NSMutableArray array] retain];
    
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Store default size
    _defaultSize = [[self window] frame].size;
    
    // Set window origin
    NSString*   originString;
    originString = [defaults objectForKey:SRInfoCenterOrigin];
    if (originString) {
        NSPoint origin;
        origin = NSPointFromString(originString);
        if (!NSEqualPoints(origin, NSZeroPoint)) {
            [[self window] setFrameOrigin:origin];
        }
    }
    
    // Set close button images
    [_closeButton setRolloverImage:[NSImage imageNamed:@"TabClose_Front_Rollover"]];
    
    // Remove template box from super view
    [_templateBox retain];
    [_templateBox removeFromSuperview];
    
    // Set itself as delegate
    [(SRAutoFadePanel*)[self window] setAutoFadeDelegate:self];
    
    [self _setHidesOnDeactivate];
    
    // Register key value observation
    [defaults addObserver:self 
            forKeyPath:SRInfoCenterAlwaysDisply 
            options:NSKeyValueObservingOptionNew 
            context:NULL];
    
    return self;
}

- (void)dealloc
{
    [_infoCenterItems release];
    [_templateBox release];
    
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    [defaults removeObserver:self forKeyPath:SRInfoCenterAlwaysDisply];
    
    [super dealloc];
}

#pragma mark -
//--------------------------------------------------------------//
// Window management
//--------------------------------------------------------------//

- (void)showWindow:(id)sender
{
    // Set fade status
    [self setFadeStatus:SRAutoFadeApperaed];
    [super showWindow:sender];
}

#pragma mark -
//--------------------------------------------------------------//
// Info center item
//--------------------------------------------------------------//

- (void)addInfoCenterItem:(SRInfoCenterItem*)item
{
    if ([_infoCenterItems containsObject:item]) {
        return;
    }
    
    // Set info cneter
    [item setInfoCenter:self];
    
    // Add to array
    [_infoCenterItems addObject:item];
    
    // Get auto fade panel
    SRAutoFadePanel*    panel;
    NSRect              panelFrame;
    panel = (SRAutoFadePanel*)[self window];
    panelFrame = [panel frame];
    
    // Extend window
    if ([_infoCenterItems count] > 1) {
        panelFrame.size.height += [_templateBox frame].size.height;
        panelFrame.origin.y -= [_templateBox frame].size.height;
        [panel setFrame:panelFrame display:YES animate:YES];
    }
    
    // Create new text field
    _SRHalfTransparentTextField*    textField;
    textField = [[_SRHalfTransparentTextField alloc] initWithFrame:NSZeroRect];
    [textField autorelease];
    [textField setEditable:NO];
    [textField setDrawsBackground:NO];
    [textField setBezeled:NO];
    [textField setStringValue:[item text]];
    
    // Create new box
    NSBox*  box;
    NSPoint origin;
    box = [[NSBox alloc] initWithFrame:[_templateBox frame]];
    [box autorelease];
    [box setBorderType:[_templateBox borderType]];
    [box setBoxType:[_templateBox boxType]];
    [box setTitlePosition:[_templateBox titlePosition]];
    origin = [box frame].origin;
    origin.y = 7;
    [box setFrameOrigin:origin];
    [box setAutoresizingMask:[_templateBox autoresizingMask]];
    
    NSRect  contentFrame;
    contentFrame = [[box contentView] frame];
    contentFrame.origin = NSZeroPoint;
    [textField setFrame:contentFrame];
    [[box contentView] addSubview:textField];
    
    [[panel contentView] addSubview:box];
    
    [item setBox:box];
    [item setTextField:textField];
}

- (void)removeInfoCenterItem:(SRInfoCenterItem*)item
{
    if (![_infoCenterItems containsObject:item]) {
        return;
    }
    
    // Get auto fade panel
    SRAutoFadePanel*    panel;
    NSRect              panelFrame;
    panel = (SRAutoFadePanel*)[self window];
    panelFrame = [panel frame];
    
    // Remove info item box
    NSBox*  removedBox;
    NSRect  removedBoxFrame;
    removedBox = [item box];
    removedBoxFrame = [removedBox frame];
    [removedBox removeFromSuperview];
    
    // Move up other boxes
    NSEnumerator*   enumerator;
    NSView*         view;
    enumerator = [[[panel contentView] subviews] objectEnumerator];
    while (view = [enumerator nextObject]) {
        NSRect  frame;
        frame = [view frame];
        if (frame.origin.y < removedBoxFrame.origin.y) {
            frame.origin.y += removedBoxFrame.size.height;
            
            [view setFrame:frame];
        }
    }
    
    // Shrink window
    if ([_infoCenterItems count] > 1) {
        panelFrame.size.height -= [_templateBox frame].size.height;
        panelFrame.origin.y += [_templateBox frame].size.height;
        [panel setFrame:panelFrame display:YES animate:YES];
    }
    
    // Remove from array
    [_infoCenterItems removeObject:item];
}

#pragma mark -
//--------------------------------------------------------------//
// Fade status
//--------------------------------------------------------------//

- (int)fadeStatus
{
    return [(SRAutoFadePanel*)[self window] fadeStatus];
}

- (void)setFadeStatus:(int)fadeStatus
{
    [(SRAutoFadePanel*)[self window] setFadeStatus:fadeStatus];
}

#pragma mark -
//--------------------------------------------------------------//
// SRAutoFadePanel delegate
//--------------------------------------------------------------//

- (void)autoFadePanelAppeared
{
    // Check info center items
    NSEnumerator*       enumerator;
    SRInfoCenterItem*   infoCenterItem;
    enumerator = [_infoCenterItems objectEnumerator];
    while (infoCenterItem = [enumerator nextObject]) {
        if ([infoCenterItem autoRemove]) {
            // Start auto remove
            [infoCenterItem startAutoRemove];
        }
    }
}

- (BOOL)autoFadePanelShouldRemainAfterFadingOut
{
    return [_infoCenterItems count] > 0;
}

#pragma mark -
//--------------------------------------------------------------//
// NSWindow delegate
//--------------------------------------------------------------//

- (void)windowDidMove:(NSNotification*)notification
{
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    // Get window origin
    NSPoint     origin = NSZeroPoint;
    NSString*   originString;
    originString = [defaults objectForKey:SRInfoCenterOrigin];
    if (originString) {
        origin = NSPointFromString(originString);
    }
    
    // Store new origin
    NSPoint newOrigin;
    newOrigin = [[self window] frame].origin;
    if (!NSEqualPoints(origin, newOrigin)) {
        [defaults setObject:NSStringFromPoint(newOrigin) forKey:SRInfoCenterOrigin];
    }
}

#pragma mark -
//--------------------------------------------------------------//
// Key value observation
//--------------------------------------------------------------//

- (void)observeValueForKeyPath:(NSString*)keyPath 
        ofObject:(id)object 
        change:(NSDictionary*)change 
        context:(void *)context
{
    // For NSUserDefault
    if (object == [NSUserDefaults standardUserDefaults]) {
        // General preferences
        if ([keyPath isEqualToString:SRInfoCenterAlwaysDisply]) {
            [self _setHidesOnDeactivate];
            return;
        }
    }
}

@end
