/*
SRTabExpose.m

Author: Makoto Kinoshita

Copyright 2004 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRTabExpose.h"

@interface SRDummyWebView : NSView
{
}
@end

@implementation SRDummyWebView

- (BOOL)canGoBack
{
    return NO;
}

- (BOOL)canGoForward
{
    return NO;
}

- (BOOL)canMakeTextLarger
{
    return NO;
}

- (BOOL)canMakeTextSmaller
{
    return NO;
}

- (WebFrame*)mainFrame
{
    return nil;
}

@end

@implementation SRTabExposeCoverView

#pragma mark -
//--------------------------------------------------------------//
// Initialize
//--------------------------------------------------------------//

#if 0
- (id)initWithFrame:(NSRect)frame
{
    self = [super initWithFrame:frame];
    if (!self) {
        return nil;
    }
    
    // Initialize instance variables
    NSProgressIndicator*    progressIndicator;
    progressIndicator = [[NSProgressIndicator alloc] initWithFrame:NSZeroRect];
    [progressIndicator setStyle:NSProgressIndicatorSpinningStyle];
    [progressIndicator setControlSize:NSSmallControlSize];
    [progressIndicator sizeToFit];
    [progressIndicator startAnimation:self];
    [self addSubview:progressIndicator];
    
    return self;
}
#endif

#pragma mark -
//--------------------------------------------------------------//
// Controller
//--------------------------------------------------------------//

- (SRTabExposeController*)tabExposeController
{
    return _tabExposeController;
}

- (void)setTabExposeController:(SRTabExposeController*)tabExposeController
{
    _tabExposeController = tabExposeController;
}

#pragma mark -
//--------------------------------------------------------------//
// Web view
//--------------------------------------------------------------//

- (WebView*)webView
{
    return _webView;
}

- (void)setWebView:(WebView*)webView
{
    _webView = webView;
}

#pragma mark -
//--------------------------------------------------------------//
// Drawing
//--------------------------------------------------------------//

- (void)mouseDown:(NSEvent*)event
{
    // Do nothing
}

- (void)mouseUp:(NSEvent*)event
{
    // Deexpose with web view
    [_tabExposeController deexposeAndSelectWebView:_webView];
}

- (void)scrollWheel:(NSEvent*)event
{
    // Pass it to web view
    // Do not work yet
    [_webView scrollWheel:event];
}

#if 0
- (void)mouseEntered:(NSEvent*)event
{
    [self setNeedsDisplay:YES];
}

- (void)mouseExited:(NSEvent*)event
{
    [self setNeedsDisplay:YES];
}

- (void)drawRect:(NSRect)frame
{
    // Check mouse location
    NSPoint mousePoint;
    mousePoint = [self convertPoint:[NSEvent mouseLocation] fromView:nil];
    
    // When mouse is inside
    if ([self mouse:mousePoint inRect:[self frame]]) {
        static NSColor* _backgroundColor = nil;
        if (!_backgroundColor) {
            _backgroundColor = [[NSColor colorWithDeviceRed:0.0f green:0.0f blue:1.0f alpha:0.5f] retain];
        }
        [_backgroundColor set];
        NSRectFill(frame);
    }
    
    // When mouse is outside
    else {
    }
}
#endif

@end

#pragma mark -

@implementation SRTabExposeView

#pragma mark -
//--------------------------------------------------------------//
// Initialize
//--------------------------------------------------------------//

- (id)initWithFrame:(NSRect)frame
{
    self = [super initWithFrame:frame];
    if (!self) {
        return nil;
    }
    
    // Initialize instance variables
    _tabViews = [[NSMutableArray array] retain];
    _coverViews = [[NSMutableArray array] retain];
    
    return self;
}

- (void)dealloc
{
    [_tabViews release];
    [_coverViews release];
    
    [super dealloc];
}

#pragma mark -
//--------------------------------------------------------------//
// Controller
//--------------------------------------------------------------//

- (SRTabExposeController*)tabExposeController
{
    return _tabExposeController;
}

- (void)setTabExposeController:(SRTabExposeController*)tabExposeController
{
    _tabExposeController = tabExposeController;
}

#pragma mark -
//--------------------------------------------------------------//
// Expose
//--------------------------------------------------------------//

- (int)divideNumberOfNumber:(int)number
{
    if (number < 2) {
        return 1;
    }
    
    int divideNumber = 1;
    while (1) {
        if ((number / divideNumber <= divideNumber) || 
            ((number / divideNumber == divideNumber + 1) && 
             (number % divideNumber == 0)))
        {
            return divideNumber;
        }
        
        divideNumber++;
    }
    
    // Do not reach here
    return 1;
}

- (void)exposeTabViews
{
    static int  _margin = 20;
    static int  _padding = 5;
    
    // Get number of views
    int numberOfViews;
    numberOfViews = [_tabViews count];
    if (numberOfViews < 2) {
        return;
    }
    
    // Get divide number
    int divideNumber;
    divideNumber = [self divideNumberOfNumber:numberOfViews];
    
    // Get screen size
    NSRect  screenFrame, visibleScreenFrame;
    screenFrame = [[[self window] screen] frame];
    visibleScreenFrame = [[[self window] screen] visibleFrame];
    
    // Decide view size
    NSRect  frame;
    NSSize  origSize;
    int     height;
    float   aspectRatio = 1.0f;
    origSize = [[_tabViews objectAtIndex:0] frame].size;
    height = origSize.height * divideNumber + _padding * (divideNumber - 1) + _margin * 2;
    if (height > visibleScreenFrame.size.height) {
        aspectRatio = visibleScreenFrame.size.height / height;
    }
    frame.size.width = origSize.width * aspectRatio;
    frame.size.height = origSize.height * aspectRatio;
    
    // Expose image views
    int i, index = 0;
    for (i = 0; i < divideNumber; i++) {
        int count = 1, remainRow;
        remainRow = divideNumber - (i + 1);
        while (count < divideNumber + 1 && index + count < numberOfViews) {
            if (remainRow == 0) {
                count++;
                continue;
            }
            
            int remain;
            remain = numberOfViews - index - count;
            if ((remain / remainRow < count) || 
                ((remain / remainRow == count) && 
                 (remain % remainRow == 0)))
            {
                break;
            }
            count++;
        }
        
        // Calc image view position
        float   totalWidth;
        totalWidth = frame.size.width * count + _padding * (count - 1) + _margin * 2;
        
        if (totalWidth > visibleScreenFrame.size.width) {
            // Shrink size
            float   width;
            width = origSize.width * count + _padding * (count - 1) + _margin * 2;
            aspectRatio = visibleScreenFrame.size.width / width;
            frame.size.width = origSize.width * aspectRatio;
            frame.size.height = origSize.height * aspectRatio;
            totalWidth = frame.size.width * count + _padding * (count - 1) + _margin * 2;
        }
        
        float   totalHeight;
        totalHeight = frame.size.height * divideNumber + _padding * (divideNumber - 1) + _margin * 2;
        frame.origin.y = (visibleScreenFrame.origin.y - screenFrame.origin.y) + 
                visibleScreenFrame.size.height - _margin - 
                (visibleScreenFrame.size.height / 2 - totalHeight / 2) - 
                frame.size.height * (i + 1) - _padding * i;
        
        int j;
        for (j = 0; j < count; j++) {
            frame.origin.x = (visibleScreenFrame.origin.x - screenFrame.origin.x) + 
                    _margin + visibleScreenFrame.size.width / 2 - totalWidth / 2 + 
                    (frame.size.width + _padding) * j;
            
            WebView*    tabView;
            tabView = [_tabViews objectAtIndex:index];
            //[[[[tabView mainFrame] frameView] documentView] scaleUnitSquareToSize:NSMakeSize(aspectRatio, aspectRatio)];
            [tabView scaleUnitSquareToSize:NSMakeSize(aspectRatio, aspectRatio)];
            [tabView setFrame:frame];
            
            SRTabExposeCoverView*   coverView;
            coverView = [_coverViews objectAtIndex:index];
            [coverView setFrame:frame];
            
            index++;
        }
    }
}

#pragma mark -
//--------------------------------------------------------------//
// Tab views
//--------------------------------------------------------------//

- (void)setTabViews:(NSArray*)tabViews
{
    // Remove old tab views
    [_tabViews removeAllObjects];
    [_coverViews removeAllObjects];
    
    // Add tab views
    NSEnumerator*   enumerator;
    WebView*        tabView;
    enumerator = [tabViews objectEnumerator];
    while (tabView = [enumerator nextObject]) {
        // Add tab view in view hierarchy
        [self addSubview:tabView];
        [_tabViews addObject:tabView];
        
        // Create cover view
        SRTabExposeCoverView*   coverView;
        coverView = [[SRTabExposeCoverView alloc] initWithFrame:NSZeroRect];
        [coverView autorelease];
        [coverView setTabExposeController:_tabExposeController];
        [coverView setWebView:tabView];
        [self addSubview:coverView];
        
        [_coverViews addObject:coverView];
    }
}

- (NSArray*)tabViews
{
    return _tabViews;
}

- (void)mouseDown:(NSEvent*)event
{
    // Deexpose
    [_tabExposeController deexpose];
}

- (void)drawRect:(NSRect)frame
{
    // Fill frame
    static NSColor* _backgroundColor = nil;
    if (!_backgroundColor) {
        _backgroundColor = [[NSColor colorWithDeviceWhite:0.0 alpha:0.6] retain];
    }
    
    [_backgroundColor set];
    NSRectFill(frame);
}

@end

#pragma mark -

@implementation SRTabExposeWindow

#pragma mark -
//--------------------------------------------------------------//
// Event handling
//--------------------------------------------------------------//

- (void)sendEvent:(NSEvent*)event
{
    // Wheel scroll
    if ([event type] == NSScrollWheel) {
        NSView* view;
        view = [[self contentView] hitTest:[event locationInWindow]];
        if (view) {
            [view scrollWheel:event];
        }
        return;
    }
    
    [super sendEvent:event];
}

@end

#pragma mark -

@implementation SRTabExposeController

#pragma mark -
//--------------------------------------------------------------//
// Expose
//--------------------------------------------------------------//

static BOOL _isExposing = NO;

+ (BOOL)isExposing
{
    return _isExposing;
}

- (void)exposeWithTabView:(SRTabView*)tabView
{
    _tabView = tabView;
    
    // Get tab view frame
    NSRect  tabFrame;
    tabFrame = [[tabView tabView] frame];
    
    // Get screen frame
    NSRect  screenFrame;
    screenFrame = [[[tabView window] screen] frame];
    
    // Create window
    _window = [[SRTabExposeWindow alloc] initWithContentRect:screenFrame 
            styleMask:NSBorderlessWindowMask 
            backing:NSBackingStoreBuffered 
            defer:YES];
    [_window setLevel:NSFloatingWindowLevel];
    [_window setOpaque:NO];
    
    // Create expose view
    _tabExposeView = [[SRTabExposeView alloc] initWithFrame:
            NSMakeRect(0, 0, screenFrame.size.width, screenFrame.size.height)];
    [[_window contentView] addSubview:_tabExposeView];
    [_tabExposeView setTabExposeController:self];
    [_window makeFirstResponder:_tabExposeView];
    
    // Show window
    [_window makeKeyAndOrderFront:self];
    
    // Get tab views
    NSMutableArray* tabViews;
    NSMutableArray* titles;
    tabViews = [NSMutableArray array];
    titles = [NSMutableArray array];
    
    int i;
    for (i = 0; i < [tabView numberOfItems]; i++) {
        WebView*    webView;
        webView = (WebView*)[tabView viewAtIndex:i];
        
        // Check web view size
        NSRect  webFrame;
        webFrame = [webView frame];
        if (!NSEqualSizes(tabFrame.size, webFrame.size)) {
            [webView setFrameSize:tabFrame.size];
        }
        
        [tabViews addObject:webView];
        
        // Remove web view from tab view
        SRDummyWebView* dummyView;
        dummyView = [[SRDummyWebView alloc] initWithFrame:NSZeroRect];
        [[[tabView tabView] tabViewItemAtIndex:i] setView:dummyView];
        [dummyView release];
        [webView setHostWindow:_window];
    }
    
    // Add tab views
    [_tabExposeView setTabViews:tabViews];
    
    // Expose view
    _isExposing = YES;
    [_tabExposeView exposeTabViews];
}

- (void)deexpose
{
    [self deexposeAndSelectWebView:nil];
}

- (void)deexposeAndSelectWebView:(WebView*)selectedWebView
{
    // Get frame of tab view
    NSRect  frame;
    frame = [[_tabView tabView] frame];
    frame.origin = NSZeroPoint;
    
    // Restore tab views
    NSArray*        tabViews;
    NSEnumerator*   enumerator;
    WebView*        webView;
    int             index = 0;
    tabViews = [_tabExposeView tabViews];
    enumerator = [tabViews objectEnumerator];
    while (webView = [enumerator nextObject]) {
        // Remove from expose view
        [webView removeFromSuperview];
        [webView setHostWindow:[_tabView window]];
        
        [webView setFrame:frame];
        [webView scaleUnitSquareToSize:NSMakeSize(1.0f, 1.0f)];
        [webView setBounds:frame];
        [webView setAutoresizingMask:NSViewWidthSizable | NSViewHeightSizable];
        
        [[[_tabView tabView] tabViewItemAtIndex:index] setView:webView];
        
        index++;
    }
    
    // Resize subviews
    [[_tabView tabView] resizeSubviewsWithOldSize:frame.size];
    
    // Select web view
    if (selectedWebView) {
        [_tabView selectItemAtIndex:[_tabView indexOfView:selectedWebView]];
    }
    
    // Change first responder
    [[_tabView window] makeFirstResponder:[_tabView viewOfSelectedTabViewItem]];
    
    // Close window
    [_window orderOut:self];
    
    // Release window and expose view
    [_tabExposeView release];
    _tabExposeView = nil;
    [_window release];
    _window = nil;
    
    _isExposing = NO;
}

@end

