/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.kosho;

import java.io.InputStream;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;

import shohaku.core.beans.BindMethod;
import shohaku.ginkgo.Ginkgo;
import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.Node;
import shohaku.ginkgo.NodeCompositeRule;

/**
 * XMLデータで定義された情報を基に、メソッドを特殊な規則で拘束する機能を生成する機能を提供します。<br>
 * このクラスは <code>shohaku.core.beans.BindMethod</code> をXMLデータから生成し、識別子でマッピングします。
 */
public class BindMethodFactory implements KoshoResourcesLoader {

    /* リソースデータを格納します。 */
    private final Map lookup = new Hashtable();

    /**
     * デフォルトコンストラクタ。
     */
    public BindMethodFactory() {
        // no op
    }

    /**
     * デフォルトの構成ルールを用いて、入力ストリームから構成リソースを読み込みます。
     * 
     * @param inStream
     *            構成リソースの入力ストリーム
     * @throws GinkgoException
     *             構成情報例外
     */
    public void load(InputStream inStream) throws GinkgoException {
        load(inStream, getDefaultNodeCompositeRule());
    }

    /**
     * 指定された構成ルールを用いて、入力ストリームから構成リソースを読み込みます。
     * 
     * @param inStream
     *            構成リソースの入力ストリーム
     * @param rule
     *            構成ルール
     * @throws GinkgoException
     *             構成情報例外
     */
    public void load(InputStream inStream, NodeCompositeRule rule) throws GinkgoException {
        // XMLの解析
        Ginkgo ginkgo = new Ginkgo();
        ginkgo.parse(rule, inStream);
        Node root = ginkgo.getDocument().getContext().getRoot();
        initValues(root);
    }

    /* リソースの初期化および登録します。 */
    private void initValues(Node root) {
        Map m = new HashMap();
        initValues(root, m);
        lookup.putAll(m);
    }

    /* 再起的にリソースを登録します。 */
    private void initValues(Node parent, Map m) {
        for (Iterator i = parent.getContext().elementIterator("bind-method-mapping"); i.hasNext();) {
            Node node = (Node) i.next();
            String groupId = node.getAttribute("groupId");
            initValue(node, groupId + ':', m);
        }
    }

    /* リソースを評価して登録します。 */
    private void initValue(Node parent, String groupId, Map m) {
        for (Iterator i = parent.getContext().elementIterator("bind-method"); i.hasNext();) {
            Node node = (Node) i.next();
            String id = node.getAttribute("id");
            m.put(groupId + id, node.getNodeValue());
        }
    }

    /**
     * 指定された識別子の拘束されたメソッド情報を取得します。
     * 
     * @param id
     *            識別子
     * @return 指定された識別子の拘束されたメソッド情報
     */
    public BindMethod getBindMethod(String id) {
        if (id == null) {
            throw new NullPointerException("id as null");
        }
        return (BindMethod) lookup.get(id);
    }

    /*
     * static
     */

    /**
     * デフォルトの構成ルールを返却します。
     * 
     * @return デフォルトの構成ルール
     */
    public static NodeCompositeRule getDefaultNodeCompositeRule() {
        return Kosho.getDefaultNodeCompositeRule(BindMethodFactory.class);
    }
}
