/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.kosho;

import java.io.InputStream;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;

import shohaku.core.collections.IteratorUtils;
import shohaku.ginkgo.Ginkgo;
import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.Node;
import shohaku.ginkgo.NodeCompositeRule;

/**
 * クラスと１対１で対応付けられたXMLデータを読み込み保存する機能を提供します。
 */
public class ClassBindConfiguration implements KoshoResources, KoshoResourcesLoader {

    /* リソースデータを格納します。 */
    private final Map lookup = new Hashtable();

    /**
     * デフォルトコンストラクタ。
     */
    public ClassBindConfiguration() {
        // no op
    }

    /**
     * デフォルトの構成ルールを用いて、入力ストリームから構成リソースを読み込みます。
     * 
     * @param inStream
     *            構成リソースの入力ストリーム
     * @throws GinkgoException
     *             構成情報例外
     */
    public void load(InputStream inStream) throws GinkgoException {
        load(inStream, getDefaultNodeCompositeRule());
    }

    /**
     * 指定された構成ルールを用いて、入力ストリームから構成リソースを読み込みます。
     * 
     * @param inStream
     *            構成リソースの入力ストリーム
     * @param rule
     *            構成ルール
     * @throws GinkgoException
     *             構成情報例外
     */
    public void load(InputStream inStream, NodeCompositeRule rule) throws GinkgoException {
        // XMLの解析
        Ginkgo ginkgo = new Ginkgo();
        ginkgo.parse(rule, inStream);
        Node root = ginkgo.getDocument().getContext().getRoot();
        initValues(root);
    }

    /* リソースの初期化および登録します。 */
    private void initValues(Node root) {
        Map m = new HashMap();
        initValues(root, "", m, '.');
        lookup.putAll(m);
    }

    /* 再起的にリソースを登録します。 */
    private void initValues(Node parent, String parentName, Map m, char separator) {
        for (Iterator iter = parent.getContext().ownerIterator(); iter.hasNext();) {
            Node node = (Node) iter.next();
            initValue(node, parentName, m, separator);
        }
    }

    /* リソースを評価して登録します。 */
    private void initValue(Node node, String parentName, Map m, char separator) {
        if (node.isType(Node.TYPE_VALUE)) {
            String name = node.getAttribute("name");
            if (name != null && name.length() != 0) {
                m.put(parentName + name, node.getNodeValue());
            }
        } else if (node.isType(Node.TYPE_CONTAINER)) {
            if ("constants".equals(node.getContext().getNodeName())) {
                initValues(node, parentName, m, '_');
            } else {
                String name = node.getAttribute("name");
                if (name != null && name.length() != 0) {
                    initValues(node, parentName + name + separator, m, separator);
                }
            }
        }
    }

    /*
     * implements KoshoResources
     */

    /**
     * 全てのプロパティキーを含む反復子を返却します。
     * 
     * @return 全てのプロパティキーを含む反復子
     */
    public Iterator keyIterator() {
        return IteratorUtils.unmodifiableIterator(lookup.keySet().iterator());
    }

    /**
     * プロパティキーが示す値を返却します。 指定されたキーが存在しない場合<code>null</code>返却します。
     * 
     * @param key
     *            プロパティ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     */
    public Object getObject(String key) {
        if (key == null) {
            throw new NullPointerException("resource key as null");
        }
        Object obj = lookup.get(key);
        return obj;
    }

    /**
     * プロパティキーが示す値を返却します。 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            プロパティキー
     * @param defaultValue
     *            プロパティキー
     * @return プロパティキーが示す値
     * @throws NullPointerException
     *             key が null の場合発生する
     */
    public Object getObject(String key, Object defaultValue) {
        Object o = getObject(key);
        if (o == null) {
            o = defaultValue;
        }
        return o;
    }

    /**
     * 指定されたキーがプロパティセットに含まれている場合に<code>true</code>を返す。
     * 
     * @param key
     *            プロパティキー
     * @return 指定されたキーが含まれている場合 true
     * @throws NullPointerException
     *             key が null の場合発生する
     */
    public boolean containsKey(String key) {
        if (key == null) {
            throw new NullPointerException("resource key as null");
        }
        return lookup.containsKey(key);
    }

    /**
     * 指定されたキーがプロパティセットに含まれていると同時に、<br>
     * 指定されたクラスとキャスト可能な関係に有る場合に<code>true</code>を返す。 <br>
     * 値がNullの場合キャスト可能(true)を返却します。
     * 
     * @param key
     *            プロパティキー
     * @param type
     *            キャスト可能な関係に有るか検証するクラス
     * @return 指定されたキーが含まれている場合 true
     * @throws NullPointerException
     *             key または type が null の場合発生する
     */
    public boolean containsKey(String key, Class type) {
        if (key == null || type == null) {
            throw new NullPointerException("resource key or type as null");
        }
        Object o = getObject(key);
        if (o != null) {
            return type.isInstance(o);
        } else {
            return containsKey(key);
        }
    }

    /*
     * static
     */

    /**
     * デフォルトの構成ルールを返却します。
     * 
     * @return デフォルトの構成ルール
     */
    public static NodeCompositeRule getDefaultNodeCompositeRule() {
        return Kosho.getDefaultNodeCompositeRule(ClassBindConfiguration.class);
    }
}
