/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.kosho;

import java.io.InputStream;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;

import shohaku.core.collections.IteratorUtils;
import shohaku.ginkgo.Ginkgo;
import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.Node;
import shohaku.ginkgo.NodeCompositeRule;

/**
 * クラスと対応付けられたXMLデータを読み込み保存する機能を提供します。
 */
public class ClassMappingConfiguration implements KoshoResourcesLoader {

    /** クラスをマッピングするリソースデータを格納します。 */
    private final Map lookup = new Hashtable();

    /** Ginkgo。 */
    private final Ginkgo ginkgo = new Ginkgo();

    /** 解析処理に使用するClassLoader. */
    private ClassLoader classLoader;

    /**
     * デフォルトコンストラクタ。
     */
    public ClassMappingConfiguration() {
        super();
    }

    /**
     * 解析処理に使用する ClassLoader を返却します.
     * 
     * @return 解析処理に使用する ClassLoader
     */
    public ClassLoader getClassLoader() {
        return this.classLoader;
    }

    /**
     * 解析処理に使用する ClassLoader を設定します.
     * 
     * @param classLoader
     *            解析処理に使用する ClassLoader
     */
    public void setClassLoader(ClassLoader classLoader) {
        this.classLoader = classLoader;

    }

    /*
     * implements KoshoResourcesLoader
     */

    /**
     * デフォルトの構成ルールを用いて、入力ストリームから構成リソースを読み込みます。
     * 
     * @param inStream
     *            構成リソースの入力ストリーム
     * @throws GinkgoException
     *             構成情報例外
     */
    public void load(InputStream inStream) throws GinkgoException {
        load(inStream, getDefaultNodeCompositeRule());
    }

    /**
     * 指定された構成ルールを用いて、入力ストリームから構成リソースを読み込みます。
     * 
     * @param inStream
     *            構成リソースの入力ストリーム
     * @param rule
     *            構成ルール
     * @throws GinkgoException
     *             構成情報例外
     */
    public void load(InputStream inStream, NodeCompositeRule rule) throws GinkgoException {
        // XMLの解析
        ginkgo.parse(rule, inStream);
        Node root = ginkgo.getDocument().getContext().getRoot();
        initValues(root);
    }

    /* リソースの初期化および登録をします。 */
    private void initValues(Node root) {
        Map m = new HashMap();
        for (Iterator i = root.getContext().elementIterator("class"); i.hasNext();) {
            Node classNode = (Node) i.next();
            String className = classNode.getAttribute("name");
            if (className == null || className.length() == 0) {
                throw new GinkgoException("class name isn't specified.");
            }
            KoshoResources resources = getClassResources(classNode);
            m.put(className, resources);
        }
        lookup.putAll(m);
    }

    /* クラス名単位でリソースを生成します。 */
    private ConfigurationResourcesImpl getClassResources(Node classNode) {
        Map m = new HashMap();
        initClassResources(classNode, "", m, '.');
        return new ConfigurationResourcesImpl(m);
    }

    /* クラス名単位のリソースを再帰的に初期化および登録をします。 */
    private void initClassResources(Node parent, String parentName, Map m, char separator) {
        for (Iterator i = parent.getContext().ownerIterator(); i.hasNext();) {
            Node n = (Node) i.next();
            if (n.isType(Node.TYPE_VALUE)) {
                String name = n.getAttribute("name");
                if (name != null && name.length() != 0) {
                    m.put(parentName + name, n.getNodeValue());
                }
            } else if (n.isType(Node.TYPE_CONTAINER)) {
                if ("constants".equals(n.getContext().getNodeName())) {
                    initClassResources(n, parentName, m, '_');
                } else {
                    String name = n.getAttribute("name");
                    if (name != null && name.length() != 0) {
                        initClassResources(n, parentName + name + separator, m, separator);
                    }
                }
            }
        }
    }

    /**
     * 指定されたクラスに関連づくリソースデータを返却します。 指定されたクラスの情報が存在しない場合<code>null</code>返却します。
     * 
     * @param key
     *            取得するクラス
     * @return 指定されたクラスに関連づくリソースデータ
     * @throws NullPointerException
     *             key が null の場合発生する
     */
    public KoshoResources getResources(Class key) {
        if (key == null) {
            throw new NullPointerException("resource key is null");
        }
        return (KoshoResources) lookup.get(key.getName());
    }

    /**
     * 指定されたクラス名に関連づくリソースデータを返却します。 指定されたクラスの情報が存在しない場合<code>null</code>返却します。
     * 
     * @param className
     *            取得するクラス名
     * @return 指定されたクラスに関連づくリソースデータ
     * @throws NullPointerException
     *             key が null の場合発生する
     */
    public KoshoResources getResources(String className) {
        if (className == null) {
            throw new NullPointerException("className is null");
        }
        return (KoshoResources) lookup.get(className);
    }

    /**
     * 指定されたクラスが含まれている場合に<code>true</code>を返す。
     * 
     * @param key
     *            検証するクラス
     * @return 指定されたクラスが含まれている場合に true
     * @throws NullPointerException
     *             key が null の場合発生する
     */
    public boolean containsClass(Class key) {
        if (key == null) {
            throw new NullPointerException("resource key is null");
        }
        return lookup.containsKey(key.getName());
    }

    /**
     * クラス名の反復子を返却します。
     * 
     * @return クラス名の反復子
     */
    public Iterator classNameIterator() {
        return IteratorUtils.unmodifiableIterator(lookup.keySet().iterator());
    }

    /**
     * リソースの反復子を返却します。
     * 
     * @return リソースの反復子
     */
    public Iterator resourceIterator() {
        return IteratorUtils.unmodifiableIterator(lookup.values().iterator());
    }

    /*
     * static
     */

    /**
     * デフォルトの構成ルールを返却します。
     * 
     * @return デフォルトの構成ルール
     */
    public static NodeCompositeRule getDefaultNodeCompositeRule() {
        return Kosho.getDefaultNodeCompositeRule(ClassMappingConfiguration.class);
    }

    /* class configuration resources. */
    private static class ConfigurationResourcesImpl implements KoshoResources {

        private final Map src;

        private ConfigurationResourcesImpl(Map resources) {
            this.src = resources;
        }

        public Iterator keyIterator() {
            return IteratorUtils.unmodifiableIterator(src.keySet().iterator());
        }

        public Object getObject(String key) {
            if (key == null) {
                throw new NullPointerException("resource key is null");
            }
            return src.get(key);
        }

        public Object getObject(String key, Object defaultValue) {
            if (key == null) {
                throw new NullPointerException("resource key is null");
            }
            return getObject(key, defaultValue);
        }

        public boolean containsKey(String key) {
            if (key == null) {
                throw new NullPointerException("resource key is null");
            }
            return src.containsKey(key);
        }

        public boolean containsKey(String key, Class type) {
            if (key == null || type == null) {
                throw new NullPointerException("resource key or resource type is null");
            }
            Object o = getObject(key);
            if (o != null) {
                return type.isInstance(o);
            } else {
                return containsKey(key);
            }
        }
    }

}
