/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.kosho;

import java.io.InputStream;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

import shohaku.core.beans.BeansFactory;
import shohaku.core.lang.NoSuchResourceException;
import shohaku.core.lang.ObjectCreationException;
import shohaku.core.lang.ResourceLoader;
import shohaku.ginkgo.Ginkgo;
import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.Node;
import shohaku.ginkgo.NodeCompositeRule;

/**
 * XMLデータで定義された情報を基に、指定のスコープでコンポーネントを生成する機能を提供します。
 * <p>
 * 例：
 * 
 * <pre>
 * // 湖を表現するオブジェクト
 * public class Lake {
 *     private String name;// 湖の名前
 *     public static Lake(String name) { this.name = name; }
 *     public String getName() { return name; }
 *     public void setName(String name) { this.name = name; }
 * }
 *
 * //定義ファイル lake-factory.xml
 * &lt;component-factory&gt;
 *
 *    &lt;!-- インスタンスを生成する(シングルトンで定義、一度のみ生成する) --&gt;
 *    &lt;components name="new"&gt;
 *
 *        &lt;!-- コンストラクタ --&gt;
 *        &lt;component id="new.constructor" name="constructor" class="Foo" instance="singleton"&gt;
 *            &lt;init&gt;
 *                &lt;arg&gt;&lt;string&gt;支笏湖&lt;/string&gt;&lt;/arg&gt;
 *            &lt;/init&gt;
 *        &lt;/component&gt;
 *
 *        &lt;!-- デフォルトコンストラクタとプロパティ --&gt;
 *        &lt;component id="new.property" name="property" class="Foo" instance="singleton"&gt;
 *            &lt;property name="name"&gt;
 *                &lt;string&gt;阿寒湖&lt;/string&gt;
 *            &lt;/property&gt;
 *        &lt;/component&gt;
 *
 *        &lt;!-- デフォルトコンストラクタとメソッド --&gt;
 *        &lt;component id="new.method" name="method" class="Foo" instance="singleton"&gt;
 *            &lt;method name="setName"&gt;
 *                &lt;arg&gt;&lt;string&gt;倶多楽湖&lt;/string&gt;&lt;/arg&gt;
 *            &lt;/method&gt;
 *        &lt;/component&gt;
 *
 *    &lt;/components&gt;
 *
 *    &lt;!-- コンポーネントを参照する(プロトタイプで定義、毎回新規生成する)   --&gt;
 *    &lt;components name="ref"&gt;
 *
 *        &lt;!-- コンポーネントを参照してリストを生成します --&gt;
 *        &lt;component name="list" class="java.util.ArrayList"  instance="prototype"&gt;
 *            &lt;method name="add"&gt;
 *                &lt;arg&gt;&lt;ref&gt;new.constructor&lt;/ref&gt;&lt;/arg&gt;
 *            &lt;/method&gt;
 *            &lt;method name="add"&gt;
 *                &lt;arg&gt;&lt;ref&gt;new.property&lt;/ref&gt;&lt;/arg&gt;
 *            &lt;/method&gt;
 *            &lt;method name="add"&gt;
 *                &lt;arg&gt;&lt;ref&gt;new.method&lt;/ref&gt;&lt;/arg&gt;
 *            &lt;/method&gt;
 *        &lt;/component&gt;
 *
 *    &lt;/components&gt;
 *
 * &lt;component-factory&gt;
 *
 * グループ名とコンポーネント名を ':' で区切りコンポーネント識別子として認識されます。
 * ファイルの読み取りとコンポーネント取得は以下の様に為ります。
 *
 * ComponentFactory factory = ComponentFactory.getFactory("lake-factory.xml");
 *
 * Lake lake = (Lake) factory.getComponent("new:constructor");
 * List list = (List) factory.getComponent("ref:list");
 * System.out.println("湖名=" + lake.getName());
 * System.out.println("リスト=" + list.toString());
 *
 * ＞＞湖名=支笏湖
 * ＞＞リスト=[支笏湖, 阿寒湖, 倶多楽湖]
 *
 * </pre>
 */
public class ComponentFactory {

    /* リソースデータを格納します。 */
    private final Map lookup = new LinkedHashMap();

    /* 使用するGinkgo。 */
    private final Ginkgo ginkgo = new Ginkgo();

    /* コンポーネントを読み取り初期化します。 */
    private ComponentFactory(InputStream inStream, ClassLoader classLoader) {
        load(inStream, getDefaultNodeCompositeRule(), classLoader);
    }

    /* 入力ストリームからリソースデータを読み込む。 */
    private void load(InputStream inStream, NodeCompositeRule rule, ClassLoader classLoader) throws GinkgoException {
        // XMLの解析
        ginkgo.setClassLoader(classLoader);
        ginkgo.parse(rule, inStream);
        Node root = ginkgo.getDocument().getContext().getRoot();
        Map lookupValues = new LinkedHashMap();
        initValues(root, lookupValues);
        lookup.putAll(lookupValues);
    }

    /* リソースの初期化および引数に登録をする */
    private void initValues(Node root, Map values) throws GinkgoException {
        for (Iterator iter = root.getContext().elementIterator("components"); iter.hasNext();) {
            Node components = (Node) iter.next();
            String name = components.getAttribute("name");
            if (name == null || name.length() == 0) {
                throw new GinkgoException("components name is null.");
            }
            initComponentValues(components, name, values);
        }
    }

    /* コンポーネントの初期化および登録をする */
    private void initComponentValues(Node components, String groupName, Map values) throws GinkgoException {
        for (Iterator iter = components.getContext().elementIterator("component"); iter.hasNext();) {
            Node component = (Node) iter.next();
            String name = component.getAttribute("name");
            if (name == null || name.length() == 0) {
                throw new GinkgoException("component name is null.");
            }
            values.put(groupName + ':' + name, component.getNodeValue());
        }
    }

    /*
     * new component instance
     */

    /**
     * 指定の識別子を持つコンポーネントを生成して返却します。
     * 
     * @param componentId
     *            コンポーネント識別子
     * @return 指定の識別子を持つコンポーネント
     * @throws ObjectCreationException
     *             オブジェクトの生成に失敗した場合に発生します
     */
    public Object getComponent(String componentId) throws ObjectCreationException {
        if (componentId == null) {
            throw new NullPointerException("componentId is null.");
        }
        BeansFactory o = (BeansFactory) lookup.get(componentId);
        if (o == null) {
            throw new ObjectCreationException("componentId is illegal. componentId:" + componentId);
        }
        return o.getInstance();
    }

    /**
     * デフォルトの構成ルールを返却します。
     * 
     * @return デフォルトの構成ルール
     */
    public NodeCompositeRule getDefaultNodeCompositeRule() {
        return Kosho.getDefaultNodeCompositeRule(ComponentFactory.class);
    }

    /*
     * new factory instance
     */

    /**
     * 定義ファイルからコンポーネントファクトリのインスタンスを生成して返却します。
     * 
     * @param resourcePath
     *            定義ファイルへのパス
     * @return コンポーネントファクトリのインスタンス
     * @throws NoSuchResourceException
     *             指定の定義ファイルが発見出来ない場合
     */
    public static ComponentFactory getFactory(String resourcePath) throws NoSuchResourceException {
        return getFactory(resourcePath, ComponentFactory.class.getClassLoader());
    }

    /**
     * 定義ファイルからコンポーネントファクトリのインスタンスを生成して返却します。
     * 
     * @param resourcePath
     *            定義ファイルへのパス
     * @return コンポーネントファクトリのインスタンス
     * @param classLoader
     *            生成に使用するクラスローダ
     * @throws NoSuchResourceException
     *             指定の定義ファイルが発見出来ない場合
     */
    public static ComponentFactory getFactory(String resourcePath, ClassLoader classLoader)
            throws NoSuchResourceException {
        return getFactoryForCache(resourcePath, classLoader);
    }

    /* 定義ファイルのパスを基にコンポーネントファクトリのインスタンスを保存します。 */
    private static final class FactoryCache {
        static final Map cache = new Hashtable();
    }

    /* 定義ファイルからコンポーネントファクトリのインスタンスを生成して返却します。 */
    private static ComponentFactory getFactoryForCache(String resourcePath, ClassLoader classLoader)
            throws NoSuchResourceException {
        if (resourcePath == null) {
            throw new NullPointerException("resourcePath is null.");
        }
        synchronized (FactoryCache.cache) {
            ComponentFactory factory = (ComponentFactory) FactoryCache.cache.get(resourcePath);
            if (factory != null) {
                return factory;
            } else {
                InputStream inStream = ResourceLoader.getResourceAsStream(resourcePath, classLoader);
                factory = new ComponentFactory(inStream, classLoader);
                FactoryCache.cache.put(resourcePath, factory);
            }
            return factory;
        }
    }
}
