/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.kosho;

import java.io.InputStream;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;

import shohaku.core.collections.Entry;
import shohaku.core.collections.EntryUtils;
import shohaku.core.lang.NoSuchResourceException;
import shohaku.core.lang.ResourceLoader;
import shohaku.ginkgo.NodeCompositeRule;

/**
 * ライブラリの利用を簡略化するユーティリティ機能を提供します。
 */
public class Kosho {

    /* 内部的に使用する情報のキャッシュ機能を格納します。 */
    private final static class Cache {

        /** クラスをキーとして ClassBindConfiguration をキャッシュします。 */
        public static final Map classBindConfiguration = new IdentityHashMap();

        /** 構成ルールをキャッシュします。 */
        public static final Map xmlProperties = new HashMap();

        /** クラスをキーとして ClassConfiguration をキャッシュします。 */
        public static final Map classConfiguration = new HashMap();

        /** デフォルトの構成ルールをキャッシュします。 */
        public static final Map defaultNodeCompositeRule = new IdentityHashMap();

    }

    // //////////////////////////////////////////////////////////////
    // XMLProperties
    // //////////////////////////////////////////////////////////////

    /**
     * リソースへのクラスパスからXMLプロパティの構成リソースを生成します。
     * 
     * @param path
     *            リソースへのクラスパス
     * @return XMLプロパティラッパー
     * @throws NoSuchResourceException
     *             リソースが見つからなかった場合
     */
    public static KoshoResources getXMLProperties(String path) throws NoSuchResourceException {
        return getXMLProperties(path, Kosho.class.getClassLoader());
    }

    /**
     * リソースへのクラスパスからXMLプロパティの構成リソースを生成します。
     * 
     * @param path
     *            リソースへのクラスパス
     * @param loader
     *            構成リソースを生成に使用するクラスローダ
     * @return XMLプロパティラッパー
     * @throws NoSuchResourceException
     *             リソースが見つからなかった場合
     */
    public static KoshoResources getXMLProperties(String path, ClassLoader loader) throws NoSuchResourceException {
        synchronized (Cache.xmlProperties) {
            KoshoResources resources = (KoshoResources) Cache.xmlProperties.get(path);
            if (resources == null) {
                resources = loadXMLProperties(path, loader);
                Cache.xmlProperties.put(path, resources);
            }
            return resources;
        }
    }

    /* XMLプロパティの読み取りを実行します。 */
    private static KoshoResources loadXMLProperties(String path, ClassLoader loader) throws NoSuchResourceException {
        InputStream inStream = ResourceLoader.getResourceAsStream(path, loader);
        XMLProperties resources = new XMLProperties();
        resources.setClassLoader(loader);
        resources.load(inStream);
        return resources;
    }

    // //////////////////////////////////////////////////////////////
    // XMLLocaleResourceBundle
    // //////////////////////////////////////////////////////////////

    /**
     * 指定された基底名、デフォルトのロケール、および呼び出し側のクラスローダを使用してXMLリソースバンドルを取得します。
     * 
     * @param baseName
     *            基底名
     * @return 指定された基底名とデフォルトのロケールのXMLリソースバンドル
     * @throws MissingResourceException
     *             指定された基底名のリソースバンドルが見つからない場合
     */
    public static KoshoResources getXMLLocaleResourceBundle(String baseName) {
        return XMLLocaleResourceBundle.getBundle(baseName);
    }

    /**
     * 指定された基底名、ロケール、および呼び出し側のクラスローダを使用してXMLリソースバンドルを取得します。
     * 
     * @param baseName
     *            基底名
     * @param locale
     *            ロケール
     * @return 指定された基底名とロケールのXMLリソースバンドル
     * @throws MissingResourceException
     *             指定された基底名のリソースバンドルが見つからない場合
     */
    public static KoshoResources getXMLLocaleResourceBundle(String baseName, Locale locale) {
        return XMLLocaleResourceBundle.getBundle(baseName, locale);
    }

    /**
     * 指定された基底名、ロケール、クラスローダを使用してXMLリソースバンドルを取得します。
     * 
     * @param baseName
     *            基底名
     * @param locale
     *            ロケール
     * @param loader
     *            リソースのロード元のクラスローダ
     * @return 指定された基底名とロケールのXMLリソースバンドル
     * @throws MissingResourceException
     *             指定された基底名のリソースバンドルが見つからない場合
     */
    public static KoshoResources getXMLLocaleResourceBundle(String baseName, Locale locale, ClassLoader loader) {
        return XMLLocaleResourceBundle.getBundle(baseName, locale, loader);
    }

    // //////////////////////////////////////////////////////////////
    // ClassBindConfiguration
    // //////////////////////////////////////////////////////////////

    /**
     * 指定されたクラス、呼び出し側のクラスローダを使用して <code>ClassBindConfiguration</code> を取得します。
     * 
     * @param key
     *            対応するクラス
     * @return 指定されたクラスに対応付けられた <code>ClassBindConfiguration</code>
     * @throws NoSuchResourceException
     *             指定されたクラスに対応付けられた定義ファイルが見つからない場合
     */
    public static KoshoResources getClassBindConfiguration(Class key) throws NoSuchResourceException {
        return getClassBindConfiguration(key, key.getClassLoader());
    }

    /**
     * 指定されたクラスとクラスローダを使用して <code>ClassBindConfiguration</code> を取得します。
     * 
     * @param key
     *            対応するクラス
     * @param loader
     *            クラスローダ
     * @return 指定されたクラスに対応付けられた <code>ClassBindConfiguration</code>
     * @throws NoSuchResourceException
     *             指定されたクラスに対応付けられた定義ファイルが見つからない場合
     */
    public static KoshoResources getClassBindConfiguration(Class key, ClassLoader loader)
            throws NoSuchResourceException {
        synchronized (Cache.classBindConfiguration) {
            KoshoResources resources = (KoshoResources) Cache.classBindConfiguration.get(key);
            if (resources == null) {
                resources = loadClassBindConfiguration(key, loader);
                Cache.classBindConfiguration.put(key, resources);
            }
            return resources;
        }
    }

    /* ClassBindConfiguration の読み取りを実行します。 */
    private static KoshoResources loadClassBindConfiguration(Class key, ClassLoader loader)
            throws NoSuchResourceException {
        String resource = key.getName().replace('.', '/') + "-config.xml";
        ClassBindConfiguration classBindConfig = new ClassBindConfiguration();
        InputStream inStream = ResourceLoader.getResourceAsStream(resource, loader);
        classBindConfig.load(inStream);
        return classBindConfig;
    }

    // //////////////////////////////////////////////////////////////
    // ClassConfiguration
    // //////////////////////////////////////////////////////////////

    /**
     * 指定された定義ファイルパス、呼び出し側のクラスローダを使用して、指定のクラスに対応する <code>ClassMappingConfiguration</code> を取得します。
     * 
     * @param path
     *            定義ファイルパス
     * @param key
     *            マッピングするクラス
     * @return 指定された <code>ClassMappingConfiguration</code>
     * @throws NoSuchResourceException
     *             指定された定義ファイル又はマッピングされたクラス定義が見つからない場合
     */
    public static KoshoResources getClassMappingConfiguration(String path, Class key) throws NoSuchResourceException {
        return getClassMappingConfiguration(path, key, key.getClassLoader());
    }

    /**
     * 指定された定義ファイルパスとクラスローダを使用して、指定のクラスに対応する <code>ClassMappingConfiguration</code> を取得します。
     * 
     * @param path
     *            定義ファイルパス
     * @param key
     *            マッピングするクラス
     * @param loader
     *            クラスローダ
     * @return 指定された <code>ClassMappingConfiguration</code>
     * @throws NoSuchResourceException
     *             指定された定義ファイル又はマッピングされたクラス定義が見つからない場合
     */
    public static KoshoResources getClassMappingConfiguration(String path, Class key, ClassLoader loader)
            throws NoSuchResourceException {
        synchronized (Cache.classConfiguration) {
            Entry entry = EntryUtils.arrayEntry(new Object[] { path, key.getName() });
            KoshoResources resources = (KoshoResources) Cache.classConfiguration.get(entry);
            if (resources == null) {
                loadClassMappingConfiguration(path, loader);
                resources = (KoshoResources) Cache.classConfiguration.get(entry);
            }
            if (resources == null) {
                throw new NoSuchResourceException("no such configuration resource. class:" + key + ", path:" + path);
            }
            return resources;
        }
    }

    /* ClassConfiguration の読み取りを実行します。 */
    private static void loadClassMappingConfiguration(String path, ClassLoader loader) throws NoSuchResourceException {
        InputStream inStream = ResourceLoader.getResourceAsStream(path, loader);
        ClassMappingConfiguration classConfiguration = new ClassMappingConfiguration();
        classConfiguration.load(inStream);
        synchronized (Cache.classConfiguration) {
            for (Iterator i = classConfiguration.classNameIterator(); i.hasNext();) {
                String className = (String) i.next();
                Entry entry = EntryUtils.arrayEntry(new Object[] { path, className });
                Cache.classConfiguration.put(entry, classConfiguration.getResources(className));
            }
        }
    }

    /**
     * デフォルトの構成ルールを生成して返却します。 <br>
     * カスタマイズされた構成ルールが定義されていない場合デフォルトを使用します。
     * 
     * @param c
     *            構成リソースを生成するクラス
     * @return デフォルトの構成ルール
     */
    public static NodeCompositeRule getDefaultNodeCompositeRule(Class c) {
        synchronized (Cache.defaultNodeCompositeRule) {
            NodeCompositeRule rule = (NodeCompositeRule) Cache.defaultNodeCompositeRule.get(c);
            if (rule == null) {
                rule = KoshoHelper.loadDefaultNodeCompositeRule(c, c.getClassLoader());
                Cache.defaultNodeCompositeRule.put(c, rule);
            }
            return rule;
        }
    }
}
