/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.kosho;

import java.io.InputStream;

import shohaku.core.lang.NoSuchResourceException;
import shohaku.core.lang.ResourceLoader;
import shohaku.core.util.XResourceBundleBaseCalculator;
import shohaku.ginkgo.Document;
import shohaku.ginkgo.DocumentCompositeRule;
import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.Node;
import shohaku.ginkgo.NodeCompositeRule;
import shohaku.ginkgo.helpers.NodeCompositeRuleLoader;

/**
 * 作業を省力化する為のヘルパー群を提供します。
 */
public class KoshoHelper {

    /**
     * リソースファイルの入力ストリームを生成して返却します。 <br>
     * リソースファイルを発見出来なかった場合<code>null</code>を返却します。
     * 
     * @param loader
     *            クラスリーダ
     * @param bundleName
     *            バンドル名
     * @return リソースファイルの入力ストリーム
     */
    public static InputStream getGinkgoBundleInputStream(ClassLoader loader, String bundleName) {

        // XMLファイルパスに変換します。
        final String resName = bundleName.replace('.', '/') + ".xml";

        // 特権を用いて強欲に検索します。
        InputStream stream = null;
        try {
            stream = ResourceLoader.getResourceAsStream(resName, loader);
        } catch (NoSuchResourceException e) {
            // no op
        }
        if (stream != null) {
            // make sure it is buffered
            return new java.io.BufferedInputStream(stream);
        }
        // 発見できなかった場合<code>null</code>を返却します。
        return null;
    }

    /**
     * 基底名の拡張リソースバンドルが利用する構成ルールを検索して返却します。 <br>
     * カスタマイズされた構成ルールが配置されていない場合デフォルトを返却します。
     * 
     * @param resourceBundle
     *            拡張リソースバンドルのクラス
     * @param loader
     *            クラスリーダ
     * @param baseName
     *            バンドル名
     * @return 構成ルール
     */
    public static NodeCompositeRule findGinkgoBundleNodeCompositeRule(Class resourceBundle, ClassLoader loader,
            String baseName) {

        // 構成ルールファイルパスに変換します。
        final String ruleName = baseName.replace('.', '/') + "_rule.xml";

        NodeCompositeRule rule = null;
        try {
            rule = NodeCompositeRuleLoader.getNodeCompositeRule(ruleName, loader);
        } catch (NoSuchResourceException e) {
            // no op
        }
        if (rule == null) {
            rule = loadDefaultNodeCompositeRule(resourceBundle, resourceBundle.getClassLoader());
        }

        return rule;
    }

    /**
     * 束縛基準の識別子をドキュメントIDとして返すドキュメント構成ルールを生成して返却します。
     * 
     * @param calculator
     *            束縛基準の算出機能
     * @param bundleBase
     *            束縛基準
     * @return ドキュメント構成ルール
     */
    public static DocumentCompositeRule getGinkgoBundleDocumentCompositeRule(XResourceBundleBaseCalculator calculator,
            Object bundleBase) {

        final String id = calculator.candidateBundleId(bundleBase);
        DocumentCompositeRule rule = new DocumentCompositeRule() {
            public String getDocumentId(Document document, Node root) {
                return id;
            }
        };
        return rule;

    }

    /*
     * NodeCompositeRule
     */

    /**
     * デフォルトの構成ルールを生成して返却します。 <br>
     * カスタマイズされた構成ルールが定義されていない場合デフォルトを使用します。
     * 
     * @param c
     *            構成リソースを生成するクラス
     * @param classLoader
     *            生成に使用するクラスローダ
     * @return デフォルトの構成ルール
     */
    public static NodeCompositeRule loadDefaultNodeCompositeRule(Class c, ClassLoader classLoader) {
        String ruleFileName = getRuleFileBaseName(c);

        // デフォルトのファイル名
        String packagePath = c.getPackage().getName().replace('.', '/');
        String defaultRuleFile = packagePath + "/" + ruleFileName + "-default-rule.xml";

        // カスタマイズされたファイル名
        String customRuleFile = ruleFileName + "-rule.xml";

        NodeCompositeRule rule = null;
        try {
            rule = NodeCompositeRuleLoader.getNodeCompositeRule(customRuleFile, classLoader);
        } catch (NoSuchResourceException e) {
            // no op
        }
        if (rule == null) {
            try {
                rule = NodeCompositeRuleLoader.getNodeCompositeRule(defaultRuleFile, classLoader);
            } catch (NoSuchResourceException e) {
                throw new GinkgoException("It failed in reading of the file of the composition rule.", e);
            }
        }
        return rule;
    }

    /* 大文字を '-' + 小文字に変換する */
    private static String getRuleFileBaseName(Class c) {

        StringBuffer buf = new StringBuffer();

        String className = c.getName();
        int off = className.lastIndexOf('.');
        if (0 <= off) {
            off++;
        }
        char[] chars = className.substring(off).toCharArray();

        for (int i = 0; i < chars.length; i++) {
            char ch = chars[i];
            if (0 < i && 'A' <= ch && ch <= 'Z') {
                if ((i + 1) < chars.length) {
                    char next = chars[i + 1];
                    if ('a' <= next && next <= 'z') {
                        buf.append('-');
                    }
                } else {
                    buf.append('-');
                }
            }
            buf.append(Character.toLowerCase(ch));
        }

        return buf.toString();
    }

}
