/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.kosho;

import java.io.InputStream;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;

import shohaku.core.util.XResourceBundle;
import shohaku.core.util.XResourceBundleCache;
import shohaku.ginkgo.Document;
import shohaku.ginkgo.DocumentCompositeRule;
import shohaku.ginkgo.NodeCompositeRule;

/**
 * XMLで定義されたプロパティを用いたロケールで階層化するリソースバンドルを提供します。
 * <P>
 * <b>XMLリソースバンドルの特徴 </b>
 * <p>
 * XMLリソースバンドルには、ロケール固有のオブジェクトが含まれます。 プログラムで<code>String</code>などのロケール固有のリソースが必要なときは、
 * ユーザのロケールに合ったXMLリソースバンドルからロードできます。 このように、XMLリソースバンドルから、ロケール固有の情報のすべてでなくてもその大部分を切り離すことで、
 * ユーザのロケールにはほとんど依存しないプログラムコードを書くことができます。
 * </p>
 * これにより、以下の特徴を持つプログラムを書くことが可能になります。
 * <UL type=SQUARE>
 * <LI>異なる言語への地域対応、または翻訳が容易
 * <LI>複数のロケールを同時に処理可能
 * <LI>将来サポートするロケールを追加する際の修正が容易
 * </UL>
 * <p>
 * この特徴は java.util.ResourceBundle と同様のものであり、XMLリソースバンドルの設計は<code>java.util.ResourceBundle</code>の国際化モデルを継承しています。
 * 大きな相違点は<code>java.util.ResourceBundle</code>が<code>java.util.Properties</code>を内部的に使用するのに対し、XMLリソースバンドルは
 * <code>shohaku.kosho.XMLProperties</code>をデータモデルに使用している点です。
 * </p>
 * この機能により<code>java.util.ResourceBundle</code>と同等の柔軟性と、より複雑なデータ型と複雑な階層データモデルを利用して情報を外部ファイル化する事が出来ます。 <br>
 * XMLリソースバンドルは内部的に<code>shohaku.kosho.XMLProperties</code>を使用するため、解析ルールを独自にカスタマイズすることが出来ます。 <br>
 * 解析ルールの指定に関しては後記の「XMLリソースバンドル固有の名前規約」で説明します。
 * <P>
 * XMLリソースバンドルの基底名と各ロケールをファミリとして定義する名前規約に関しては<code>java.util.ResourceBundle</code>を参照してください。
 * <P>
 * <b>XMLリソースバンドル固有の名前規約 </b>
 * <p>
 * XMLリソースバンドルには唯一<code>java.util.ResourceBundle</code>とは別の名前規約が有ります。 <br>
 * XMLリソースバンドルでは基底名を基にして複数のファイルが読取の対象となるため、同一の基底名に属すファイルは同一の解析ルールが摘要される必要があります。 <br>
 * よって基底名単位で解析ルールを指定する為の以下の配置規約が定義されています。
 * <UL type=SQUARE>
 * <LI>基底名単位で解析ルールを指定する場合のファイル名は '基底名 + _rule.properties' で指定します。
 * </UL>
 * 上記位置に配置しておけばシステムが自動で読み取ります, 配置されていなければデフォルトが使用されます。
 * </p>
 * <P>
 * <b>XMLリソースバンドル固有の問題 </b>
 * <p>
 * XMLリソースバンドルは複数のデータ型に対応する一方で<code>java.util.ResourceBundle</code>と同様に、
 * 上位階層から値が継承されるモデルを採用しているため、上位階層で定義される型とは別の型を下位の階層が定義する可能性が有ります。 <br>
 * この特徴はデータ型の管理を難しくする可能性があります。 この特徴を踏まえて設計および管理を行うことが推奨されます。 <br>
 * DTDファイルを作成して検証を行う、使用する解析ルールの型制約を厳しくする、ドキュメントレベルで厳密に規定する等、何らかの補助を設ける事で回避することが出来ると考えます。
 * </p>
 */
public class XMLLocaleResourceBundle extends AbstractGinkgoResourceBundle {

    /* 拡張リソースバンドルのキャッシュ機能のシングルトンインスタンスを格納します。 */
    private static final class XResourceBundleCacheInstance {

        private static final class XMLPropertiesResourceBundleCreater extends AbstractGinkgoResourceBundleCreater {
            protected AbstractGinkgoResourceBundle getGinkgoResourceBundle(ClassLoader loader, XResourceBundle parent,
                    Document parentDoc, Object bundleBase, InputStream stream, DocumentCompositeRule docRule,
                    NodeCompositeRule nodeRule) {

                XMLProperties props = new XMLProperties(docRule);
                props.setClassLoader(loader);
                props.load(stream, nodeRule, parentDoc);
                Map resources = props.toMap();
                return new XMLLocaleResourceBundle(parent, bundleBase, resources, props.getDocument());

            }

            protected Class getGinkgoResourceBundleClass() {
                return XMLLocaleResourceBundle.class;
            }
        }

        // new Singleton Instance
        static final XResourceBundleCache instance = new XResourceBundleCache(new XMLPropertiesResourceBundleCreater());
    }

    /**
     * プロパティリストを初期化します。
     * 
     * @param parent
     *            親バンドル
     * @param bundleBase
     *            束縛基準
     * @param resources
     *            リソース
     * @param document
     *            ドキュメント
     */
    XMLLocaleResourceBundle(XResourceBundle parent, Object bundleBase, Map resources, Document document) {
        super(parent, bundleBase, resources, document);
    }

    /**
     * ロケールを取得して返却します。
     * 
     * @return ロケール
     */
    public Locale getLocale() {
        return (Locale) getBundleBase();
    }

    /*
     * static
     */

    /**
     * 指定された基底名、デフォルトのロケール、および呼び出し側のクラスローダを使用して、リソースバンドルを取得します。 <br>
     * このメソッドを呼び出すことは、以下を呼び出すことと同じです。 <br>
     * getBundle(baseName, Locale.getDefault(), XMLLocaleResourceBundle.class.getClassLoader()) <br>
     * 検索とインスタンス生成方法の詳細については、getBundle を参照してください。
     * 
     * @param baseName
     *            基底名
     * @return 指定された基底名とデフォルトのロケールの拡張リソースバンドル
     * @throws MissingResourceException
     *             指定された基底名のリソースバンドルが見つからない場合
     */
    public static XMLLocaleResourceBundle getBundle(String baseName) {
        return getBundle(baseName, Locale.getDefault(), getDefaultLoader());
    }

    /**
     * 指定された基底名、クラスローダ、およびデフォルトのロケールを使用して、リソースバンドルを取得します。 <br>
     * このメソッドを呼び出すことは、以下を呼び出すことと同じです。 <br>
     * getBundle(baseName, Locale.getDefault(), loader) <br>
     * 検索とインスタンス生成方法の詳細については、getBundle を参照してください。
     * 
     * @param baseName
     *            基底名
     * @param loader
     *            リソースのロード元のクラスローダ
     * @return 指定された基底名とロケールの拡張リソースバンドル
     * @throws MissingResourceException
     *             指定された基底名のリソースバンドルが見つからない場合
     */
    public static XMLLocaleResourceBundle getBundle(String baseName, ClassLoader loader) {
        return getBundle(baseName, Locale.getDefault(), loader);
    }

    /**
     * 指定された基底名、ロケール、および呼び出し側のクラスローダを使用して、リソースバンドルを取得します。 <br>
     * このメソッドを呼び出すことは、以下を呼び出すことと同じです。 <br>
     * getBundle(baseName, locale, XMLLocaleResourceBundle.class.getClassLoader()) <br>
     * 検索とインスタンス生成方法の詳細については、getBundle を参照してください。
     * 
     * @param baseName
     *            基底名
     * @param locale
     *            ロケール
     * @return 指定された基底名とロケールの拡張リソースバンドル
     * @throws MissingResourceException
     *             指定された基底名のリソースバンドルが見つからない場合
     */
    public static XMLLocaleResourceBundle getBundle(String baseName, Locale locale) {
        return getBundle(baseName, locale, getDefaultLoader());
    }

    /**
     * 指定された基底名、ロケール、クラスローダを使用して、拡張リソースバンドルを取得します。
     * <p>
     * 理論的には<code>getBundle</code>では次の方法を使用して、拡張リソースバンドルの検出および生成を行います。
     * <p>
     * <code>getBundle</code>は、基底名、指定されたロケール、およびデフォルトのロケール (<CODE>Locale.getDefault</CODE> から取得したロケール) を使用して、
     * 候補のバンドル名のシーケンスを生成します。 <br>
     * 指定されたロケールの言語、国、およびバリアントがすべて空の文字列の場合、基底名は候補のバンドル名のみになります。 <br>
     * それ以外の場合には、指定したロケール (language1、country1、variant1) およびデフォルトのロケール (language2、country2、variant2)
     * の属性値から、次のシーケンスが生成されます。
     * <ul>
     * <li>baseName + "_" + language1 + "_" + country1 + "_" + variant1
     * <li>baseName + "_" + language1 + "_" + country1
     * <li>baseName + "_" + language1
     * <li>baseName + "_" + language2 + "_" + country2 + "_" + variant2
     * <li>baseName + "_" + language2 + "_" + country2
     * <li>baseName + "_" + language2
     * <li>baseName
     * </ul>
     * <p>
     * 最終コンポーネントが空の文字列の場合、候補のバンドル名は省略されます。 <br>
     * たとえば、country1 が空の文字列の場合、2 番目の候補のバンドル名は省略されます。
     * <p>
     * <code>getBundle</code>は実際の拡張リソースバンドルを「インスタンス化」できる最初の名前を見つけるために、
     * 候補のバンドル名を繰り返し処理します。各候補バンドル名ごとに、拡張リソースバンドルを次のように作成しようとします。
     * <ul>
     * <li>まず、候補のバンドル名を使用して、クラスをロードしようとします。 <br>
     * 指定したクラスローダを使用して、XMLプロパティリソースファイルを見つけようとします。 <br>
     * すべての「。」文字を「/」文字に置き換え、「。xml」文字列を追加して、候補のバンドル名からパス名を生成します。 <br>
     * <CODE>ClassLoader.getResource</CODE> を使用して、この名前の「リソース」を見つけようとします (<code>getResource</code>
     * の「リソース」とは、拡張リソースバンドルのコンテンツとは無関係であり、ファイルなどのデータのコンテナにすぎません)。 <br>
     * 「リソース」を見つけた場合、リソースのコンテンツから新しい <CODE>XResourceBundle</CODE> インスタンスを作成します。
     * </ul>
     * <p>
     * 拡張リソースバンドルが見つからない場合<code>MissingResourceException</code>がスローされます。
     * <p>
     * 拡張リソースバンドルが見つかったときは、その親連鎖をインスタンス化します。 <br>
     * <code>getBundle</code>は候補のバンドル名を繰り返し処理します。 <br>
     * このバンドル名は、拡張リソースバンドルのバンドル名からバリアント、国、および言語を (前に「_」についたものに関して毎回) 連続して削除することによって、取得します。 <br>
     * 上述のように、最終コンポーネントが空の文字列である場合、候補のバンドル名は省略されます。 <br>
     * 各候補のバンドル名により、上述のように、拡張リソースバンドルをインスタンス化しようとします。 <br>
     * インスタンスに成功した場合は常に、前にインスタント化した拡張リソースバンドルに null 以外の親拡張リソースバンドルがない場合は <br>
     * 前にインスタンス化した拡張リソースバンドルの <CODE>setParent</CODE> メソッドを新しい拡張リソースバンドルにより呼び出します。
     * <p>
     * <code>getBundle</code>の実装はインスタンス化した拡張リソースバンドルをキャッシュし、同じ拡張リソースバンドルのインスタンスを複数回返却します。 <br>
     * こうした実装では、拡張リソースバンドルおよびその親連鎖の選択が上述のものと互換性がある限り、拡張リソースバンドルをインスタンス化するシーケンスを変更する場合もあります。
     * <p>
     * <code>baseName</code>引数は完全指定のクラス名である必要があります。
     * <p>
     * <strong>例: </strong>
     * XMLプロパティファイル、MyResources.xml、MyResources_fr_CH.xml、MyResources_fr.xml、MyResources_en.xml、が提供されています。 <br>
     * すべてのファイルのコンテンツが有効です。 デフォルトのロケールは<code>Locale("en", "UK")</code>です。
     * <p>
     * 表示されたロケール引数値を持つ<code>getBundle</code>の呼び出しは、次のソースから拡張リソースバンドルをインスタンス化します。
     * <ul>
     * <li>Locale("fr", "CH"): result MyResources_fr_CH.xml, parent MyResources_fr.xml, parent MyResources.xml
     * <li>Locale("fr", "FR"): result MyResources_fr.xml, parent MyResources.xml
     * <li>Locale("de", "DE"): result MyResources_en.xml, parent MyResources.xml
     * <li>Locale("en", "US"): result MyResources_en.xml, parent MyResources.xml
     * </ul>
     * <p>
     * 
     * @param baseName
     *            基底名
     * @param locale
     *            ロケール
     * @param loader
     *            リソースのロード元のクラスローダ
     * @return 指定された基底名とロケールの拡張リソースバンドル
     * @throws MissingResourceException
     *             指定された基底名のリソースバンドルが見つからない場合
     */
    public static XMLLocaleResourceBundle getBundle(String baseName, Locale locale, ClassLoader loader) {
        return (XMLLocaleResourceBundle) getXResourceBundleCache().getBundle(baseName, locale, loader);
    }

    /**
     * デフォルトの構成ルールを返却します。
     * 
     * @return デフォルトの構成ルール
     */
    static NodeCompositeRule getDefaultNodeCompositeRule() {
        return Kosho.getDefaultNodeCompositeRule(XMLLocaleResourceBundle.class);
    }

    /**
     * リソースに使用するデフォルトのクラスローダを返却します。
     * 
     * @return デフォルトのクラスローダ
     */
    static ClassLoader getDefaultLoader() {
        return getXResourceBundleCache().getDefaultLoader();
    }

    /**
     * キャッシュ機能のシングルトンインスタンスを返却します。
     * 
     * @return キャッシュ機能のシングルトンインスタンス
     */
    static XResourceBundleCache getXResourceBundleCache() {
        return XResourceBundleCacheInstance.instance;
    }

}
