/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.kosho;

import java.io.InputStream;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

import shohaku.core.helpers.Eval;
import shohaku.ginkgo.Document;
import shohaku.ginkgo.DocumentCompositeRule;
import shohaku.ginkgo.Ginkgo;
import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.Node;
import shohaku.ginkgo.NodeCompositeRule;

/**
 * XMLで定義されたプロパティを読み込み保存する機能を提供します。 <br>
 * <p>
 * 例：
 * 
 * <pre>
 * &lt;xml-properties&gt;
 *     &lt;content&gt;         
 *         &lt;byte name="byte"&gt;-123&lt;/byte&gt;
 *         &lt;double name="double"&gt;+1,521,414.012411&lt;/double&gt;
 *         &lt;int name="int"&gt;100&lt;/int&gt;
 *         &lt;string name="string"&gt;文字列&lt;/string&gt;
 *         &lt;ns name="foo"&gt;
 *             &lt;list id="season" desc="四季"&gt;
 *                 &lt;string&gt;春&lt;/ref&gt;
 *                 &lt;string&gt;夏&lt;/ref&gt;
 *                 &lt;string&gt;秋&lt;/ref&gt;
 *                 &lt;string&gt;冬&lt;/ref&gt;
 *             &lt;/list&gt;
 *         &lt;/ns"&gt;
 *     &lt;/content&gt;
 * &lt;/xml-properties&gt;
 * </pre>
 * 
 * は以下の様に '/' で区切られた名前階層パスにマッピングされ生成されます。 <br>
 * 
 * <pre>
 * /byte   = -123
 * /double = 1521414.012411
 * /int    = 100
 * /string = "文字列"
 * /foo/season = ["春", "夏", "秋", "冬"]
 * </pre>
 * 
 * 有効な階層の深さは定義ファイルで変更出来ますが、デフォルトは４階層です。
 */
public class XMLProperties implements KoshoResources, KoshoResourcesLoader {

    /** プロパティを格納します。 */
    private final Map lookup;

    /** プロパティを読取り専用で格納します。 */
    private final Map unmodifiableLookup;

    /** Ginkgo。 */
    private final Ginkgo ginkgo;

    /** 解析処理に使用するClassLoader. */
    private ClassLoader classLoader;

    /**
     * デフォルトコンストラクタ。
     */
    public XMLProperties() {
        this.ginkgo = new Ginkgo();
        this.lookup = new LinkedHashMap();
        this.unmodifiableLookup = Collections.unmodifiableMap(this.lookup);
    }

    /**
     * ドキュメントの合成ルールを指定して初期化します。
     * 
     * @param docRule
     *            ドキュメントの合成ルール
     */
    public XMLProperties(DocumentCompositeRule docRule) {
        this.ginkgo = new Ginkgo(docRule);
        this.lookup = new LinkedHashMap();
        this.unmodifiableLookup = Collections.unmodifiableMap(this.lookup);
    }

    /**
     * 解析処理に使用する ClassLoader を返却します.
     * 
     * @return 解析処理に使用する ClassLoader
     */
    public ClassLoader getClassLoader() {
        return this.classLoader;
    }

    /**
     * 解析処理に使用する ClassLoader を設定します.
     * 
     * @param classLoader
     *            解析処理に使用する ClassLoader
     */
    public void setClassLoader(ClassLoader classLoader) {
        this.classLoader = classLoader;

    }

    /**
     * 入力ストリームからプロパティリストを読み込みます。
     * 
     * @param inStream
     *            プロパティの入力ストリーム
     * @throws GinkgoException
     *             構成情報例外
     */
    public synchronized void load(InputStream inStream) throws GinkgoException {
        load(inStream, getDefaultNodeCompositeRule());
    }

    /**
     * 入力ストリームからプロパティリストを読み込みます。
     * 
     * @param inStream
     *            プロパティの入力ストリーム
     * @param content
     *            親のドキュメントオブジェクト
     * @throws GinkgoException
     *             構成情報例外
     */
    public synchronized void load(InputStream inStream, Document content) throws GinkgoException {
        load(inStream, getDefaultNodeCompositeRule(), content);
    }

    /**
     * 入力ストリームからプロパティリストを読み込みます。
     * 
     * @param inStream
     *            プロパティの入力ストリーム
     * @param rule
     *            構成ルール
     * @throws GinkgoException
     *             構成情報例外
     */
    public synchronized void load(InputStream inStream, NodeCompositeRule rule) throws GinkgoException {
        load(inStream, rule, null);
    }

    /**
     * 入力ストリームからプロパティリストを読み込みます。
     * 
     * @param inStream
     *            プロパティの入力ストリーム
     * @param rule
     *            構成ルール
     * @param content
     *            親のドキュメントオブジェクト
     * @throws GinkgoException
     *             構成情報例外
     */
    public synchronized void load(InputStream inStream, NodeCompositeRule rule, Document content)
            throws GinkgoException {
        // XMLの解析
        ginkgo.parse(rule, inStream, content);
        Node root = ginkgo.getDocument().getContext().getRoot();
        initValues(root);
    }

    /* プロパティの初期化および登録をします。 */
    private void initValues(Node root) {
        Iterator i = root.getContext().elementIterator("content");
        while (i.hasNext()) {
            Node content = (Node) i.next();
            initValues(content, content);
            break;
        }
    }

    /* 再帰的にノードを読み取り値を登録します。 */
    private void initValues(Node root, Node content) {

        // 子のノードを全登録
        Iterator i = content.getContext().ownerIterator();
        while (i.hasNext()) {
            Node n = (Node) i.next();
            if (n.isType(Node.TYPE_CONTAINER)) {
                // 再起呼出
                initValues(root, n);

            } else if (n.isType(Node.TYPE_VALUE)) {
                String path = getPath(root, n);
                if (path != null) {
                    saveValue(path, n);
                }
            }

        }
    }

    private void saveValue(Object path, Node node) {
        this.lookup.put(path, node.getNodeValue());
    }

    /* この情報のアドレスを返す。 */
    private String getPath(Node root, Node node) {

        // 名前階層へ参加する情報は親情報から再帰処理で構築する
        StringBuffer sb = new StringBuffer(80);
        Node c = node;
        while (c != root && c != null) {
            String name = getNodeName(c);
            if (Eval.isBlank(name)) {
                return null;
            }
            sb.insert(0, name);
            sb.insert(0, '/');
            c = c.getContext().getParent();
        }
        if (node.isType(Node.TYPE_CONTAINER)) {
            sb.append('/');
        }
        return sb.toString();
    }

    /* ノード名を返却します。 */
    private String getNodeName(Node node) {
        String name = null;
        if (!node.isType(Node.TYPE_TEXT)) {
            name = node.getAttribute("name");
            if (Eval.isBlank(name)) {
                name = null;
            }
        } else {
            name = "#TEXT#";
        }
        return name;
    }

    /**
     * 解析中又は直前に解析したドキュメントを返却します。
     * 
     * @return 解析中又は直前に解析したドキュメント
     * @see shohaku.ginkgo.Document
     * @see shohaku.ginkgo.Ginkgo#getDocument()
     */
    public Document getDocument() {
        return ginkgo.getDocument();
    }

    /*
     * utils
     */

    /**
     * 全てのプロパティをMapに格納して返却します。 <br>
     * 読取専用Mapで返却します。
     * 
     * @return 全てのプロパティ
     */
    public Map toMap() {
        return unmodifiableLookup;
    }

    /**
     * 全てのプロパティ名を返す。
     * 
     * @return 全てのプロパティ名
     */
    public Set getNames() {
        return unmodifiableLookup.keySet();
    }

    /*
     * implements KoshoResources
     */

    /**
     * 全てのプロパティキーを含む反復子を返却します。
     * 
     * @return 全てのプロパティキーを含む反復子
     */
    public Iterator keyIterator() {
        return getNames().iterator();
    }

    /**
     * プロパティキーが示す値を返却します。 指定されたキーが存在しない場合<code>null</code>返却します。
     * 
     * @param key
     *            プロパティ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     */
    public Object getObject(String key) {
        if (key == null) {
            throw new NullPointerException("resource key as null");
        }
        Object obj = lookup.get(key);
        return obj;
    }

    /**
     * プロパティキーが示す値を返却します。 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            プロパティキー
     * @param defaultValue
     *            プロパティキー
     * @return プロパティキーが示す値
     * @throws NullPointerException
     *             key が null の場合発生する
     */
    public Object getObject(String key, Object defaultValue) {
        Object o = getObject(key);
        if (o == null) {
            o = defaultValue;
        }
        return o;
    }

    /**
     * 指定されたキーがプロパティセットに含まれている場合ni<code>true</code>を返す。
     * 
     * @param key
     *            プロパティキー
     * @return 指定されたキーが含まれている場合 true
     * @throws NullPointerException
     *             key が null の場合発生する
     */
    public boolean containsKey(String key) {
        if (key == null) {
            throw new NullPointerException("resource key as null");
        }
        return lookup.containsKey(key);
    }

    /**
     * 指定されたキーがプロパティセットに含まれていると同時に、指定されたクラスとキャスト可能な関係に有る場合に<code>true</code>を返す。 値がNullの場合キャスト可能(True)を返却します。
     * 
     * @param key
     *            プロパティキー
     * @param type
     *            キャスト可能な関係に有るか検証するクラス
     * @return 指定されたキーが含まれている場合 true
     * @throws NullPointerException
     *             key または type が null の場合発生する
     */
    public boolean containsKey(String key, Class type) {
        if (Eval.isOrNull(key, type)) {
            throw new NullPointerException("resource key or type as null");
        }
        Object o = getObject(key);
        if (o != null) {
            return type.isInstance(o);
        } else {
            return containsKey(key);
        }
    }

    /*
     * static
     */

    /**
     * デフォルトの構成ルールを返却します。
     * 
     * @return デフォルトの構成ルール
     */
    public static NodeCompositeRule getDefaultNodeCompositeRule() {
        return Kosho.getDefaultNodeCompositeRule(XMLProperties.class);
    }

}
