/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.sample.kosho;

import java.io.InputStream;

import shohaku.core.beans.BindMethod;
import shohaku.core.beans.InvocationBeansException;
import shohaku.core.collections.params.Parameters;
import shohaku.core.lang.IllegalIdentifierException;
import shohaku.core.lang.NoSuchResourceException;
import shohaku.core.lang.ObjectConversionException;
import shohaku.core.lang.ResourceLoader;
import shohaku.core.util.debug.Debug;
import shohaku.kosho.BindMethodFactory;

/**
 * shohaku.kosho.BindMethodFactory の実装サンプルです。定義ファイルは同フォルダ下の bind-method-converter.xml です。<br>
 * 識別子から定義される変換メソッドを呼び出して結果を返します。<br>
 * サンプルでは一般的なデータ型の変換を扱いますが、多彩な変換処理を任意に追加出来ます。<br>
 * <br>
 * また構成ルールを独自に定義すれば使用する機能の縮小や拡張を自在に変更する事が出来ます。<br>
 * また必要に応じて（使用可能なタグの種別等の）制約を追加する事が可能です。<br>
 * デフォルト構成ルールは conf フォルダ下の"shohaku/kosho//bind-method-factory-default-rule.xml" に配置されています。
 */
public class BindMethodConverter {

    /**
     * サンプルを実行します、処理のコンソール出力を行います。
     * 
     * @param args
     */
    public static void main(String[] args) {
        Object retValue = null;
        try {
            Debug.out.printLineln(60);
            Debug.out.println("日付変換を実行します。");
            Debug.out.printLineln(60);
            Debug.out.printlf(1);

            Debug.out.println("id = date:ymd, 変換値 = 2005-12-01");
            retValue = BindMethodConverter.convert("date:ymd", "2005-12-01");
            Debug.out.println("結果", retValue);
            Debug.out.printlf(1);

            Debug.out.println("id = date:ymd, 変換値 = 2012/01/01");
            retValue = BindMethodConverter.convert("date:ymd", "2012/01/01");
            Debug.out.println("結果", retValue);
            Debug.out.printlf(1);

            Debug.out.println("id = date:time, 変換値 = 20:41:55");
            retValue = BindMethodConverter.convert("date:time", "20:41:55");
            Debug.out.println("結果", retValue);
            Debug.out.printlf(1);

            Debug.out.printLineln(60);
            Debug.out.println("不正な書式で指定します、この場合 NULL を返します。");
            Debug.out.printLineln(60);
            Debug.out.printlf(1);

            Debug.out.println("id = date:ymd, 変換値 = 2005,12,01, <コンマは不正>");
            retValue = BindMethodConverter.convert("date:ymd", "2005,12,01");
            Debug.out.println("結果", retValue);
            Debug.out.printlf(1);

            Debug.out.println("id = date:time, 変換値 = 25:42:08, <時刻は 0-23 の範囲のみ有効>");
            retValue = BindMethodConverter.convert("date:time", "25:42:08");
            Debug.out.println("結果", retValue);
            Debug.out.printlf(1);

            Debug.out.printLineln(60);
            Debug.out.println("日付変換のエラーを発生させます。");
            Debug.out.printLineln(60);
            Debug.out.printlf(1);

            Debug.out.println("id = date:ymd, 変換値 = new Long(112222), <文字列以外不正>");
            try {
                retValue = BindMethodConverter.convert("date:ymd", new Long(112222));
            } catch (Exception e) {
                Debug.out.println("結果", e);
            }
            Debug.out.printlf(1);

            Debug.out.printlf(2);

            Debug.out.printLineln(60);
            Debug.out.println("数値変換を実行します（国際化対応の例）。");
            Debug.out.printLineln(60);
            Debug.out.printlf(1);

            Debug.out.println("id = num:jp, 変換値 = ￥23.22");
            retValue = BindMethodConverter.convert("num:jp", "￥23.22");
            Debug.out.println("結果", retValue);
            Debug.out.printlf(1);

            Debug.out.println("id = num:jp, 変換値 = ￥２３.２２");
            retValue = BindMethodConverter.convert("num:jp", "￥２３.２２");
            Debug.out.println("結果", retValue);
            Debug.out.printlf(1);

            Debug.out.println("id = num:en, 変換値 = $23.22");
            retValue = BindMethodConverter.convert("num:en", "$23.22");
            Debug.out.println("結果", retValue);
            Debug.out.printlf(1);

            Debug.out.printLineln(60);
            Debug.out.println("不正な書式で指定します、この場合 NULL を返します。");
            Debug.out.printLineln(60);
            Debug.out.printlf(1);

            Debug.out.println("id = num:jp, 変換値 = \\23.22, <１バイト円マーク不正>");
            retValue = BindMethodConverter.convert("num:jp", "\\23.22");
            Debug.out.println("結果", retValue);
            Debug.out.printlf(1);

            Debug.out.println("id = num:en, 変換値 = ＄23.22, <２バイト$マーク不正>");
            retValue = BindMethodConverter.convert("num:en", "＄23.22");
            Debug.out.println("結果", retValue);
            Debug.out.printlf(1);

            Debug.out.printLineln(60);
            Debug.out.println("数値変換のエラーを発生させます。");
            Debug.out.printLineln(60);
            Debug.out.printlf(1);

            Debug.out.println("id = num:jp, 変換値 = new Double(23.22), <文字列以外不正>");
            try {
                retValue = BindMethodConverter.convert("num:jp", new Double(23.22));
            } catch (Exception e) {
                Debug.out.println("結果", e);
            }
            Debug.out.printlf(1);

        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 変換するメソッドの拘束情報を保管するクラス。
     */
    private static class Factory {
        static final BindMethodFactory instance;
        static {
            BindMethodFactory factory = new BindMethodFactory();
            try {
                InputStream inStream = null;
                inStream = ResourceLoader.getResourceAsStream("shohaku/sample/kosho/bind-method-converter.xml");
                factory.load(inStream);
            } catch (NoSuchResourceException e) {
                e.printStackTrace();
            }
            instance = factory;
        }
    }

    /**
     * 識別子で定義される変換処理を実行してい結果を返却します。
     * 
     * @param id
     *            処理の識別子
     * @param value
     *            変換する値
     * @return 変換結果の値
     * @throws ObjectConversionException
     *             何らかの理由から変換に失敗した場合
     */
    public static Object convert(String id, Object value) throws ObjectConversionException {
        if (id == null) {
            throw new NullPointerException("id as null");
        }
        BindMethod bindMethod = getBindMethod(id);
        if (bindMethod == null) {
            throw new IllegalIdentifierException("no definition id. " + id);
        }
        Object retValue = null;
        try {
            retValue = bindMethod.invoke(new Parameters(new Object[] { "value", value }));
        } catch (InvocationBeansException e) {
            throw new ObjectConversionException(e);
        }
        return retValue;
    }

    /**
     * 指定された識別子の拘束されたメソッド情報を取得します。
     * 
     * @param id
     *            識別子
     * @return 指定された識別子の拘束されたメソッド情報
     */
    private static BindMethod getBindMethod(String id) {
        return Factory.instance.getBindMethod(id);
    }
}
