/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.ginkgo;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.NoSuchElementException;

import shohaku.core.collections.ListUtils;
import shohaku.core.collections.params.Parameters;

/**
 * ノードのコンテキスト情報を提供します。
 */
public class NodeContext {

    /* ドキュメント。 */
    private Document document;

    /* ノードの設定情報。 */
    private NodeRule nodeRule;

    /* 親のノードを保管します。 */
    private Node parent;

    /* ノードの階層URI. */
    private String nodeURI;

    /* ノードの名前空間。 */
    private String nodeNamespaceURI;

    /* ノードのローカル名。 */
    private String nodeLocalName;

    /* ノード名。 */
    private String nodeQName;

    /* ノード名。 */
    private String nodeName;

    /* ノードの全属性。 */
    private TagAttributes attributes;

    /* 子のノードを保管します。 */
    private List children;

    /**
     * エレメントノードの情報を格納して初期化します。
     * 
     * @param document
     *            ドキュメント
     * @param nodeRule
     *            ノードの設定情報
     * @param parent
     *            親ノード
     * @param uri
     *            階層URI
     * @param namespace
     *            名前空間
     * @param localName
     *            ローカル名
     * @param qName
     *            タグ名
     * @param nodeName
     *            ノード名
     * @param attributes
     *            属性情報
     */
    NodeContext(Document document, NodeRule nodeRule, Node parent, String uri, String namespace, String localName,
            String qName, String nodeName, TagAttributes attributes) {
        this.document = document;
        this.nodeRule = nodeRule;
        this.parent = parent;
        this.nodeURI = uri;
        this.nodeNamespaceURI = namespace;
        this.nodeLocalName = localName;
        this.nodeQName = qName;
        this.nodeName = nodeName;
        this.attributes = attributes;
        this.children = new LinkedList();
    }

    /**
     * ドキュメントを返却します。
     * 
     * @return ドキュメント
     */
    public Document getDocument() {
        return document;
    }

    /**
     * <code>Ginkgo</code>を返却します。
     * 
     * @return <code>Ginkgo</code>
     */
    public Ginkgo getGinkgo() {
        return getDocumentContext().getGinkgo();
    }

    /**
     * 解析処理に使用する構成ルールを返却します。
     * 
     * @return 解析処理に使用する構成ルール
     */
    public NodeCompositeRule getNodeCompositeRule() {
        return getDocumentContext().getNodeCompositeRule();
    }

    /**
     * ドキュメントのコンテキスト情報を返却します。
     * 
     * @return ドキュメントのコンテキスト情報
     */
    public DocumentContext getDocumentContext() {
        return getDocument().getContext();
    }

    /**
     * のコンテキスト情報を返却します。
     * 
     * @return のコンテキスト情報
     */
    public DocumentCompositeRule getDocumentCompositeRule() {
        return getGinkgo().getDocumentCompositeRule();
    }

    /**
     * 解析処理に使用する<code>ClassLoader</code>を返却します.
     * 
     * @return 解析処理に使用する<code>ClassLoader</code>
     */
    public ClassLoader getClassLoader() {
        return getGinkgo().getClassLoader();
    }

    /**
     * タグ名を返却します。
     * 
     * @return タグ名
     */
    public String getNodeQName() {
        return nodeQName;
    }

    /**
     * タグのローカル名を返却します。
     * 
     * @return タグのローカル名
     */
    public String getNodeLocalName() {
        return nodeLocalName;
    }

    /**
     * ノード名を返却する（プレフィックスを含まないタグ名）。
     * 
     * @return ノード名
     */
    public String getNodeName() {
        return nodeName;
    }

    /**
     * タグの名前空間を返却します。
     * 
     * @return タグの名前空間
     */
    public String getNodeNamespaceURI() {
        return nodeNamespaceURI;
    }

    /**
     * タグの階層URIを返却します。
     * 
     * @return タグの階層URI
     */
    public String getNodeURI() {
        return nodeURI;
    }

    /**
     * 親のノードを格納します。
     * 
     * @return 親のノード
     */
    public Node getParent() {
        return this.parent;
    }

    /**
     * 子の要素が空の場合<code>true</code>を返却します。
     * 
     * @return 子の要素が空の場合<code>true</code>
     */
    public boolean isChildEmpty() {
        return this.children.isEmpty();
    }

    /**
     * 子の要素のエレメントノードが空の場合<code>true</code>を返却します。
     * 
     * @return 子の要素のエレメントノードが空の場合<code>true</code>
     */
    public boolean isElementEmpty() {
        int count = 0;
        for (Iterator i = elementIterator(); i.hasNext(); i.next()) {
            count++;
        }
        return (count == 0);
    }

    /**
     * 全ての子のテキスト情報とエレメントノードを反復子で返却します。
     * 
     * @return 全ての子のテキスト情報とエレメントノード
     */
    public Iterator childIterator() {
        return this.children.iterator();
    }

    /**
     * 全ての子のテキスト情報とエレメントノードをリストで返却します。
     * 
     * @return 全ての子のテキスト情報とエレメントノード
     */
    public List getChildren() {
        return this.children;
    }

    /**
     * 全ての子のテキスト情報を反復子で返却します。
     * 
     * @return 全ての子のテキスト情報
     */
    public Iterator textIterator() {
        final Iterator i = childIterator();
        return new SelectionNodeIterator(i) {
            boolean select(Node n) {
                return (n.isType(Node.TYPE_TEXT));
            }
        };
    }

    /**
     * 全ての子のテキスト情報をリストで返却します。
     * 
     * @return 全ての子のテキスト情報
     */
    public List getTextChildren() {
        return ListUtils.addAll(new LinkedList(), textIterator());
    }

    /**
     * 全ての子のエレメントノードを反復子で返却します。
     * 
     * @return 全ての子のエレメントノード
     */
    public Iterator elementIterator() {
        final Iterator i = childIterator();
        return new SelectionNodeIterator(i) {
            boolean select(Node n) {
                return (!n.isType(Node.TYPE_TEXT));
            }
        };
    }

    /**
     * 全ての子のエレメントノードをリストで返却します。
     * 
     * @return 全ての子のエレメントノード
     */
    public List getElementChildren() {
        return ListUtils.addAll(new LinkedList(), elementIterator());
    }

    /**
     * 指定されたノード名のエレメントノードを反復子で返却します。
     * 
     * @param nodeName
     *            対象のノード名
     * @return 全ての子のエレメントノード
     */
    public Iterator elementIterator(final String nodeName) {
        final Iterator i = childIterator();
        final String nm = nodeName;
        return new SelectionNodeIterator(i) {
            boolean select(Node n) {
                return (!n.isType(Node.TYPE_TEXT) && nm.equals(n.getContext().getNodeName()));
            }
        };
    }

    /**
     * 指定されたノード名のエレメントノードをリストで返却します。
     * 
     * @param nodeName
     *            対象のノード名
     * @return 全ての子のエレメントノード
     */
    public List getElementChildren(String nodeName) {
        return ListUtils.addAll(new LinkedList(), elementIterator(nodeName));
    }

    /**
     * <code>Node.SCOPE_OWNER</code>以上の可視性のある子のノードを反復子で返却します。
     * 
     * @return <code>Node.SCOPE_OWNER</code>以上の可視性のある子のノード
     */
    public Iterator ownerIterator() {
        final Iterator i = childIterator();
        return new SelectionNodeIterator(i) {
            boolean select(Node n) {
                return (Node.SCOPE_OWNER >= n.getScope());
            }
        };
    }

    /**
     * <code>Node.SCOPE_OWNER</code>以上の可視性のある子のノードをリストで返却します。
     * 
     * @return <code>Node.SCOPE_OWNER</code>以上の可視性のある子のノード
     */
    public List getOwnerChildren() {
        return ListUtils.addAll(new LinkedList(), ownerIterator());
    }

    /**
     * <code>Node.isType(Node.TYPE_VALUE)</code>が<code>true</code>である子のノードを反復子で返却します。
     * 
     * @return 値ノードである子のノード
     */
    public Iterator valueIterator() {
        final Iterator i = childIterator();
        return new SelectionNodeIterator(i) {
            boolean select(Node n) {
                return (n.isType(Node.TYPE_VALUE));
            }
        };
    }

    /**
     * <code>Node.isType(Node.TYPE_VALUE)</code>が<code>true</code>である子のノードをリストで返却します。
     * 
     * @return 値ノードである子のノード
     */
    public List getValueChildren() {
        return ListUtils.addAll(new LinkedList(), valueIterator());
    }

    /**
     * <code>Node.isType(Node.TYPE_VALUE)</code>が<code>true</code>である子のエレメントノードを反復子で返却します。
     * 
     * @return 値ノードである子のエレメントノード
     */
    public Iterator valueElementIterator() {
        final Iterator i = childIterator();
        return new SelectionNodeIterator(i) {
            boolean select(Node n) {
                return (n.isType(Node.TYPE_VALUE) && !n.isType(Node.TYPE_TEXT));
            }
        };
    }

    /**
     * <code>Node.isType(Node.TYPE_VALUE)</code>が<code>true</code>である子のエレメントノードをリストで返却します。
     * 
     * @return 値ノードである子のエレメントノード
     */
    public List getValueElementChildren() {
        return ListUtils.addAll(new LinkedList(), valueElementIterator());
    }

    /*
     * Attributes
     */

    /**
     * ノードの拡張属性を全て返却します。
     * 
     * @return 全てのノードの拡張属性
     */
    public Parameters getXAttributes() {
        return this.nodeRule.getXAttributes();
    }

    /**
     * 指定された名前を持つ拡張属性の値を返却します。 <br>
     * 指定された名前が存在しない場合は<code>null</code>が返されます。
     * 
     * @param name
     *            属性名
     * @return 指定された名前の拡張属性値
     */
    public String getXAttribute(String name) {
        return (String) getXAttributes().getValue(name);
    }

    /**
     * 指定された名前を持つ拡張属性の値を返却します。 <br>
     * 指定された名前が存在しない場合は<code>defaultValue</code>が返されます。
     * 
     * @param name
     *            属性名
     * @param defaultValue
     *            指定の属性が存在しない場合に返却される値
     * @return 指定された名前の拡張属性値又は<code>defaultValue</code>
     */
    public String getXAttribute(String name, String defaultValue) {
        return (String) getXAttributes().getValue(name, defaultValue);
    }

    /**
     * 属性の構成ルールのパラメータリストを返却します。
     * 
     * @return 属性の構成ルールのパラメータリスト
     */
    public Parameters getNodeAttributesRules() {
        return this.nodeRule.getNodeAttributesRules();
    }

    /**
     * 指定された名前を持つ属性の構成ルールを返却します。 <br>
     * 指定された名前が存在しない場合は<code>null</code>が返されます。
     * 
     * @param name
     *            属性名
     * @return 指定された名前の属性の構成ルール
     */
    public NodeAttributesRule getNodeAttributesRule(String name) {
        return (NodeAttributesRule) getNodeAttributesRules().getValue(name);
    }

    /**
     * ノードの属性を全て返却します。
     * 
     * @return 全てのノードの属性
     */
    public TagAttributes getTagAttributes() {
        return this.attributes;
    }

    /**
     * 指定された名前を持つ属性の値を返却します。 <br>
     * 指定された名前が存在しない場合は<code>null</code>が返されます。
     * 
     * @param name
     *            属性名
     * @return 指定された名前の属性値
     */
    public String getTagAttribute(String name) {
        return getTagAttributes().getValue(name);
    }

    /**
     * 指定された名前を持つ属性の値を返却します。 <br>
     * 指定された名前が存在しない場合は<code>defaultValue</code>が返されます。
     * 
     * @param name
     *            属性名
     * @param defaultValue
     *            指定の属性が存在しない場合に返却される値
     * @return 指定された名前の属性値又は<code>defaultValue</code>
     */
    public String getTagAttribute(String name, String defaultValue) {
        String value = getTagAttribute(name);
        return (value != null) ? value : defaultValue;
    }

    /*
     * package
     */

    /**
     * 子の要素を追加します。
     * 
     * @param child
     *            子の要素
     */
    void addChild(Node child) {
        children.add(child);
    }

    private static abstract class SelectionNodeIterator implements Iterator {
        private final Object NULL = new Object();

        private final Object NOSUCH = new Object();

        private final Iterator i;

        private Object next = NULL;

        private SelectionNodeIterator(Iterator iter) {
            this.i = iter;
        }

        public void remove() {
            i.remove();
        }

        public boolean hasNext() {
            while (i.hasNext()) {
                Node n = (Node) i.next();
                if (select(n)) {
                    this.next = n;
                    return true;
                }
            }
            this.next = NOSUCH;
            return false;
        }

        public Object next() {
            if (this.next == NULL) {
                throw new IllegalStateException();
            }
            if (this.next == NOSUCH) {
                throw new NoSuchElementException();
            }
            return this.next;
        }

        abstract boolean select(Node n);
    }

}
