/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.ginkgo.helpers;

import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.Map;

import org.xml.sax.DTDHandler;
import org.xml.sax.EntityResolver;
import org.xml.sax.ErrorHandler;

import shohaku.core.collections.params.Parameters;
import shohaku.ginkgo.Document;
import shohaku.ginkgo.Ginkgo;
import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.NodeCompositeRule;
import shohaku.ginkgo.NodeRule;
import shohaku.ginkgo.SAXDocumentParser;

/**
 * <code>shohaku.ginkgo.NodeCompositeRule</code> のデフォルト実装を提供します。
 */
public class DefaultNodeCompositeRule implements NodeCompositeRule {

    /* <code>SAXParserFactory#validating</code>（nullは未設定）。 */
    private Boolean validating;

    /* <code>SAXParserFactory#namespaceAware</code>（nullは未設定）。 */
    private Boolean namespaceAware;

    /* <code>org.xml.sax.EntityResolver</code>（nullは未設定）。 */
    private EntityResolver entityResolver;

    /* <code>org.xml.sax.DTDHandler</code>（nullは未設定）。 */
    private DTDHandler dtdHandler;

    /* <code>org.xml.sax.ErrorHandler</code>（nullは未設定）。 */
    private ErrorHandler errorHandler;

    /* 構造化処理の拡張機能を格納する。 */
    private Map features = new IdentityHashMap();

    /* 名前空間URI単位でパラメータリストを格納する。 */
    private Map nodeRuleParameters = new HashMap();

    /*
     * implement NodeCompositeRule
     */

    /**
     * ノードの構成情報を返却します。
     * 
     * @param namespaceURI
     *            解析対象の名前空間URI
     * @param nodeURI
     *            解析対象のXML階層URI
     * @param tagName
     *            タグ名
     * @return ノード構成情報
     * @see shohaku.ginkgo.NodeCompositeRule#findNodeRule(java.lang.String, java.lang.String, java.lang.String)
     */
    public NodeRule findNodeRule(String namespaceURI, String nodeURI, String tagName) {
        String nsuri = (namespaceURI != null) ? namespaceURI : "";
        Parameters param = (Parameters) nodeRuleParameters.get(nsuri);
        for (Iterator i = param.iterator(); i.hasNext();) {
            Parameters.Entry e = (Parameters.Entry) i.next();
            String suffix = e.getName();
            if (suffix.equals("*") || nodeURI.endsWith(suffix)) {
                return (NodeRule) e.getValue();
            }
        }
        return null;
    }

    /**
     * 名前空間URI単位でノードの構成ルールをXMLURIのパターンで保管するパラメータリストを格納します。
     * 
     * @param namespaceURI
     *            ノードの名前空間URI
     * @param params
     *            ノードの構成ルールをXMLURIのパターンで保管するパラメータリスト
     */
    public void addNodeRuleParameters(String namespaceURI, Parameters params) {
        String nsuri = (namespaceURI != null) ? namespaceURI : "";
        nodeRuleParameters.put(nsuri, params);
    }

    /**
     * 指定された識別子が示す拡張機能を返却します。 <br>
     * 指定された識別子に対応する拡張機能が存在しない場合<code>null</code>を返却します。
     * 
     * @param id
     *            機能を示すクラス
     * @return 拡張機能
     * @see shohaku.ginkgo.NodeCompositeRule#getFeature(java.lang.Class)
     */
    public Object getFeature(Class id) {
        synchronized (features) {
            return features.get(id);
        }
    }

    /**
     * 指定された識別子が示す構造化処理の拡張機能を追加します。
     * 
     * @param id
     *            機能を示すクラス
     * @param feature
     *            拡張機能
     */
    public void addFeature(Class id, Object feature) {
        if (!id.isInstance(feature)) {
            throw new GinkgoException("not assignable class, id:" + id + " , feature:" + feature);
        }
        synchronized (features) {
            features.put(id, feature);
        }
    }

    /**
     * 解析処理の直前に呼び出されます、SAXへの設定情報が指定されていれば設定します。
     * 
     * @param ginkgo
     *            解析処理を実行する<code>Ginkgo</code>
     * @param document
     *            解析対象のドキュメント
     * @see shohaku.ginkgo.NodeCompositeRule#parse(shohaku.ginkgo.Ginkgo, shohaku.ginkgo.Document)
     */
    public void parse(Ginkgo ginkgo, Document document) {

        SAXDocumentParser _saxDocumentParser = ginkgo.getSAXDocumentParser();
        Boolean _validating = getValidating();
        if (_validating != null) {
            _saxDocumentParser.setValidating(_validating);
        }
        Boolean _namespaceAware = getNamespaceAware();
        if (_namespaceAware != null) {
            _saxDocumentParser.setNamespaceAware(_namespaceAware);
        }
        EntityResolver _entityResolver = getEntityResolver();
        if (_entityResolver != null) {
            _saxDocumentParser.setEntityResolver(_entityResolver);
        }
        DTDHandler _dtdHandler = getDTDHandler();
        if (_dtdHandler != null) {
            _saxDocumentParser.setDTDHandler(_dtdHandler);
        }
        ErrorHandler _errorHandler = getErrorHandler();
        if (_errorHandler != null) {
            _saxDocumentParser.setErrorHandler(_errorHandler);
        }

    }

    /*
     * 実装固有の機能とプロパティ
     */

    /**
     * 解析に使用する<code>org.xml.sax.EntityResolver</code>を格納します。
     * 
     * @param entityResolver
     *            設定値、設定を行わない場合<code>null</code>
     */
    public void setEntityResolver(EntityResolver entityResolver) {
        this.entityResolver = entityResolver;
    }

    /**
     * 解析に使用する<code>org.xml.sax.EntityResolver</code>を返却します。
     * 
     * @return 設定値、設定を行わない場合<code>null</code>
     */
    public EntityResolver getEntityResolver() {
        return this.entityResolver;
    }

    /**
     * 解析に使用する<code>org.xml.sax.DTDHandler</code>を格納します。
     * 
     * @param dtdHandler
     *            設定値、設定を行わない場合<code>null</code>
     */
    public void setDTDHandler(DTDHandler dtdHandler) {
        this.dtdHandler = dtdHandler;
    }

    /**
     * 解析に使用する<code>org.xml.sax.DTDHandler</code>を返却します。
     * 
     * @return 設定値、設定を行わない場合<code>null</code>
     */
    public DTDHandler getDTDHandler() {
        return this.dtdHandler;
    }

    /**
     * 解析に使用する<code>org.xml.sax.ErrorHandler</code> を格納します。
     * 
     * @param errorHandler
     *            設定値、設定を行わない場合<code>null</code>
     */
    public void setErrorHandler(ErrorHandler errorHandler) {
        this.errorHandler = errorHandler;
    }

    /**
     * 解析に使用する<code>org.xml.sax.ErrorHandler</code> を返却します。
     * 
     * @return 設定値、設定を行わない場合<code>null</code>
     */
    public ErrorHandler getErrorHandler() {
        return this.errorHandler;
    }

    /**
     * 解析に使用する<code>javax.xml.parsers.SAXParserFactory#validating</code>へ設定する値を格納します。
     * 
     * @param validating
     *            設定値、設定を行わない場合<code>null</code>
     */
    public void setValidating(Boolean validating) {
        this.validating = validating;
    }

    /**
     * 解析に使用する<code>javax.xml.parsers.SAXParserFactory#validating</code>へ設定する値を返却します。
     * 
     * @return 設定値、設定を行わない場合<code>null</code>
     */
    public Boolean getValidating() {
        return this.validating;
    }

    /**
     * 解析に使用する<code>javax.xml.parsers.SAXParserFactory#namespaceAware</code>へ設定する値を格納します。
     * 
     * @param namespaceAware
     *            設定値、設定を行わない場合<code>null</code>
     */
    public void setNamespaceAware(Boolean namespaceAware) {
        this.namespaceAware = namespaceAware;
    }

    /**
     * 解析に使用する<code>javax.xml.parsers.SAXParserFactory#namespaceAware</code>へ設定する値を返却します。
     * 
     * @return 設定値、設定を行わない場合<code>null</code>
     */
    public Boolean getNamespaceAware() {
        return this.namespaceAware;
    }

}