/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ginkgo;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import shohaku.core.lang.Eval;

/**
 * XMLドキュメントの構成情報を表現する機能を提供します。
 */
public class Document {

    /* ドキュメントのコンテキスト情報。 */
    private final DocumentContext context;

    /* 親ドキュメント。 */
    private final Document parent;

    /* ドキュメントの公開識別子。 */
    private String id;

    /* 識別子を持つタグの値を保管します。 */
    private final HashMap nodeValueMap;

    /* 子の構成処理に必要な情報のみを保持したドキュメントであるかを示す。 */
    private final boolean isPreserve;

    /* 初期化します（XML解析時に生成されます）。 */
    Document(Ginkgo ginkgo, NodeCompositeRule rule, Document parent) {
        this.context = new DocumentContext(ginkgo, rule);
        this.parent = parent;
        this.nodeValueMap = new HashMap();
        this.isPreserve = false;

    }

    /* Preserve Document を初期化します。 */
    private Document(Document srcDocument) {
        this.context = null;
        this.parent = srcDocument.parent;
        this.id = srcDocument.id;
        this.nodeValueMap = srcDocument.nodeValueMap;
        this.isPreserve = true;
    }

    /**
     * ドキュメントのコンテキスト情報を返却します。
     * 
     * @return ドキュメントのコンテキスト情報
     */
    public DocumentContext getContext() {
        return context;
    }

    /**
     * 親ドキュメントを返却します。
     * 
     * @return 親ドキュメント
     */
    public Document getParent() {
        return parent;
    }

    /**
     * ドキュメントの公開識別子を返却します。
     * 
     * @return ドキュメントの公開識別子
     */
    public String getId() {
        return id;
    }

    /* ドキュメントの公開識別子を格納します、解析システムから呼ばれます。 */
    void setId(String documentId) {
        this.id = documentId;
    }

    /*
     * ID
     */

    /* タグの値の読み取り専用のマップビュー */
    private Map tagValueMapView;

    /**
     * タグの値の読み取り専用のマップビューを返却します。
     * 
     * @return タグの値の読み取り専用のマップビュー
     */
    public Map getTagValueMapView() {
        Map m = tagValueMapView;
        return (m != null) ? m : (tagValueMapView = Collections.unmodifiableMap(this.nodeValueMap));
    }

    /**
     * 引数のタグの値の識別子が存在する場合は true を返却します。
     * 
     * @param tagId
     *            ID
     * @return 引数のタグの値の識別子が存在する場合は true
     */
    public boolean containsId(String tagId) {
        // return this.nodeValueMap.containsKey(id);
        if (tagId == null) {
            throw new NullPointerException();
        }
        if (Eval.isContains(tagId, ':')) {
            int off = tagId.indexOf(':');
            String documentId = tagId.substring(0, off);
            String nodeId = tagId.substring(++off);
            return containsId(documentId, nodeId);
        } else {
            return this.nodeValueMap.containsKey(tagId);
        }
    }

    /* ドキュメント識別子と識別子が示すタグの値が存在する場合は true を返却します。 */
    private boolean containsId(String documentId, String nodeId) {
        if (documentId.equals(getId())) {
            return this.nodeValueMap.containsKey(nodeId);
        }
        if (null != getParent()) {
            return getParent().containsId(documentId, nodeId);
        }
        return false;
    }

    /**
     * IDが示すタグの値を返す。
     * 
     * @param tagId
     *            ID
     * @return IDが示すタグの値
     * @throws NullPointerException
     *             id が null の場合
     */
    public Object getTagValueById(String tagId) {
        if (tagId == null) {
            throw new NullPointerException();
        }
        if (Eval.isContains(tagId, ':')) {
            int off = tagId.indexOf(':');
            String documentId = tagId.substring(0, off);
            String nodeId = tagId.substring(++off);
            return getTagValueById(documentId, nodeId);
        } else {
            return this.nodeValueMap.get(tagId);
        }
    }

    /* ドキュメントIDとIDが示すタグの値を返す。 */
    private Object getTagValueById(String documentId, String nodeId) {
        if (documentId.equals(getId())) {
            return this.nodeValueMap.get(nodeId);
        }
        if (null != getParent()) {
            return getParent().getTagValueById(documentId, nodeId);
        }
        return null;
    }

    /* IDを持つタグの値を追加します。 */
    void addReferenceNodeValue(TagNode node) {
        if (node == null) {
            throw new NullPointerException();
        }
        if (node instanceof ValueNode) {
            String nodeId = node.getId();
            if (!Eval.isBlank(nodeId)) {
                if (this.nodeValueMap.containsKey(nodeId)) {
                    throw new GinkgoException("That ID has already been registered. id:" + nodeId + ".");
                }
                this.nodeValueMap.put(nodeId, ((ValueNode) node).getNodeValue());
            }
        }

    }

    /**
     * 子の構成処理に必要な情報のみを保持したドキュメントの場合は true を返却します。
     * 
     * @return 子の構成処理に必要な情報のみを保持したドキュメントの場合は true
     */
    public boolean isPreserve() {
        return this.isPreserve;
    }

    /**
     * 子のドキュメントの構成処理に必要な情報のみを保持したドキュメントを返却します。<br>
     * 
     * @return 子の構成処理に必要な情報のみを保持したドキュメント
     */
    public Document getPreserveDocument() {
        return new Document(this);
    }

    /*
     * Ginkgo.parse
     */

    // /**
    // * タグの解析処理プロセスの開始の通知を受けます。
    // *
    // * @param node
    // * タグ
    // */
    // void startElement(TagNode node) {
    // // no op
    // }
    /**
     * タグの解析処理プロセスの終了の通知を受けます。
     * 
     * @param node
     *            タグ
     */
    void endElement(TagNode node) {
        addReferenceNodeValue(node);
    }

}
