/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ginkgo;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import shohaku.core.collections.MapUtils;
import shohaku.core.helpers.HClass;

/**
 * XMLドキュメントの解析コンテキスト情報を提供します。
 */
public class DocumentContext {

    /* Ginkgo。 */
    private final Ginkgo ginkgo;

    /* ドキュメント解析の構成ルール。 */
    private final NodeCompositeRule rule;

    /* ルートノードを保管します。 */
    private TagNode root;

    /* 属性情報を保管します。 */
    private final HashMap attributes = new HashMap();

    /* インポートクラスを保管します */
    private final HashMap imports = new HashMap();

    /* 初期化します（Ginkgo解析実行時に生成されます）。 */
    DocumentContext(Ginkgo ginkgo, NodeCompositeRule rule) {
        this.ginkgo = ginkgo;
        this.rule = rule;

    }

    /**
     * ドキュメント解析の構成ルールを返却します。
     * 
     * @return ドキュメント解析の構成ルール
     */
    public NodeCompositeRule getNodeCompositeRule() {
        return rule;
    }

    /**
     * Ginkgo を返却します。
     * 
     * @return Ginkgo
     */
    public Ginkgo getGinkgo() {
        return ginkgo;
    }

    /**
     * 解析処理に使用する ClassLoader を返却します.
     * 
     * @return 解析処理に使用する ClassLoader
     */
    public ClassLoader getClassLoader() {
        return getGinkgo().getClassLoader();
    }

    /**
     * ドキュメントのルートノードを返却します。
     * 
     * @return ドキュメントのルートノード
     */
    public TagNode getRoot() {
        return root;
    }

    /* ドキュメントのルートノードを格納します。 */
    void setRoot(TagNode root) {
        this.root = root;
    }

    /*
     * Import Class
     */

    /* 属性の読み取り専用のマップビュー */
    private Map importMapView;

    /**
     * インポートされたクラスの読み取り専用のマップビューを返却します。
     * 
     * @return インポートされたクラスの読み取り専用のマップビュー
     */
    public Map getImportMapView() {
        Map m = importMapView;
        return (m != null) ? m : (importMapView = Collections.unmodifiableMap(this.imports));
    }

    /**
     * クラスをインポートします。<br>
     * ドキュメント内でパッケージ名を除いたクラス名で参照することが出来る様になります。
     * 
     * @param c
     *            インポートするクラス
     * @return インポートが上書きされた場合は true
     */
    public boolean addImport(Class c) {
        return (null != imports.put(HClass.getShortClassName(c), c));
    }

    /**
     * クラス名に対応するクラスの参照を返却します。
     * 
     * @param name
     *            クラス名
     * @return クラス名に対応するクラスの参照、該当クラスが無い場合は null
     */
    public Class forClass(String name) {
        return (Class) imports.get(name);
    }

    /*
     * Attribute
     */

    /* 属性の読み取り専用のマップビュー */
    private Map attributeMapView;

    /**
     * 属性の読み取り専用のマップビューを返却します。
     * 
     * @return 属性の読み取り専用のマップビュー
     */
    public Map getAttributeMapView() {
        Map m = attributeMapView;
        return (m != null) ? m : (attributeMapView = Collections.unmodifiableMap(this.attributes));
    }

    /**
     * 引数の名前を持つ属性が存在する場合は true を返却します。
     * 
     * @param name
     *            属性名
     * @return 引数の名前を持つ属性が存在する場合は true
     */
    public boolean containsName(String name) {
        return this.attributes.containsKey(name);
    }

    /**
     * 引数の名前を持つ属性値を返却します。 <br>
     * 指定された属性が存在しない場合は null が返されます。
     * 
     * @param name
     *            属性名
     * @return 属性値
     */
    public Object getAttribute(String name) {
        if (name == null) {
            throw new NullPointerException();
        }
        return this.attributes.get(name);
    }

    /**
     * 引数の名前を持つ属性が存在する場合その属性値を存在しない場合は第二引数を返却します。
     * 
     * @param name
     *            属性名
     * @param defaultValue
     *            属性が存在しない場合に返却される値
     * @return 属性値
     */
    public Object getAttribute(String name, Object defaultValue) {
        Object value = getAttribute(name);
        if (value == null) {
            return defaultValue;
        }
        return value;
    }

    /**
     * 接頭辞の一致する属性を Map に格納して返却します。
     * 
     * @param prefix
     *            属性名の接頭辞
     * @return 接頭辞の一致する属性を格納する Map
     */
    public Map getAttributePrefix(String prefix) {
        return MapUtils.putPrefixKey(this.attributes, prefix, new HashMap());
    }

    /**
     * 属性を登録します、既存の属性が存在する場合は既存の属性値を返却し上書きします。 <br>
     * 既存の属性が存在しない場合は null が返されます。
     * 
     * @param name
     *            属性名
     * @param value
     *            属性値
     * @return 既存の属性値、存在しない場合は null
     */
    public Object setAttribute(String name, Object value) {
        if (name == null) {
            throw new NullPointerException();
        }
        return this.attributes.put(name, value);
    }

    /**
     * マップの全ての要素を属性に追加します、既存の属性が存在する場合は既存の属性値を上書きします。
     * 
     * @param map
     *            属性に追加するマップ
     */
    public void setAttributeAll(Map map) {
        if (map == null) {
            throw new NullPointerException();
        }
        this.attributes.putAll(map);
    }

    /**
     * 属性を削除し削除された属性値を返却します。 属性が存在しない場合は null が返されます。
     * 
     * @param name
     *            属性名
     * @return 属性値
     */
    public Object removeAttribute(String name) {
        if (name == null) {
            throw new NullPointerException();
        }
        return this.attributes.remove(name);
    }

    /**
     * 接頭辞の一致する属性を削除し削除された属性を Map に格納して返却します。
     * 
     * @param prefix
     *            属性名の接頭辞
     * @return 接頭辞の一致する属性の Map
     */
    public Map removeAttributePrefix(String prefix) {
        return MapUtils.removePrefixKey(this.attributes, prefix, new HashMap());
    }

    /**
     * 属性を全て削除します。
     */
    public void removeAttributeAll() {
        this.attributes.clear();
    }

}
