/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ginkgo;

import java.io.IOException;
import java.io.InputStream;
import java.util.Locale;

import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import shohaku.core.lang.feature.FeatureFactory;
import shohaku.core.lang.feature.LogFeature;
import shohaku.core.resource.IOResourceLoader;

/**
 * <p>
 * XMLドキュメントをタグオブジェクトにマッピングし、複雑なオブジェクトグラフを構築する機能を提供します。<br>
 * XMLドキュメントからのオブジェクトグラフ生成に着目したライブラリです。
 * </p>
 * <p>
 * このライブラリはXMLで定義された情報を shohaku.ginkgo.NodeCompositeRule の指定するルールに基づき shohaku.ginkgo.TagNode のツリー構造に変換します。 <br>
 * 値型の shohaku.ginkgo.TagNode は自身と内包する要素に基づきオブジェクトを生成します。 <br>
 * 複数のタグを再起的に合成する事により、単純なデータ型の集合から複雑なグラフ構造を生成します。 <br>
 * また一度定義されたタグは一定の依存関係を含むものの相対的に独立するため複数の文脈で再利用することができます。<br>
 * <br>
 * 具体的には、入れ子のタグやタグの参照を、そのタグ名等の個別の属性ではなく、その生成する値型でマッピングする事により実現しています。<br>
 * このアプローチの利点は、タグの階層関係や参照関係から、タグ名やタグの実装等の個別の依存関係を無くすことが出来る点です。<br>
 * その為、タグ名の変更、タグ実装の交換、値の生成プロセスの変更等を既存実装への影響なく行えます。<br>
 * <br>
 * また属性やテキストは自動的に変換されて、タグのプロパティに格納されるため、宣言のみで多くの機能を実現できます。<br>
 * 属性やテキストに対して式言語が使用できるため、XMLの記載量を少なくする事が出来ます。<br>
 * デフォルトではOGDL式を使用します（JXEL, OGNL等の他の式言語への拡張も容易です）。
 * </p>
 * <p>
 * デフォルトのタグ実装として、基本型、コレクション型、ユーティリティ型、コンテキスト制御、オブジェクト生成を行う汎用型のタグ等が提供されています。 <br>
 * また再帰的な構造を容易に生成するために反復処理や条件分岐を実現する制御インターフェースが提供されています。 <br>
 * <br>
 * XMLタグとタグオブジェクトのマッピングは、ルールXMLファイルに定義されるため、タグの関係、制約等を容易にカスタマイズ出来ます。<br>
 * またタグ属性に対するデフォルト値の指定や別名の指定を定義する事が出来ます。<br>
 * この機能の応用からデータ構造の再定義や、汎用ノードを基に特殊なノードを定義する事も可能です。
 * </p>
 * <p>
 * Ginkgo は汎用的なデータ構造を基に複雑なグラフ構造を生成したい場合に有用に設計されています。 <br>
 * 単なる XML-Object マッピングとして使用する事も可能ですが、その場合は他のXMLマッピングツールの使用を検討してください。
 * </p>
 * <p>
 * 使用例として shohaku.shoin パッケージを参照する事が出来ます。
 * </p>
 */
public class Ginkgo {

    /* shohaku core logging. */
    private static final LogFeature defaultLogger = FeatureFactory.getLog(Ginkgo.class);

    /* 解析処理に使用するSAXパーサー */
    private final SAXDocumentParser saxDocumentParser;

    /* 解析処理に使用するドキュメント構成ルールを保管 */
    private DocumentCompositeRule documentCompositeRule;

    /* 解析処理に使用するノード構成ルールを保管 */
    private NodeCompositeRule nodeCompositeRule;

    /* 解析処理に使用する親ドキュメントを保管 */
    private Document parentDocument;

    /* 解析処理に使用するログ */
    private LogFeature logger;

    /* 解析処理に使用するIOリソース生成機能 */
    private IOResourceLoader ioResourceLoader;

    /* 解析処理に使用するクラスローダ */
    private ClassLoader classLoader;

    /* 解析処理内でエラーが発生した場合に例外を発生させるかを指定する（デフォルトはfalse） */
    private boolean errThrowable;

    /* 解析処理に使用するロケール */
    private Locale locale;

    /* 直前の解析処理のドキュメントを保管 */
    private Document document;

    /**
     * デフォルトの構成ルールで初期化します。
     */
    public Ginkgo() {
        this(getDefaultDocumentCompositeRule());
    }

    /**
     * ドキュメントの構成ルールを指定して初期化します。
     * 
     * @param docRule
     *            ドキュメントの構成ルール
     */
    public Ginkgo(DocumentCompositeRule docRule) {
        if (null == docRule) {
            throw new NullPointerException("DocumentCompositeRule is null.");
        }
        // initialize
        this.documentCompositeRule = docRule;
        this.logger = null;
        this.saxDocumentParser = new SAXDocumentParser(this);
        this.classLoader = null;
        this.nodeCompositeRule = null;
        this.errThrowable = false;
    }

    /*
     * Parser
     */

    /**
     * 解析処理を実行します。
     * 
     * @param inStream
     *            XMLデータ入力ストリーム
     * @throws GinkgoException
     *             構成する情報に何らかの誤りが在る場合発生します。
     */
    public void parse(InputStream inStream) {
        parse(new InputSource(inStream));
    }

    /**
     * 解析処理を実行します。
     * 
     * @param input
     *            XMLデータソース
     * @throws GinkgoException
     *             構成する情報に何らかの誤りが在る場合発生します。
     */
    public void parse(InputSource input) {
        if (null == input) {
            throw new NullPointerException("input source is null.");
        }
        if (null == nodeCompositeRule) {
            throw new NullPointerException("this.nodeCompositeRule is null.");
        }
        Document parent = getParentDocument();
        if (parent != null && !parent.isPreserve()) {
            parent = parent.getPreserveDocument();
        }

        // prepare
        this.document = new Document(this, nodeCompositeRule, parent);
        this.nodeCompositeRule.prepare(this);

        try {

            getSAXDocumentParser().parse(input);

        } catch (SAXException e) {
            throw new GinkgoException("ginkgo parse err.", e);
        } catch (IOException e) {
            throw new GinkgoException("ginkgo parse err.", e);
        }

    }

    /*
     * getter
     */

    /**
     * 解析処理に使用するSAXパーサーを返却します。
     * 
     * @return 解析処理に使用するSAXパーサー
     */
    public SAXDocumentParser getSAXDocumentParser() {
        return saxDocumentParser;
    }

    /**
     * 解析中または直前に解析したドキュメントを返却します。
     * 
     * @return 解析中または直前に解析したドキュメント
     */
    public Document getDocument() {
        return document;
    }

    /*
     * property
     */

    /**
     * 解析処理プロセス内で使用するログを返却します。<br>
     * 登録されていない場合は、デフォルトログが返却されます。
     * 
     * @return ログ
     */
    public LogFeature getLogger() {
        return (logger != null) ? logger : Ginkgo.defaultLogger;
    }

    /**
     * 解析処理プロセス内で使用するログを格納します。
     * 
     * @param log
     *            ログ
     */
    public void setLogger(LogFeature log) {
        this.logger = log;
    }

    /**
     * 解析処理に使用するIOリソース生成機能を返却します。<br>
     * 登録されていない場合は、null が返却されます。
     * 
     * @return IOリソース生成機能
     */
    public IOResourceLoader getIOResourceLoader() {
        return ioResourceLoader;
    }

    /**
     * 解析処理に使用するIOリソース生成機能を格納します。
     * 
     * @param loader
     *            IOリソース生成機能
     */
    public void setIOResourceLoader(IOResourceLoader loader) {
        this.ioResourceLoader = loader;
    }

    /**
     * 解析処理に使用するクラスローダを返却します.<br>
     * 登録されていない場合は、このクラスのクラスローダが返却されます。
     * 
     * @return クラスローダ
     */
    public ClassLoader getClassLoader() {
        return (classLoader != null) ? classLoader : this.getClass().getClassLoader();
    }

    /**
     * 解析処理に使用するクラスローダを設定します.
     * 
     * @param loader
     *            クラスローダ
     */
    public void setClassLoader(ClassLoader loader) {
        this.classLoader = loader;
    }

    /**
     * 解析処理内でエラーが発生した場合に例外を発生させるかを示すフラグを返却します（デフォルトはfalse）。
     * 
     * @return true の場合は解析処理内でエラーが発生した場合に例外を発生させる
     */
    public boolean isErrThrowable() {
        return errThrowable;
    }

    /**
     * 解析処理内でエラーが発生した場合に例外を発生させるかを示すフラグを格納します。<br>
     * デフォルトでは、この値は false に設定されます。
     * 
     * @param errThrowable
     *            true の場合は解析処理内でエラーが発生した場合に例外を発生させる
     */
    public void setErrThrowable(boolean errThrowable) {
        this.errThrowable = errThrowable;
    }

    /**
     * 解析処理に使用するロケールを返却します。<br>
     * 登録されていない場合はデフォルトロケールが返却されます。
     * 
     * @return 解析処理に使用するロケール
     */
    public Locale getLocale() {
        return (locale != null) ? locale : Locale.getDefault();
    }

    /**
     * 解析処理に使用するロケールを格納します。
     * 
     * @param locale
     *            解析処理に使用するロケール
     */
    public void setLocale(Locale locale) {
        this.locale = locale;
    }

    /**
     * ドキュメントの構成ルールを返却します。
     * 
     * @return ドキュメントの構成ルール
     */
    public DocumentCompositeRule getDocumentCompositeRule() {
        return documentCompositeRule;
    }

    /**
     * ドキュメントの構成ルールを格納します。<br>
     * 引数が null の場合は NullPointerException が発生します。
     * 
     * @param docRule
     *            ドキュメントの構成ルール
     */
    public void setDocumentCompositeRule(DocumentCompositeRule docRule) {
        if (null == docRule) {
            throw new NullPointerException("DocumentCompositeRule is null.");
        }
        this.documentCompositeRule = docRule;
    }

    /**
     * 解析処理に使用するノード構成ルールを返却します。
     * 
     * @return 解析処理に使用するノード構成ルール
     */
    public NodeCompositeRule getNodeCompositeRule() {
        return nodeCompositeRule;
    }

    /**
     * 解析処理に使用するノード構成ルールを格納します。<br>
     * 引数が null の場合は NullPointerException が発生します。
     * 
     * @param nodeCompositeRule
     *            解析処理に使用するノード構成ルール
     */
    public void setNodeCompositeRule(NodeCompositeRule nodeCompositeRule) {
        if (null == nodeCompositeRule) {
            throw new NullPointerException("NodeCompositeRule is null.");
        }
        this.nodeCompositeRule = nodeCompositeRule;
    }

    /**
     * 解析処理に使用する親ドキュメントを返却します。
     * 
     * @return 親ドキュメント
     */
    public Document getParentDocument() {
        return parentDocument;
    }

    /**
     * 解析処理に使用する親ドキュメントを格納します。
     * 
     * @param parentDocument
     *            親ドキュメント
     */
    public void setParentDocument(Document parentDocument) {
        this.parentDocument = parentDocument;
    }

    /*
     * static
     */

    /* 複数のノードとドキュメントを合成するためのルールのデフォルト実装を生成して返却します。 */
    static DocumentCompositeRule getDefaultDocumentCompositeRule() {
        return new DocumentCompositeRuleImpl();
    }

}
