/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ginkgo;

import java.util.Iterator;
import java.util.LinkedList;

import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;

/**
 * 構造化に使用するコンテンツハンドラを実装します。
 */
class SAXContentHandler implements ContentHandler {

    /* スキップカウンタの無効状態を示す値。 */
    private static final int NO_SKIP_BODY = -1;

    /* キャッシュカウンタの無効状態を示す値。 */
    private static final int INIT_CACHE_TAG = -1;

    /* Ginkgo */
    private final Ginkgo ginkgo;

    /* TagPropertyTransfer */
    private final TagPropertyTransfer tagAttsTransfer;

    /* 解析中のノード構成ルールを一時保管します。 */
    private final LinkedList stackTagCreateRule;

    /* ノード構成ルールを保管します。 */
    private final LinkedList cacheTagCreateRule;

    /* 解析中の名前階層URIを保管します。 */
    private String currentUri;

    /* コンテンツの解析を無視する場合のスキップカウンタ。 */
    private int skipBodyCount;

    /* コンテンツのキャッシュを行うかを示すキャッシュカウンタ。 */
    private int cacheTagCount;

    /* 初期化します（Ginkgo解析実行時に生成されます）。 */
    SAXContentHandler(Ginkgo ginkgo) {
        this.ginkgo = ginkgo;
        this.tagAttsTransfer = ginkgo.getNodeCompositeRule().getTagPropertyTransfer();
        this.stackTagCreateRule = new LinkedList();
        this.cacheTagCreateRule = new LinkedList();
        this.currentUri = "";
        this.skipBodyCount = NO_SKIP_BODY;
        this.cacheTagCount = INIT_CACHE_TAG;
    }

    /*
     * SAX ContentHandler
     */

    public void setDocumentLocator(Locator locator) {
        // no op
    }

    public void startDocument() throws SAXException {
        // no op
    }

    public void startPrefixMapping(String prefix, String uri) throws SAXException {
        // no op
    }

    public void endPrefixMapping(String prefix) throws SAXException {
        // no op
    }

    /**
     * 要素の開始通知を受け取ります。
     * 
     * @see org.xml.sax.ContentHandler#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
     */
    public void startElement(String namespaceURI, String localName, String qName, Attributes atts) throws SAXException {

        if (this.skipBodyCount != NO_SKIP_BODY) {
            this.skipBodyCount++;
            return;
        }
        try {

            String tagName = getTagName(localName, qName);

            startElement(namespaceURI, tagName, localName, qName, new TagAttributes(atts));

        } catch (Exception e) {
            handleException("Ginkgo.startElement:", e);
        }
    }

    /**
     * 要素の開始通知を受け取ります。
     * 
     * @see org.xml.sax.ContentHandler#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
     */
    void startElement(String namespaceURI, String tagName, String localName, String qName, TagAttributes atts) throws SAXException {

        try {

            downCurrentURI(tagName);

            TagCreateRule targetCreateRule = createTagCreateRule(namespaceURI, tagName);

            TagCreateRule parentCreateRule = (!empty()) ? peek() : null;

            push(targetCreateRule);

            if (!isCacheTag()) {
                // substituor attributes
                tagAttsTransfer.substitut(this.ginkgo.getDocument(), atts);
            }

            targetCreateRule.begin(this.currentUri, namespaceURI, tagName, localName, qName, atts, parentCreateRule);

            TagNode tag = targetCreateRule.getTagNode();

            if (isCurrentRoot()) {
                Document doc = this.ginkgo.getDocument();
                doc.getContext().setRoot(tag);
                String documentId = this.ginkgo.getDocumentCompositeRule().getDocumentId(doc, tag);
                doc.setId(documentId);
            }

            if (!isCacheTag()) {
                // set attributes for tag property
                tagAttsTransfer.setAttributes(tag);
            }

            if (tag instanceof CacheBodyTag) {
                this.cacheTagCount++;
            }

            if (tag instanceof EvaluationTag) {
                startEvaluation(tag);
            }

            // this.ginkgo.getDocument().startElement(tag);

        } catch (Exception e) {
            handleException("Ginkgo.startElement:", e);
        }
    }

    /* 現在解析されているノードがルートノードの場合trueを返す。 */
    private boolean isCurrentRoot() {
        return size() == 1;
    }

    /**
     * 要素内の文字データの通知を受け取ります。
     * 
     * @see org.xml.sax.ContentHandler#characters(char[], int, int)
     */
    public void characters(char[] buffer, int start, int length) throws SAXException {

        if (this.skipBodyCount != NO_SKIP_BODY) {
            return;
        }

        try {

            if (!empty()) {

                TagCreateRule targetCreateRule = peek();
                char[] chars = new char[length];
                System.arraycopy(buffer, start, chars, 0, length);
                targetCreateRule.addChars(this.currentUri, chars);

            }

        } catch (Exception e) {
            handleException("Ginkgo.characters:", e);
        }
    }

    /**
     * 要素の終了通知を受け取ります。
     * 
     * @see org.xml.sax.ContentHandler#endElement(java.lang.String, java.lang.String, java.lang.String)
     */
    public void endElement(String namespaceURI, String localName, String qName) throws SAXException {

        if (this.skipBodyCount != NO_SKIP_BODY) {
            if (this.skipBodyCount == 0) {
                this.skipBodyCount = NO_SKIP_BODY;
            } else {
                this.skipBodyCount--;
                return;
            }
        }

        try {

            TagCreateRule targetCreateRule = pop();

            TagCreateRule parentCreateRule = (!empty()) ? peek() : null;

            TagNode tag = targetCreateRule.getTagNode();

            // テキストノードを生成、テキストとエレメントは相互排他
            if (tag.getTagContext().isElementEmpty()) {
                StringBuffer text = targetCreateRule.getCharBuffer();
                if (!isCacheTag()) {
                    tagAttsTransfer.substitut(this.ginkgo.getDocument(), text);
                }
                targetCreateRule.endBody(new TextNodeImpl(text.toString()));
            } else {
                targetCreateRule.endBody(new TextNodeImpl(null));
            }

            if (tag instanceof CacheBodyTag) {
                this.cacheTagCount--;
            }

            if (tag instanceof EvaluationTag) {
                endEvaluation(tag);
            }

            if (!isCacheTag()) {
                // set text for tag property
                if (!tag.getTagContext().isTextEmpty()) {
                    tagAttsTransfer.setText(tag);
                }
            }

            String name = getTagName(localName, qName);
            targetCreateRule.end(this.currentUri, namespaceURI, name, localName, qName);

            if (!isParentCacheTag() && parentCreateRule != null) {
                // add Element for tag property
                TagNode parent = parentCreateRule.getTagNode();
                tagAttsTransfer.addElement(parent, tag);
            }

            upCurrentURI();

            this.ginkgo.getDocument().endElement(tag);

        } catch (Exception e) {
            handleException("Ginkgo.endElement:", e);
        }
    }

    /**
     * 文書の終了通知を受け取ります。
     * 
     * @see org.xml.sax.ContentHandler#endDocument()
     */
    public void endDocument() throws SAXException {
        try {

            for (Iterator i = this.cacheTagCreateRule.iterator(); i.hasNext();) {
                TagCreateRule rule = (TagCreateRule) i.next();
                rule.finish();
            }

        } catch (Exception e) {
            handleException("Ginkgo.endDocument:", e);
        }

    }

    public void ignorableWhitespace(char ch[], int start, int length) throws SAXException {
        // no op
    }

    public void processingInstruction(String target, String data) throws SAXException {
        // no op
    }

    public void skippedEntity(String name) throws SAXException {
        // no op
    }

    /*
     * helper
     */

    private boolean isCacheTag() {
        return (this.cacheTagCount > INIT_CACHE_TAG);
    }

    private boolean isParentCacheTag() {
        return (!isCurrentRoot() && (this.cacheTagCount - 1) > INIT_CACHE_TAG);
    }

    /*
     * Stack TagCreateRule
     */

    private boolean empty() {
        return (size() == 0);
    }

    private int size() {
        return this.stackTagCreateRule.size();
    }

    private void push(TagCreateRule rule) {
        this.stackTagCreateRule.addLast(rule);
        this.cacheTagCreateRule.add(rule);
    }

    private TagCreateRule pop() {
        return (TagCreateRule) this.stackTagCreateRule.removeLast();
    }

    private TagCreateRule peek() {
        return (TagCreateRule) this.stackTagCreateRule.getLast();
    }

    /*
     * EvaluationNode
     */

    private void startEvaluation(TagNode tag) {
        int eval = ((EvaluationTag) tag).doInitBody();
        if (eval == EvaluationTag.SKIP_BODY) {
            this.skipBodyCount = 0;
        }
    }

    private void endEvaluation(TagNode tag) throws SAXException {
        EvaluationTag bodyEvalNode = ((EvaluationTag) tag);
        while (EvaluationTag.EVAL_BODY == bodyEvalNode.doEvalBody()) {
            // children or chars
            evaluationChilds(tag);
        }
    }

    private void evaluationContent(TagNode tag) throws SAXException {
        TagContext nc = tag.getTagContext();
        String namespaceURI = nc.getTagNamespaceURI();
        String tagName = nc.getTagName();
        String localName = nc.getTagLocalName();
        String qName = nc.getTagQName();
        TagAttributes atts = nc.getTagAttributes();

        // startElement
        startElement(namespaceURI, tagName, localName, qName, new TagAttributes(atts));
        // children or chars
        evaluationChilds(tag);
        // endElement
        endElement(namespaceURI, localName, qName);
    }

    private void evaluationChilds(TagNode parent) throws SAXException {
        // children or chars
        final TagContext context = parent.getTagContext();
        if (!context.isTextEmpty()) {
            String text = context.getTextNode().getText();
            characters(text.toCharArray(), 0, text.length());
        }
        if (!context.isElementEmpty()) {
            for (Iterator i = context.elementIterator(); i.hasNext();) {
                Node child = (Node) i.next();
                evaluationContent((TagNode) child);
            }
        }
    }

    /*
     * Creater
     */

    private TagCreateRule createTagCreateRule(String namespaceURI, String tagName) throws SAXException {

        if (this.cacheTagCount > INIT_CACHE_TAG) {

            TagRule tagRule = createCacheTagRule();
            return new TagCreateRule(ginkgo.getDocument(), tagRule);

        } else {

            TagRule tagRule = findTagRule(namespaceURI, tagName);
            if (tagRule == null) {
                throw new SAXException("TagRule couldn't be created. uri:" + this.currentUri + ", name:" + tagName);
            }
            return new TagCreateRule(ginkgo.getDocument(), tagRule);
        }
    }

    private TagRule createCacheTagRule() {
        TagRule tagRule = new TagRule();
        tagRule.setTagClass(CacheTag.class.getName());
        tagRule.setPattern(this.currentUri);
        return tagRule;
    }

    /*
     * Current URI
     */

    private void downCurrentURI(String name) {
        StringBuffer sb = new StringBuffer(this.currentUri);
        sb.append('/');
        sb.append(name);
        this.currentUri = sb.toString();
    }

    private void upCurrentURI() {
        String uri = this.currentUri;
        int slash = uri.lastIndexOf('/');
        if (slash >= 0) {
            this.currentUri = uri.substring(0, slash);
        } else {
            this.currentUri = "";
        }
    }

    /*
     * Node
     */

    private TagRule findTagRule(String namespaceURI, String tagName) {
        return this.ginkgo.getNodeCompositeRule().findTagRule(namespaceURI, this.currentUri, tagName);
    }

    /*
     * error handler
     */

    private void handleException(String msg, Exception e) throws SAXException {

        if (this.ginkgo.isErrThrowable()) {
            if (e instanceof SAXException) {
                throw (SAXException) e;
            } else {
                throw new SAXException(msg, e);
            }
        } else {
            error(msg, e);
        }

    }

    /*
     * Logger
     */

    private void error(Object message, Throwable t) {
        this.ginkgo.getLogger().error(message, t);
    }

    /*
     * static
     */

    private static String getTagName(String localName, String qName) {
        String name = localName;
        if ((name == null) || (name.length() < 1)) {
            name = qName;
        }
        return name;
    }

}