/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ginkgo;

import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import shohaku.core.collections.IteratorUtils;
import shohaku.core.helpers.HSeek;
import shohaku.core.lang.Eval;
import shohaku.core.lang.Predicate;

/**
 * タグのコンテキスト情報を提供します。
 */
public class TagContext {

    /* ドキュメント。 */
    private Document document;

    /* タグの設定情報。 */
    private TagRule tagRule;

    /* 対象のタグを保管します。 */
    private TagNode target;

    /* 親のタグを保管します。 */
    private TagNode parent;

    /* タグの階層URI. */
    private String tagURI;

    /* タグの名前空間。 */
    private String tagNamespaceURI;

    /* タグのローカル名。 */
    private String tagLocalName;

    /* タグ名。 */
    private String tagQName;

    /* タグ名。 */
    private String tagName;

    /* タグの全属性。 */
    private TagAttributes tagAttributes;

    /* 子のノードを保管します。 */
    private List elements;

    /* テキストノードを保管します。 */
    private TextNode textNode;

    /**
     * エレメントノードの情報を格納して初期化します。
     * 
     * @param document
     *            ドキュメント
     * @param tagRule
     *            ノードの設定情報
     * @param target
     *            対象のタグ
     * @param parent
     *            親ノード
     * @param uri
     *            階層URI
     * @param namespace
     *            名前空間
     * @param localName
     *            ローカル名
     * @param tagQName
     *            タグ名
     * @param tagName
     *            ノード名
     * @param attributes
     *            属性情報
     */
    TagContext(Document document, TagRule tagRule, TagNode target, TagNode parent, String uri, String namespace, String localName, String tagQName, String tagName, TagAttributes attributes) {
        this.document = document;
        this.tagRule = tagRule;
        this.target = target;
        this.parent = parent;
        this.tagURI = uri;
        this.tagNamespaceURI = namespace;
        this.tagLocalName = localName;
        this.tagQName = tagQName;
        this.tagName = tagName;
        this.tagAttributes = attributes;
        this.elements = null;
        this.textNode = null;
    }

    /**
     * ドキュメントを返却します。
     * 
     * @return ドキュメント
     */
    public Document getDocument() {
        return document;
    }

    /**
     * Ginkgo を返却します。
     * 
     * @return Ginkgo
     */
    public Ginkgo getGinkgo() {
        return getDocumentContext().getGinkgo();
    }

    /**
     * 解析処理に使用する構成ルールを返却します。
     * 
     * @return 解析処理に使用する構成ルール
     */
    public NodeCompositeRule getNodeCompositeRule() {
        return getDocumentContext().getNodeCompositeRule();
    }

    /**
     * ドキュメントのコンテキスト情報を返却します。
     * 
     * @return ドキュメントのコンテキスト情報
     */
    public DocumentContext getDocumentContext() {
        return getDocument().getContext();
    }

    /**
     * ドキュメント構成ルールを返却します。
     * 
     * @return ドキュメント構成ルール
     */
    public DocumentCompositeRule getDocumentCompositeRule() {
        return getGinkgo().getDocumentCompositeRule();
    }

    /**
     * 解析処理に使用する ClassLoader を返却します.
     * 
     * @return 解析処理に使用する ClassLoader
     */
    public ClassLoader getClassLoader() {
        return getGinkgo().getClassLoader();
    }

    /**
     * タグ名を返却します。
     * 
     * @return タグ名
     */
    public String getTagQName() {
        return tagQName;
    }

    /**
     * タグのローカル名を返却します。
     * 
     * @return タグのローカル名
     */
    public String getTagLocalName() {
        return tagLocalName;
    }

    /**
     * タグ名を返却する（プレフィックスを含まないタグ名）。
     * 
     * @return タグ名
     */
    public String getTagName() {
        return tagName;
    }

    /**
     * タグの名前空間を返却します。
     * 
     * @return タグの名前空間
     */
    public String getTagNamespaceURI() {
        return tagNamespaceURI;
    }

    /**
     * タグの階層URIを返却します。
     * 
     * @return タグの階層URI
     */
    public String getTagURI() {
        return tagURI;
    }

    /**
     * タグのスコープを返却します。
     * 
     * @return タグのスコープ
     */
    public int getScope() {
        int scope = ContainTag.SCOPE_PUBLIC;
        TagNode tag = target;
        // bubble up xml hierarchy
        while (tag != null) {
            if (tag instanceof ContainTag) {
                int parentScope = ((ContainTag) tag).getScope();
                scope = (parentScope > scope) ? parentScope : scope;// select narrow scope
            }
            tag = tag.getTagContext().getParent();
        }
        return scope;
    }

    /**
     * 親のタグを格納します。
     * 
     * @return 親のタグ
     */
    public TagNode getParent() {
        return this.parent;
    }

    /*
     * childs
     */

    /**
     * 子の要素が空の場合は true を返却します。
     * 
     * @return 子の要素が空の場合は true
     */
    public boolean isChildsEmpty() {
        return (isElementEmpty() && isTextEmpty());
    }

    /*
     * text
     */

    /**
     * 子の要素が空の場合は true を返却します。
     * 
     * @return 子の要素が空の場合は true
     */
    public boolean isTextEmpty() {
        return (textNode.getText() == null);
    }

    /**
     * テキストノードを返却します。
     * 
     * @return テキスト情報
     */
    public TextNode getTextNode() {
        return textNode;
    }

    /*
     * elements
     */

    private List getElements() {
        return (elements != null) ? elements : (elements = new LinkedList());
    }

    /**
     * 子の要素が空の場合は true を返却します。
     * 
     * @return 子の要素が空の場合は true
     */
    public boolean isElementEmpty() {
        return (elements == null || elements.isEmpty());
    }

    /**
     * 全ての子のエレメントタグを反復子で返却します。
     * 
     * @return 全ての子のエレメントタグ
     */
    public Iterator elementIterator() {
        return (isElementEmpty()) ? IteratorUtils.emptyIterator() : getElements().iterator();
    }

    /**
     * 指定されたタグ名のエレメントタグを反復子で返却します。
     * 
     * @param predicateTagName
     *            対象のタグ名
     * @return 全ての子のエレメントタグ
     */
    public Iterator elementIterator(String predicateTagName) {
        final String nm = predicateTagName;
        return IteratorUtils.predicateIterator(elementIterator(), new Predicate() {
            public boolean evaluate(Object o) {
                return (nm.equals(((TagNode) o).getTagContext().getTagName()));
            }
        });
    }

    /**
     * ContainTag.SCOPE_OWNER 以上の可視性のある子のタグを反復子で返却します。
     * 
     * @return ContainTag.SCOPE_OWNER 以上の可視性のある子のタグ
     */
    public Iterator ownerIterator() {
        return IteratorUtils.predicateIterator(elementIterator(), new Predicate() {
            public boolean evaluate(Object o) {
                return (ContainTag.SCOPE_OWNER >= ((TagNode) o).getTagContext().getScope());
            }
        });
    }

    /**
     * Node.isType(Node.TYPE_VALUE) が true である子のエレメントタグを反復子で返却します。
     * 
     * @return 値ノードである子のエレメントタグ
     */
    public Iterator valueIterator() {
        return IteratorUtils.predicateIterator(elementIterator(), new Predicate() {
            public boolean evaluate(Object o) {
                return (o instanceof ValueNode);
            }
        });
    }

    /**
     * 指定されたタグ名の Node.isType(Node.TYPE_VALUE) が true である子のエレメントタグを反復子で返却します。
     * 
     * @param predicateTagName
     *            対象のタグ名
     * @return 値ノードである子のエレメントタグ
     */
    public Iterator valueIterator(String predicateTagName) {
        final String nm = predicateTagName;
        return IteratorUtils.predicateIterator(elementIterator(), new Predicate() {
            public boolean evaluate(Object o) {
                return (o instanceof ValueNode && nm.equals(((TagNode) o).getTagContext().getTagName()));
            }
        });
    }

    /*
     * Attributes
     */

    private Map getAttributeMap() {

        final Map m = new HashMap(4);
        final TagAttributes tagAtts = getTagAttributes();
        final TagAttributesRuleSet ruleSet = getTagAttributesRuleSet();

        // 構成ルールの値を格納する
        for (Iterator i = ruleSet.iterator(); i.hasNext();) {
            TagAttributesRule rule = (TagAttributesRule) i.next();
            m.put(rule.getName(), rule.getDefaultValue());
        }

        // 実際のタグ名で、タグの値を格納か上書きする
        for (int i = 0; i < tagAtts.getLength(); i++) {

            String name = HSeek.notEmpty(tagAtts.getLocalName(i), tagAtts.getQName(i));
            TagAttributesRule rule = ruleSet.find(name);
            String value = null;
            if (rule != null) {
                if (!Eval.isBlank(rule.getAlias())) {
                    name = rule.getAlias();
                }
                value = tagAtts.getValue(name);
                if (value == null) {
                    value = rule.getDefaultValue();
                }
            } else {
                value = tagAtts.getValue(name);
            }

            m.put(name, value);

        }

        return Collections.unmodifiableMap(m);

    }

    private Map attributeMapView = null;

    /**
     * 属性のマップビューを返却します。
     * 
     * @return 属性のマップビュー
     */
    public Map getAttributeMapView() {
        Map m = attributeMapView;
        return (m != null) ? m : (attributeMapView = getAttributeMap());
    }

    /**
     * 引数の名前を持つタグの属性を検索し値を返却します。 <br>
     * 属性が存在しない場合 null を返します。
     * 
     * @param name
     *            属性名
     * @return 属性値
     */
    public String getAttribute(String name) {
        return (String) getAttributeMapView().get(name);
    }

    /**
     * 引数の名前を持つタグの属性を検索し値を返却します。 <br>
     * 属性が存在しない場合 defaultValue を返します。
     * 
     * @param name
     *            属性名
     * @param defaultValue
     *            属性が存在しない場合に返却される値
     * @return 属性値
     */
    public String getAttribute(String name, String defaultValue) {
        String value = getAttribute(name);
        return (value != null) ? value : defaultValue;
    }

    /**
     * タグの属性を全て返却します。
     * 
     * @return 全てのタグの属性
     */
    public TagAttributes getTagAttributes() {
        return this.tagAttributes;
    }

    /**
     * 指定された名前を持つ属性の値を返却します。 <br>
     * 指定された名前が存在しない場合は null が返されます。
     * 
     * @param name
     *            属性名
     * @return 指定された名前の属性値
     */
    public String getTagAttribute(String name) {
        return getTagAttributes().getValue(name);
    }

    /**
     * 指定された名前を持つ属性の値を返却します。 <br>
     * 指定された名前が存在しない場合は defaultValue が返されます。
     * 
     * @param name
     *            属性名
     * @param defaultValue
     *            指定の属性が存在しない場合に返却される値
     * @return 指定された名前の属性値または defaultValue
     */
    public String getTagAttribute(String name, String defaultValue) {
        String value = getTagAttribute(name);
        return (value != null) ? value : defaultValue;
    }

    /**
     * 属性の構成ルールのパラメータリストを返却します。
     * 
     * @return 属性の構成ルールのパラメータリスト
     */
    public TagAttributesRuleSet getTagAttributesRuleSet() {
        return this.tagRule.getTagAttributesRuleSet();
    }

    /**
     * 指定された名前を持つ属性の構成ルールを返却します。 <br>
     * 指定された名前が存在しない場合は null が返されます。
     * 
     * @param name
     *            属性名
     * @return 指定された名前の属性の構成ルール
     */
    public TagAttributesRule getTagAttributesRule(String name) {
        return getTagAttributesRuleSet().find(name);
    }

    /*
     * package
     */

    /**
     * 子の要素を追加します。
     * 
     * @param element
     *            子の要素
     */
    void addElement(TagNode element) {
        getElements().add(element);
    }

    /**
     * テキストノードを格納します。
     * 
     * @param textNode
     *            テキストノード
     */
    void setTextNode(TextNode textNode) {
        this.textNode = textNode;
    }

}
