/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ginkgo;

import shohaku.core.lang.ObjectCreationException;
import shohaku.core.lang.feature.FeatureFactory;

/**
 * ノードを生成する機能を提供します。
 */
class TagCreateRule {

    /* Ginkgo。 */
    private final Ginkgo ginkgo;

    /* ドキュメント。 */
    private final Document document;

    /* 値の設定情報。 */
    private final TagRule tagRule;

    /* ノード。 */
    private TagNode tagNode;

    /* ノード属性情報。 */
    private TagContext tagContext;

    /* 本体のテキストを一時保管します。 */
    private StringBuffer charBuffer = new StringBuffer();

    /* 初期化します（SAXContentHandlerから生成されます）。 */
    TagCreateRule(Document document, TagRule tagRule) {
        this.ginkgo = document.getContext().getGinkgo();
        this.document = document;
        this.tagRule = tagRule;
    }

    /* ノードの読み取り開始時に呼び出されます。 */
    void begin(String uri, String namespace, String tagName, String localName, String qName, TagAttributes attributes, TagCreateRule parentRule) throws Exception {

        if (ginkgo.getLogger().isDebugEnabled()) {
            ginkgo.getLogger().debug("TagCreateRule#begin():uri:" + uri + ", namespace:" + namespace + ", tagName:" + tagName + ", localName:" + localName + ", qName:" + qName + ", attributes:" + attributes);
        }

        tagNode = createTagNode();

        boolean isNotHierarchical = (tagNode instanceof NotHierarchyTag);
        if (!isNotHierarchical && parentRule != null) {
            tagContext = new TagContext(document, tagRule, tagNode, parentRule.getTagNode(), uri, namespace, localName, qName, tagName, attributes);
            parentRule.addChild(uri, tagNode);
        } else {
            tagContext = new TagContext(document, tagRule, tagNode, null, uri, namespace, localName, qName, tagName, attributes);
        }
        tagNode.initialize(tagContext);
        tagNode.begin();
    }

    /* 個々のノードのテキスト解析時に呼び出されます。 */
    void addChars(String uri, char[] chars) throws Exception {
        if (ginkgo.getLogger().isDebugEnabled()) {
            ginkgo.getLogger().debug("TagCreateRule#chars():uri:" + uri + ":" + "chars:" + String.valueOf(chars));
        }
        charBuffer.append(chars);
    }

    /* 子ノードの生成直後に呼び出されます。 */
    void addChild(String uri, TagNode child) throws Exception {
        if (ginkgo.getLogger().isDebugEnabled()) {
            ginkgo.getLogger().debug("TagCreateRule#addChild():uri:" + uri + ", child:" + child);
        }
        tagContext.addElement(child);
        tagNode.child(child);
    }

    /* ノード本体の解析終了時に呼び出されます。 */
    void endBody(TextNode textNode) throws Exception {
        tagContext.setTextNode(textNode);
        tagNode.text(textNode);
    }

    /* ノードの解析終了時に呼び出されます。 */
    void end(String uri, String namespace, String tagName, String localName, String qName) throws Exception {
        if (ginkgo.getLogger().isDebugEnabled()) {
            ginkgo.getLogger().debug("TagCreateRule#end():uri:" + uri + ", namespace:" + namespace + ", tagName:" + tagName + "localName:" + localName + ", qName:" + qName);
        }
        tagNode.end();
    }

    /* ドキュメント解析終了時に呼び出されます。 */
    void finish() throws Exception {

        if (ginkgo.getLogger().isDebugEnabled()) {
            ginkgo.getLogger().debug("TagCreateRule#finish():" + tagNode.getClass());
        }

        tagNode.finish();

        // clear
        this.tagNode = null;
        this.tagContext = null;
        this.charBuffer = null;
    }

    /* ノードを生成して返します、生成に失敗した場合はObjectCreationExceptionを発生させます。 */
    private TagNode createTagNode() throws ObjectCreationException {
        String className = tagRule.getTagClass();
        ClassLoader classLoader = ginkgo.getClassLoader();
        TagNode ret = (TagNode) FeatureFactory.getLoader().getInstance(className, classLoader);
        if (ret == null) {
            throw new ObjectCreationException("className:" + tagRule.getTagClass());
        }
        return ret;
    }

    StringBuffer getCharBuffer() {
        return charBuffer;
    }

    TagNode getTagNode() {
        return tagNode;
    }

    TagRule getTagRule() {
        return tagRule;
    }
}