/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ginkgo;

/**
 * XMLタグを表現するインターフェースを提供します。 <br>
 * <br>
 * 銀杏APIでは、タグの属性とテキストおよび要素の値はシステムの自動変換機能からプロパティに格納されます。<br>
 * <br>
 * 属性に対しては、属性名と同じプロパティ名の Setter メソッドとバインドされます。<br>
 * Setter メソッドが定義されない場合は、自動転送は実行されません。<br>
 * <br>
 * テキストは固定名 textTransferValue プロパティの Setter メソッドとバインドされます。<br>
 * Setter メソッドが定義されない場合は、自動転送は実行されません。<br>
 * <br>
 * 要素の値は単一の引数を取る、要素のタグ名に接尾辞 "add" を不可したメソッドとバインドされます。<br>
 * またデフォルトメソッドとして、固定名で単一の引数を取る、addElementTransferValue メソッドとバインドされます。<br>
 * 一般的な方針としては、要素のタグ名ではなく、型のみでバインドする場合は、デフォルトメソッドのみを定義します。<br>
 * 接尾辞を持つメソッドやデフォルトメソッドはオーバーロード可能で要素の値から推測的に格納可能なメソッドを検出して設定します。<br>
 * 接尾辞を持つメソッドか addElementTransferValue メソッドが定義されない場合は、自動転送は実行されません。<br>
 * <br>
 * 型の変換処理と変換ルールは TagPropertyTransfer の実装に依存します。<br>
 * デフォルト実装では、各メソッドの引数型に対応付けられた変換処理を実行します。<br>
 */
public interface TagNode extends Node {

    /**
     * タグの識別子を返却します。
     * 
     * @return タグの識別子
     */
    String getId();

    /**
     * タグの識別名を返却します。<br>
     * 識別名はドキュメント内、同一階層内、階層下内で重複があり得ます。<br>
     * 識別名は定義上で、０個以上のデータを持つ名前コレクションとして認識されます。
     * 
     * @return タグの識別名
     */
    String getName();

    /**
     * タグのコンテキスト情報を返却します。
     * 
     * @return タグのコンテキスト情報
     */
    TagContext getTagContext();

    /**
     * ドキュメント解析処理を開始する直前に初期化の通知を受ける。
     * 
     * @param tagContext
     *            ノードのコンテキスト情報
     * @throws GinkgoException
     *             構成情報例外
     */
    void initialize(TagContext tagContext);

    /**
     * タグ解析開始の通知を受ける。 開始タグの解析終了直後に対応します。
     * 
     * @throws GinkgoException
     *             構成情報例外
     */
    void begin();

    /**
     * 子のタグの追加の通知を受ける。 子タグの解析終了直後に対応します。
     * 
     * @param tagNode
     *            子のタグ
     * @throws GinkgoException
     *             構成情報例外
     */
    void child(TagNode tagNode);

    /**
     * テキストノードの格納の通知を受ける。 終了タグの解析終了直後に対応します。
     * 
     * @param textNode
     *            テキストノード
     * @throws GinkgoException
     *             構成情報例外
     */
    void text(TextNode textNode);

    /**
     * タグ解析終了通知を受ける。 終了タグの解析終了直後に対応します。
     * 
     * @throws GinkgoException
     *             構成情報例外
     */
    void end();

    /**
     * ドキュメント解析処理プロセスを完了の通知を受ける。
     * 
     * @throws GinkgoException
     *             構成情報例外
     */
    void finish();
}
