/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ginkgo.rule;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.xml.sax.DTDHandler;
import org.xml.sax.EntityResolver;
import org.xml.sax.ErrorHandler;

import shohaku.core.lang.Eval;
import shohaku.ginkgo.Ginkgo;
import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.NodeCompositeRule;
import shohaku.ginkgo.SAXDocumentParser;
import shohaku.ginkgo.TagPropertyTransfer;
import shohaku.ginkgo.TagRule;
import shohaku.ginkgo.TagRuleSet;

/**
 * NodeCompositeRule のデフォルト実装を提供します。
 */
public class DefaultNodeCompositeRule implements NodeCompositeRule {

    /* SAXParserFactory#validating （nullは未設定）。 */
    private Boolean validating;

    /* SAXParserFactory#namespaceAware （nullは未設定）。 */
    private Boolean namespaceAware;

    /* org.xml.sax.EntityResolver （nullは未設定）。 */
    private EntityResolver entityResolver;

    /* org.xml.sax.DTDHandler （nullは未設定）。 */
    private DTDHandler dtdHandler;

    /* org.xml.sax.ErrorHandler （nullは未設定）。 */
    private ErrorHandler errorHandler;

    /* 構造化処理の拡張機能を格納する。 */
    private Map features = new HashMap();

    /* 名前空間URI単位でパラメータリストを格納する。 */
    private Map tagRuleSetNSMap = new HashMap();

    /* タグ属性をタグのプロパティに転送する機能 */
    private TagPropertyTransfer tagPropertyTransfer;

    /*
     * implement
     */

    public TagRule findTagRule(String namespaceURI, String nodeURI, String tagName) {
        String nsuri = (!Eval.isBlank(namespaceURI)) ? namespaceURI : "";
        TagRuleSet tagRuleSet = (TagRuleSet) tagRuleSetNSMap.get(nsuri);
        if (tagRuleSet != null) {
            for (Iterator i = tagRuleSet.iterator(); i.hasNext();) {
                TagRule tagRule = (TagRule) i.next();
                String pattern = tagRule.getPattern();
                if (pattern.equals("*") || nodeURI.endsWith(pattern)) {
                    return tagRule;
                }
            }
        }
        return null;
    }

    public void addTagRule(String namespaceURI, TagRuleSet tagRuleSet) {
        String nsuri = (namespaceURI != null) ? namespaceURI : "";
        tagRuleSetNSMap.put(nsuri, tagRuleSet);
    }

    public Object getFeature(Class id) {
        synchronized (features) {
            return features.get(id);
        }
    }

    public void addFeature(Class id, Object feature) {
        if (!id.isInstance(feature)) {
            throw new GinkgoException("not assignable class, id:" + id + " , feature:" + feature);
        }
        synchronized (features) {
            features.put(id, feature);
        }
    }

    public TagPropertyTransfer getTagPropertyTransfer() {
        if (tagPropertyTransfer != null) {
            return tagPropertyTransfer;
        }
        return new DefaultTagPropertyTransfer();

    }

    public void setTagPropertyTransfer(TagPropertyTransfer tagAttributesTransfer) {
        this.tagPropertyTransfer = tagAttributesTransfer;
    }

    /**
     * 解析処理の直前に呼び出されます、SAXへの設定情報が指定されていれば設定します。
     * 
     * @param ginkgo
     *            解析処理を実行する Ginkgo
     * @see shohaku.ginkgo.NodeCompositeRule#prepare(shohaku.ginkgo.Ginkgo)
     */
    public void prepare(Ginkgo ginkgo) {

        SAXDocumentParser _saxDocumentParser = ginkgo.getSAXDocumentParser();
        Boolean _validating = getValidating();
        if (_validating != null) {
            _saxDocumentParser.setValidating(_validating);
        }
        Boolean _namespaceAware = getNamespaceAware();
        if (_namespaceAware != null) {
            _saxDocumentParser.setNamespaceAware(_namespaceAware);
        }
        EntityResolver _entityResolver = getEntityResolver();
        if (_entityResolver != null) {
            _saxDocumentParser.setEntityResolver(_entityResolver);
        }
        DTDHandler _dtdHandler = getDTDHandler();
        if (_dtdHandler != null) {
            _saxDocumentParser.setDTDHandler(_dtdHandler);
        }
        ErrorHandler _errorHandler = getErrorHandler();
        if (_errorHandler != null) {
            _saxDocumentParser.setErrorHandler(_errorHandler);
        }

    }

    /*
     * 実装固有の機能とプロパティ
     */

    /**
     * 解析に使用する org.xml.sax.EntityResolver を格納します。
     * 
     * @param entityResolver
     *            設定値、設定を行わない場合 null
     */
    public void setEntityResolver(EntityResolver entityResolver) {
        this.entityResolver = entityResolver;
    }

    /**
     * 解析に使用する org.xml.sax.EntityResolver を返却します。
     * 
     * @return 設定値、設定を行わない場合 null
     */
    public EntityResolver getEntityResolver() {
        return this.entityResolver;
    }

    /**
     * 解析に使用する org.xml.sax.DTDHandler を格納します。
     * 
     * @param dtdHandler
     *            設定値、設定を行わない場合 null
     */
    public void setDTDHandler(DTDHandler dtdHandler) {
        this.dtdHandler = dtdHandler;
    }

    /**
     * 解析に使用する org.xml.sax.DTDHandler を返却します。
     * 
     * @return 設定値、設定を行わない場合 null
     */
    public DTDHandler getDTDHandler() {
        return this.dtdHandler;
    }

    /**
     * 解析に使用する org.xml.sax.ErrorHandler を格納します。
     * 
     * @param errorHandler
     *            設定値、設定を行わない場合 null
     */
    public void setErrorHandler(ErrorHandler errorHandler) {
        this.errorHandler = errorHandler;
    }

    /**
     * 解析に使用する org.xml.sax.ErrorHandler を返却します。
     * 
     * @return 設定値、設定を行わない場合 null
     */
    public ErrorHandler getErrorHandler() {
        return this.errorHandler;
    }

    /**
     * 解析に使用する javax.xml.parsers.SAXParserFactory#validating へ設定する値を格納します。
     * 
     * @param validating
     *            設定値、設定を行わない場合 null
     */
    public void setValidating(Boolean validating) {
        this.validating = validating;
    }

    /**
     * 解析に使用する javax.xml.parsers.SAXParserFactory#validating へ設定する値を返却します。
     * 
     * @return 設定値、設定を行わない場合 null
     */
    public Boolean getValidating() {
        return this.validating;
    }

    /**
     * 解析に使用する javax.xml.parsers.SAXParserFactory#namespaceAware へ設定する値を格納します。
     * 
     * @param namespaceAware
     *            設定値、設定を行わない場合 null
     */
    public void setNamespaceAware(Boolean namespaceAware) {
        this.namespaceAware = namespaceAware;
    }

    /**
     * 解析に使用する javax.xml.parsers.SAXParserFactory#namespaceAware へ設定する値を返却します。
     * 
     * @return 設定値、設定を行わない場合 null
     */
    public Boolean getNamespaceAware() {
        return this.namespaceAware;
    }

}