/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ginkgo.rule;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;

import shohaku.core.beans.DefaultTypeTransformer;
import shohaku.core.functor.FTransformer;
import shohaku.core.functor.FunctorException;
import shohaku.core.helpers.HBeans;
import shohaku.core.helpers.HCnv;
import shohaku.core.helpers.HCut;
import shohaku.core.lang.Eval;
import shohaku.core.lang.IntrospectionBeansException;
import shohaku.ginkgo.Document;
import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.TagAttributes;
import shohaku.ginkgo.TagContext;
import shohaku.ginkgo.TagNode;
import shohaku.ginkgo.TagPropertyTransfer;
import shohaku.ginkgo.ValueNode;
import shohaku.ginkgo.type.ExpressionValue;

/**
 * TagPropertyTransfer のデフォルト実装を提供します。
 */
public class DefaultTagPropertyTransfer implements TagPropertyTransfer {

    /* 型変換機能 */
    final DefaultTypeTransformer typeTransformer;

    /**
     * デフォルトの型変換機能で初期化します。
     */
    public DefaultTagPropertyTransfer() {
        this(new DefaultTagPropertyTypeTransformer());
    }

    /**
     * 指定の型変換機能と変換補助機能で初期化します。
     * 
     * @param transformer
     *            型変換機能
     */
    public DefaultTagPropertyTransfer(DefaultTypeTransformer transformer) {
        this.typeTransformer = transformer;
    }

    /*
     * impl
     */

    public void substitut(Document document, TagAttributes atts) {
        for (int i = 0; i < atts.getLength(); i++) {
            String value = atts.getValue(i);
            if (!isExpressionValue(value)) {
                value = TypeTransformHelper.toEmbeddedExpressionString(document, value);
                atts.setValue(i, value);
            }
        }
    }

    public void substitut(Document document, StringBuffer text) {
        String value = text.toString();
        if (!isExpressionValue(value)) {
            value = TypeTransformHelper.toEmbeddedExpressionString(document, value);
            text.setLength(value.length());
            text.replace(0, value.length(), value);
        }
    }

    public void setAttributes(TagNode tag) {
        final TagContext context = tag.getTagContext();
        try {

            final Class tagType = tag.getClass();
            final Map pds = HBeans.getPropertyDescriptorMap(tagType);
            for (Iterator i = context.getAttributeMapView().entrySet().iterator(); i.hasNext();) {
                final Map.Entry e = (Map.Entry) i.next();

                final String propertyName = getPropertyName((String) e.getKey());
                final PropertyDescriptor pd = (PropertyDescriptor) pds.get(propertyName);
                if (pd != null && pd.getWriteMethod() != null) {
                    setProperty(tag, pd, (String) e.getValue());
                }
            }

        } catch (FunctorException e) {
            throw new GinkgoException("attributes transfer functor error. " + tag.getClass() + ", " + context.getTagAttributes(), e);
        } catch (IntrospectionBeansException e) {
            throw new GinkgoException("tag introspection error. " + tag.getClass() + ", " + context.getTagAttributes(), e);
        }
    }

    public void setText(TagNode tag) {

        final TagContext context = tag.getTagContext();
        final Class tagType = tag.getClass();
        try {
            final PropertyDescriptor pd = HBeans.getPropertyDescriptor(tagType, "textTransferValue");
            if (pd != null && pd.getWriteMethod() != null) {
                setProperty(tag, pd, context.getTextNode().getText());
            }
        } catch (FunctorException e) {
            throw new GinkgoException("transform functor error. " + tagType + ", " + context.getTagAttributes(), e);
        } catch (IntrospectionBeansException e) {
            throw new GinkgoException("tag introspection error. " + tagType + ", " + context.getTagAttributes(), e);
        }
    }

    public void addElement(TagNode tag, TagNode element) {
        if (!(element instanceof ValueNode)) {
            return;
        }

        final String propertyName = getPropertyName(tag.getTagContext().getTagName());
        final Class tagType = tag.getClass();
        final Object value = ((ValueNode) element).getNodeValue();
        try {

            // add property
            String methodName = HBeans.propertyAsMethodName("add", propertyName);
            // is has add property
            if (HBeans.containsMethod(tagType, methodName)) {
                // invoke add property
                invokeMethod(tagType, tag, methodName, value);
                return;
            }

            // add default property
            if (HBeans.containsAccessibleMethod(tagType, "addElementTransferValue")) {
                invokeAccessibleMethod(tagType, tag, "addElementTransferValue", value);
                return;
            }

        } catch (IntrospectionBeansException e) {
            throw new GinkgoException("tag introspection error. " + tag.getClass() + ", " + element, e);
        }
    }

    /**
     * クラスを識別子として変換ファンクタを追加します。<br>
     * 既にクラスが登録されている場合は変換ファンクタを上書きします。
     * 
     * @param clazz
     *            クラス
     * @param transformer
     *            変換ファンクタ
     * @return 上書きされた場合は既存の変換ファンクタ、以外は null
     */
    public FTransformer add(Class clazz, FTransformer transformer) {
        return typeTransformer.add(clazz, transformer);
    }

    /**
     * 登録されている変換ファンクタをクリアします。
     */
    public void clear() {
        typeTransformer.clear();
    }

    /**
     * クラスに対応する変換ファンクタを削除します。
     * 
     * @param clazz
     *            クラス
     * @return 実際に削除された場合は削除された変換ファンクタ、以外は null
     */
    public FTransformer remove(Class clazz) {
        return typeTransformer.remove(clazz);
    }

    private String getPropertyName(String propertyName) {

        // class -> clazz
        if (propertyName.equals("class")) {
            return "clazz";
        }

        // key-ref -> keyRef
        if (Eval.isContains(propertyName, '-')) {
            final StringBuffer sb = new StringBuffer();
            final String[] elems = propertyName.split("-");
            sb.append(elems[0]);
            for (int i = 1; i < elems.length; i++) {
                sb.append(HCnv.capitalize(elems[i]));
            }
            return sb.toString();
        }

        return propertyName;
    }

    /**
     * 指定のオブジェクトの引数を１つ取るメソッドに値を格納します。
     * 
     * @param clazz
     *            オブジェクトのクラス
     * @param o
     *            オブジェクト
     * @param methodName
     *            メソッド名
     * @param value
     *            格納する値
     * @throws IntrospectionBeansException
     *             メソッドアクセスに失敗した場合
     */
    private void invokeMethod(Class clazz, Object o, String methodName, Object value) throws IntrospectionBeansException {
        final Class paramType = (value != null) ? value.getClass() : Object.class;
        // get Method
        Method method = null;
        try {
            method = HBeans.getAssignmentMethod(clazz, methodName, new Class[] { paramType });
        } catch (IntrospectionBeansException e) {
            throw new GinkgoException(methodName + " illegal type. " + paramType, e);
        }
        if (method != null) {
            // set value
            HBeans.invokeMethod(o, method, value);
        }
    }

    /**
     * 特権で指定のオブジェクトの引数を１つ取るメソッドに値を格納します。
     * 
     * @param clazz
     *            オブジェクトのクラス
     * @param o
     *            オブジェクト
     * @param methodName
     *            メソッド名
     * @param value
     *            格納する値
     * @throws IntrospectionBeansException
     *             メソッドアクセスに失敗した場合
     */
    private void invokeAccessibleMethod(Class clazz, Object o, String methodName, Object value) throws IntrospectionBeansException {
        final Class paramType = (value != null) ? value.getClass() : Object.class;
        // get Method
        Method method = null;
        try {
            method = HBeans.getAssignmentAccessibleMethod(clazz, methodName, new Class[] { paramType });
        } catch (IntrospectionBeansException e) {
            throw new GinkgoException(methodName + " illegal type. " + paramType, e);
        }
        if (method != null) {
            // set value
            HBeans.invokeMethod(o, method, value);
        }
    }

    /**
     * 属性値が式言語を示すか検証します。
     * 
     * @param attributeValue
     *            属性値
     * @return 属性値が式言語を示す場合は true
     */
    private boolean isExpressionValue(String attributeValue) {
        return (Eval.isEnclose(attributeValue, "${", "}"));
    }

    // /**
    // * 属性値が埋め込み式言語を示すか検証します。
    // *
    // * @param attributeValue
    // * 属性値
    // * @return 属性値が埋め込み式言語を示す場合は true
    // */
    // private boolean isEmbeddedExpressionValue(String attributeValue) {
    // return (Eval.isEnclose(attributeValue, "#{", "}"));
    // }

    /**
     * 属性値を式言語として解析して其の結果を返却します。
     * 
     * @param tag
     *            タグ
     * @param attributeValue
     *            属性値
     * @return 属性値を式言語として解析した結果
     */
    private Object executeExpression(TagNode tag, String attributeValue) {
        final String value = (String) HCut.cut(attributeValue, 2, 1);
        ExpressionValue expValue = TypeTransformHelper.toExpressionValue(tag, value);
        return expValue.getResultValue();
    }

    // /**
    // * 属性値を埋め込み式言語として解析して其の結果を返却します。
    // *
    // * @param tag
    // * タグ
    // * @param attributeValue
    // * 属性値
    // * @return 属性値を埋め込み式言語として解析した結果
    // */
    // private String executeEmbeddedExpression(TagNode tag, String attributeValue) {
    // final String value = (String) HCut.cut(attributeValue, 2, 1);
    // return TypeTransformHelper.toEmbeddedExpressionString(tag, value);
    // }

    /**
     * 指定のプロパティを対応する型に変換して格納します。
     * 
     * @param tag
     *            タグ
     * @param pd
     *            プロパティの定義情報
     * @param value
     *            格納するプロパティ値
     * @throws IntrospectionBeansException
     *             プロパティアクセスに失敗した場合
     * @throws FunctorException
     *             予測外の問題から変換に失敗した場合
     */
    private void setProperty(TagNode tag, PropertyDescriptor pd, String value) throws IntrospectionBeansException {
        Object propertyValue = getPropertyValue(tag, pd.getPropertyType(), value);
        HBeans.setProperty(tag, pd, propertyValue);
    }

    /**
     * 値を対応するプロパティ型に変換して返却します。
     * 
     * @param tag
     *            タグ
     * @param clazz
     *            プロパティ型
     * @param value
     *            変換する値
     * @return プロパティ値
     * @throws FunctorException
     *             予測外の問題から変換に失敗した場合
     */
    private Object getPropertyValue(TagNode tag, Class clazz, String value) {
        Object propertyValue = value;
        if (isExpressionValue(value)) {
            propertyValue = executeExpression(tag, value);
            if (!clazz.isInstance(propertyValue)) {
                propertyValue = getTransformValue(tag, clazz, propertyValue);
            }
        } else {
            propertyValue = getTransformValue(tag, clazz, propertyValue);
        }
        return propertyValue;
    }

    /**
     * 型変換を実行する機能が定義されている場合は、変換処理を実行して返却します。
     * 
     * @param tag
     *            タグ
     * @param clazz
     *            プロパティ型
     * @param value
     *            変換する値
     * @return 型変換が定義されている場合は変換された値、以外は基の値
     * @throws FunctorException
     *             予測外の問題から変換に失敗した場合
     */
    private Object getTransformValue(TagNode tag, Class clazz, Object value) {
        Object propertyValue = value;
        FTransformer transformer = typeTransformer.find(clazz);
        if (transformer != null) {
            Map args = Collections.singletonMap(TagNode.class, tag);
            propertyValue = transformer.transform(value, args);
        }
        return propertyValue;
    }

}
