/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ginkgo.rule;

import java.io.File;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.URI;
import java.util.Map;
import java.util.Properties;
import java.util.regex.Pattern;

import shohaku.core.beans.DefaultTypeTransformer;
import shohaku.core.functor.FTransformer;
import shohaku.core.resource.IOResource;
import shohaku.ginkgo.TagNode;
import shohaku.ginkgo.type.EvaluationValue;
import shohaku.ginkgo.type.ExpressionValue;
import shohaku.ginkgo.type.IterateValue;
import shohaku.ginkgo.type.ReferenceValue;

/**
 * タグ属性の型変換を行う機能のデフォルト実装を提供します。
 */
public class DefaultTagPropertyTypeTransformer extends DefaultTypeTransformer {

    /**
     * タグ属性の変換ルールを格納して初期化します。
     */
    public DefaultTagPropertyTypeTransformer() {
        // no op
    }

    /**
     * デフォルトの変換ファンクタを登録します。
     * 
     * @see shohaku.core.beans.DefaultTypeTransformer#init()
     */
    protected void init() {
        super.init();
        // base
        add(Boolean.class, new BooleanTransformer());
        add(Character.class, new CharacterTransformer());
        add(String.class, new StringTransformer());
        add(Byte.class, new ByteTransformer());
        add(Short.class, new ShortTransformer());
        add(Integer.class, new IntegerTransformer());
        add(Long.class, new LongTransformer());
        add(Float.class, new FloatTransformer());
        add(Double.class, new DoubleTransformer());
        add(Class.class, new ClassTransformer());
        // ginkgo
        add(ExpressionValue.class, new ExpressionTransformer());
        add(ReferenceValue.class, new ReferenceTransformer());
        add(IterateValue.class, new IterateValueTransformer());
        add(EvaluationValue.class, new EvaluationValueTransformer());

        // math
        add(BigInteger.class, new BigIntegerTransformer());
        add(BigDecimal.class, new BigDecimalTransformer());
        // util
        add(java.util.Date.class, new DateTimeTransformer());
        add(Pattern.class, new RegexTransformer());
        add(Properties.class, new PropertiesTransformer());
        // io
        add(File.class, new FileTransformer());
        add(IOResource.class, new IOResourceTransformer());
        // net
        add(URI.class, new URITransformer());
        // reference value
        add(Object.class, new ReferenceValueTransformer());
    }

    class BooleanTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            return TypeTransformHelper.toBoolean((String) o);
        }
    }

    class CharacterTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            return TypeTransformHelper.toCharacter((String) o);
        }
    }

    class StringTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            return TypeTransformHelper.toString((String) o);
        }
    }

    class ByteTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            return TypeTransformHelper.toByte((String) o);
        }
    }

    class ShortTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            return TypeTransformHelper.toShort((String) o);
        }
    }

    class IntegerTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            return TypeTransformHelper.toInteger((String) o);
        }
    }

    class LongTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            return TypeTransformHelper.toLong((String) o);
        }
    }

    class FloatTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            return TypeTransformHelper.toFloat((String) o);
        }
    }

    class DoubleTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            return TypeTransformHelper.toDouble((String) o);
        }
    }

    class ClassTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            final TagNode tag = (TagNode) args.get(TagNode.class);
            return TypeTransformHelper.toClass(tag, (String) o);
        }
    }

    class BigIntegerTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            return TypeTransformHelper.toBigInteger((String) o);
        }
    }

    class BigDecimalTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            return TypeTransformHelper.toBigDecimal((String) o);
        }
    }

    class DateTimeTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            return TypeTransformHelper.toDateTime((String) o);
        }
    }

    class RegexTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            return TypeTransformHelper.toRegexPattern((String) o);
        }
    }

    class ExpressionTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            final TagNode tag = (TagNode) args.get(TagNode.class);
            return TypeTransformHelper.toExpressionValue(tag, (String) o);
        }
    }

    class PropertiesTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            final TagNode tag = (TagNode) args.get(TagNode.class);
            return TypeTransformHelper.toProperties(tag, (String) o);
        }
    }

    class FileTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            return TypeTransformHelper.toFile((String) o);
        }
    }

    class IOResourceTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            final TagNode tag = (TagNode) args.get(TagNode.class);
            return TypeTransformHelper.toIOResource(tag, (String) o);
        }
    }

    class URITransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            return TypeTransformHelper.toURI((String) o);
        }
    }

    class ReferenceTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            final TagNode tag = (TagNode) args.get(TagNode.class);
            return TypeTransformHelper.toReferenceValue(tag, (String) o);
        }
    }

    class ReferenceValueTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            final TagNode tag = (TagNode) args.get(TagNode.class);
            ReferenceValue refValue = TypeTransformHelper.toReferenceValue(tag, (String) o);
            return (refValue != null) ? refValue.getResultValue() : null;
        }
    }

    class IterateValueTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            return TypeTransformHelper.toIterateValue(o);
        }
    }

    class EvaluationValueTransformer implements FTransformer {
        public Object transform(Object o, Map args) {
            final TagNode tag = (TagNode) args.get(TagNode.class);
            return TypeTransformHelper.toEvaluationValue(tag, (String) o);
        }
    }

}