/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ginkgo.rule;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Date;
import java.util.Properties;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import shohaku.core.helpers.HClass;
import shohaku.core.helpers.HCnv;
import shohaku.core.helpers.HCoder;
import shohaku.core.helpers.HCut;
import shohaku.core.helpers.HLog;
import shohaku.core.lang.Closure;
import shohaku.core.lang.Eval;
import shohaku.core.lang.NoSuchResourceException;
import shohaku.core.lang.ObjectCreationException;
import shohaku.core.lang.ObjectCreationProxy;
import shohaku.core.lang.Predicate;
import shohaku.core.lang.ValueOf;
import shohaku.core.lang.feature.FeatureFactory;
import shohaku.core.resource.IOResource;
import shohaku.core.resource.IOResourceLoader;
import shohaku.ginkgo.Document;
import shohaku.ginkgo.DocumentContext;
import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.TagContext;
import shohaku.ginkgo.TagNode;
import shohaku.ginkgo.type.EvaluationValue;
import shohaku.ginkgo.type.ExpressionValue;
import shohaku.ginkgo.type.IterateValue;
import shohaku.ginkgo.type.ReferenceValue;
import shohaku.ogdl.Ogdl;
import shohaku.ogdl.OgdlContext;
import shohaku.ogdl.OgdlParseIndex;
import shohaku.ogdl.OgdlSyntaxException;

/**
 * タグの属性やテキストをプロパティ型に変換するヘルパーメソッド郡を集約して提供します。
 */
class TypeTransformHelper {

    /** 有効な Boolean の true を示す文字列表記。 */
    static final Pattern TRUE_PATTERN = Pattern.compile("(?i)true|yes|on");

    /** 有効な Boolean の文字列表記。 */
    static final Pattern BOOL_PATTERN = Pattern.compile("(?i)true|false|yes|no|on|off");

    /** 拡張の数値書式の正規表現パターン。 */
    static final Pattern NUMBER_FORMAT_PATTERN;
    static {
        StringBuffer sb = new StringBuffer();
        sb.append("(?:");
        sb.append("[-+]?Infinity");
        sb.append("|");
        sb.append("[-+]?NaN");
        sb.append("|");
        sb.append("[-+]?(?:[\\p{Digit}]|0|0X|0x|\\.)(?:(?:[\\p{XDigit},]|[eE][-+])*)");
        sb.append("(?:\\.(?:[\\p{Digit}]|[eE][-+])*)?");
        sb.append("[lLfFdD]?");
        sb.append(")");
        NUMBER_FORMAT_PATTERN = Pattern.compile(sb.toString());
    }

    /** 拡張の数値書式の拡張文字を削除するパターン */
    static final Pattern NUMBER_REMOVE_PATTERN = Pattern.compile("^\\+|,");

    /**
     * 文字列を Boolean に変換して返却します。
     * 
     * @param value
     *            解析する文字列
     * @return 変換結果
     * @throws GinkgoException
     *             変換に失敗した場合
     */
    static Boolean toBoolean(String value) {
        if (!isBoolean(value)) {
            throw new GinkgoException("illegal format of Boolean. " + value);
        }
        return Boolean.valueOf(TRUE_PATTERN.matcher(value).matches());
    }

    /**
     * 文字列を Character に変換して返却します。
     * 
     * @param value
     *            解析する文字列
     * @return 変換結果
     * @throws GinkgoException
     *             変換に失敗した場合
     */
    static Character toCharacter(String value) {
        if (Eval.isBlank(value)) {
            throw new GinkgoException("text isn't specified.");
        }

        String schar = value;
        // is 'char'
        if (Eval.isEnclose(schar, '\'', '\'')) {
            schar = (String) HCut.cut(schar, 1, 1);
        }
        if (schar.length() > 1) {
            // \nXXX
            try {
                schar = HCoder.decodeUnicodeEscapes(schar);
            } catch (IllegalArgumentException e) {
                throw new GinkgoException("illegal format of Character. " + schar);
            }
            if (schar.length() != 1) {
                throw new GinkgoException("not one Character. " + schar);
            }
        }
        return new Character(schar.charAt(0));
    }

    /**
     * 文字列を変換して返却します。
     * 
     * @param value
     *            解析する文字列
     * @return 変換結果
     * @throws GinkgoException
     *             変換に失敗した場合
     */
    static String toString(String value) {
        return value;
    }

    /**
     * 文字列を Byte に変換して返却します。
     * 
     * @param value
     *            解析する文字列
     * @return 変換結果
     * @throws GinkgoException
     *             変換に失敗した場合
     */
    static Byte toByte(String value) {
        try {
            String numValue = getNumberString(value);
            return Byte.decode(numValue);
        } catch (NumberFormatException e) {
            throw new GinkgoException("illegal format of Byte. " + value, e);
        }
    }

    /**
     * 文字列を Short に変換して返却します。
     * 
     * @param value
     *            解析する文字列
     * @return 変換結果
     * @throws GinkgoException
     *             変換に失敗した場合
     */
    static Short toShort(String value) {
        try {
            String numValue = getNumberString(value);
            return Short.decode(numValue);
        } catch (NumberFormatException e) {
            throw new GinkgoException("illegal format of Short. " + value, e);
        }
    }

    /**
     * 文字列を Integer に変換して返却します。
     * 
     * @param value
     *            解析する文字列
     * @return 変換結果
     * @throws GinkgoException
     *             変換に失敗した場合
     */
    static Integer toInteger(String value) {
        try {
            String numValue = getNumberString(value);
            return Integer.decode(numValue);
        } catch (NumberFormatException e) {
            throw new GinkgoException("illegal format of Integer. " + value, e);
        }
    }

    /**
     * 文字列を Long に変換して返却します。
     * 
     * @param value
     *            解析する文字列
     * @return 変換結果
     * @throws GinkgoException
     *             変換に失敗した場合
     */
    static Long toLong(String value) {
        try {
            String numValue = getNumberString(value);
            return Long.decode(numValue);
        } catch (NumberFormatException e) {
            throw new GinkgoException("illegal format of Long. " + value, e);
        }
    }

    /**
     * 文字列を Float に変換して返却します。
     * 
     * @param value
     *            解析する文字列
     * @return 変換結果
     * @throws GinkgoException
     *             変換に失敗した場合
     */
    static Float toFloat(String value) {
        try {
            String numValue = getNumberString(value);
            return Float.valueOf(numValue);
        } catch (NumberFormatException e) {
            throw new GinkgoException("illegal format of Float. " + value, e);
        }
    }

    /**
     * 文字列を Double に変換して返却します。
     * 
     * @param value
     *            解析する文字列
     * @return 変換結果
     * @throws GinkgoException
     *             変換に失敗した場合
     */
    static Double toDouble(String value) {
        try {
            String numValue = getNumberString(value);
            return Double.valueOf(numValue);
        } catch (NumberFormatException e) {
            throw new GinkgoException("illegal format of Double. " + value, e);
        }
    }

    /**
     * 文字列を java.math.BigInteger に変換して返却します。
     * 
     * @param value
     *            解析する文字列
     * @return 変換結果
     * @throws GinkgoException
     *             変換に失敗した場合
     */
    static BigInteger toBigInteger(String value) {
        try {
            String numValue = getNumberString(value);
            return new BigInteger(numValue);
        } catch (NumberFormatException e) {
            throw new GinkgoException("illegal format of java.math.BigInteger. " + value, e);
        }
    }

    /**
     * 文字列を java.math.BigDecimal に変換して返却します。
     * 
     * @param value
     *            解析する文字列
     * @return 変換結果
     * @throws GinkgoException
     *             変換に失敗した場合
     */
    static BigDecimal toBigDecimal(String value) {
        try {
            String numValue = getNumberString(value);
            return new BigDecimal(numValue);
        } catch (NumberFormatException e) {
            throw new GinkgoException("illegal format of java.math.BigDecimal. " + value, e);
        }
    }

    /**
     * 文字列を java.util.regex.Pattern に変換して返却します。
     * 
     * @param value
     *            解析する文字列
     * @return 変換結果
     * @throws GinkgoException
     *             変換に失敗した場合
     */
    static Pattern toRegexPattern(String value) {
        try {
            return Pattern.compile(value);
        } catch (PatternSyntaxException e) {
            throw new GinkgoException("regex syntax error. " + value, e);
        }
    }

    /**
     * 文字列をリソースパスとして java.util.Properties を生成して返却します。
     * 
     * @param tag
     *            タグ
     * @param value
     *            解析する文字列
     * @return 変換結果
     * @throws GinkgoException
     *             変換に失敗した場合
     */
    static Properties toProperties(TagNode tag, String value) {
        return toProperties(tag, null, value);
    }

    /**
     * 文字列をリソースパスとして java.util.Properties を生成して返却します。
     * 
     * @param tag
     *            タグ
     * @param defaults
     *            デフォルトのプロパティ
     * @param value
     *            解析する文字列
     * @return 変換結果
     * @throws GinkgoException
     *             変換に失敗した場合
     */
    static Properties toProperties(TagNode tag, Properties defaults, String value) {

        InputStream inStream = null;
        try {

            IOResource ioResource = getIOResource(tag, value);
            inStream = ioResource.getInputStream();

            Properties properties = (defaults == null) ? new Properties() : new Properties(defaults);
            properties.load(inStream);

            return properties;

        } catch (IOException e) {
            throw new GinkgoException("load Properties err. " + value, e);
        } catch (URISyntaxException e) {
            throw new GinkgoException("load Properties err. " + value, e);
        } finally {
            try {
                if (inStream != null) {
                    inStream.close();
                }
            } catch (Exception e) {
                // no op
            }
        }

    }

    /**
     * 文字列をファイルパスとして java.io.File を生成して返却します。
     * 
     * @param value
     *            解析する文字列
     * @return 変換結果
     * @throws GinkgoException
     *             変換に失敗した場合
     */
    static File toFile(String value) {
        try {
            return new File(value);
        } catch (NullPointerException e) {
            throw new GinkgoException("file path is null. " + value, e);
        }
    }

    /**
     * 文字列をURIパスとして java.net.URI を生成して返却します。
     * 
     * @param value
     *            解析する文字列
     * @return 変換結果
     * @throws GinkgoException
     *             変換に失敗した場合
     */
    static URI toURI(String value) {
        try {
            return new URI(value);
        } catch (URISyntaxException e) {
            throw new GinkgoException("illegal URI format. " + value, e);
        }
    }

    /**
     * 文字列を java.util.Date に変換して返却します。
     * 
     * @param value
     *            解析する文字列
     * @return 変換結果
     * @throws GinkgoException
     *             変換に失敗した場合
     */
    static Date toDateTime(String value) {
        try {
            return HCnv.asDataObject(ValueOf.decodeDateTime(value));
        } catch (IllegalArgumentException e) {
            throw new GinkgoException("illegal Date format. value:" + value);
        }
    }

    /**
     * 参照名に対応するタグの値を検索して返却します。<br>
     * 値が存在しない場合は null を返却します。
     * 
     * @param tag
     *            タグ
     * @param value
     *            解析する文字列
     * @return 参照値
     */
    static ReferenceValue toReferenceValue(TagNode tag, String value) {
        if (isContainsReferenceId(tag, value)) {
            Object resultValue = getReferenceValue(tag.getTagContext().getDocument(), value);
            return new ReferenceValue(value, resultValue);
        }
        return null;
    }

    /**
     * 指定された参照名のタグの値を検索して返却します。<br>
     * 値が存在しない場合は null を返却します。
     * 
     * @param doc
     *            ドキュメント
     * @param name
     *            参照名
     * @return 参照値
     */
    static Object getReferenceValue(Document doc, String name) {
        Object resultValue = doc.getTagValueById(name);
        if (resultValue instanceof ObjectCreationProxy) {
            try {
                resultValue = ((ObjectCreationProxy) resultValue).create();
            } catch (ObjectCreationException e) {
                throw new GinkgoException(HLog.list("object creation error. ", doc, name), e);
            }
        }
        return resultValue;
    }

    /**
     * 文字列からクラスをロードして返却します。
     * 
     * @param tag
     *            タグ
     * @param value
     *            解析する文字列
     * @return クラス
     * @throws GinkgoException
     *             クラスロードに失敗した場合
     */
    static Class toClass(TagNode tag, String value) {
        if (Eval.isBlank(value)) {
            return null;
        }
        return loadClass(tag.getTagContext(), value);
    }

    /**
     * 指定された値を組込式として解釈し、その結果を返却します。
     * 
     * @param tag
     *            タグ
     * @param value
     *            解析する文字列
     * @return 組込式の実行結果の値
     */
    static ExpressionValue toExpressionValue(TagNode tag, String value) {
        try {
            final OgdlContext context = getOgdlContext(tag.getTagContext().getDocument());
            final Ogdl ogdl = new Ogdl();
            ogdl.setContext(context);
            context.setClassLoader(tag.getTagContext().getClassLoader());
            Object resultValue = ogdl.evaluate(value);
            return new ExpressionValue(value, resultValue);
        } catch (OgdlSyntaxException e) {
            throw new GinkgoException("expression err. " + value, e);
        }
    }

    /**
     * 文字列中の #{ } で囲まれた部分を組込式として、式展開した文字列を返却します。
     * 
     * @param tag
     *            タグ
     * @param value
     *            解析する文字列
     * @return 式展開された文字列
     */
    static String toEmbeddedExpressionString(TagNode tag, String value) {
        return toEmbeddedExpressionString(tag.getTagContext().getDocument(), value);
    }

    /**
     * 文字列を組込式 ”#{...}” を含む文字列として解析し、テキストと式の結果を連結して返却します。
     * 
     * @param doc
     *            ドキュメント
     * @param value
     *            解析する文字列
     * @return 変換結果
     * @throws GinkgoException
     *             変換に失敗した場合
     */
    static String toEmbeddedExpressionString(Document doc, String value) {
        // el open search
        int off = value.indexOf("#{");
        if (off >= 0) {
            // join ogdl value and text
            try {

                final OgdlContext context = getOgdlContext(doc);

                final StringBuffer sb = new StringBuffer(value.length());
                sb.append(value.substring(0, off));

                final Ogdl ogdl = new Ogdl();
                ogdl.setContext(context);
                context.setClassLoader(doc.getContext().getClassLoader());

                off += "#{".length();
                while (off >= 0) {

                    // exec ogdl
                    // final OgdlEvent event = new OgdlEvent(this, binder, value, context, off);
                    OgdlParseIndex varoff = new OgdlParseIndex(off);
                    Object ogdlValue = ogdl.evaluate(value, varoff);

                    // verify close literal
                    off = varoff.get();
                    if ('}' != value.charAt(off)) {
                        throw new GinkgoException("not find el close literal. offset=" + off + ", value=" + value);
                    }
                    off++;// + close literal size

                    // add ogdl value
                    sb.append(ogdlValue);

                    // next search
                    int next = value.indexOf("#{", off);
                    if (next >= 0) {
                        sb.append(value.substring(off, next));
                        next += "#{".length(); // + open literal size
                    } else {
                        sb.append(value.substring(off));
                    }
                    off = next;
                }

                // return join string
                return sb.toString();

            } catch (OgdlSyntaxException e) {
                throw new GinkgoException("el error. offset=" + off + ", value=" + value, e);
            }
        }
        // doesn't transform
        return value;
    }

    /**
     * 反復子を生成する値オブジェクトを返却します。
     * 
     * @param o
     *            反復子の生成基と為るオブジェクト
     * @return 反復子を生成する値オブジェクト
     */
    static IterateValue toIterateValue(Object o) {
        return new IterateValue(o);
    }

    /**
     * 評価を実行して Boolean を返す値オブジェクトを返却します。
     * 
     * @param tag
     *            タグ
     * @param value
     *            解析する文字列
     * @return 評価を実行して Boolean を返す値オブジェクト
     */
    static EvaluationValue toEvaluationValue(TagNode tag, String value) {
        final OgdlContext context = getOgdlContext(tag.getTagContext().getDocument());
        final String expression = value;
        final Ogdl ogdl = new Ogdl();
        ogdl.setContext(context);
        context.setClassLoader(tag.getTagContext().getClassLoader());
        final Predicate predicate = new Predicate() {
            public boolean evaluate(Object o) {
                try {
                    final Object resultValue = ogdl.evaluate(expression);
                    return Boolean.TRUE.equals(resultValue);
                } catch (OgdlSyntaxException e) {
                    throw new GinkgoException("expression err. " + expression, e);
                }
            }
        };
        return new EvaluationValue(predicate);
    }

    /**
     * 指定された値を リソースのパスとして、IOResource を生成し返却します。
     * 
     * @param tag
     *            タグ
     * @param value
     *            リソースのパス
     * @return IOリソース
     */
    static IOResource toIOResource(TagNode tag, String value) {
        try {
            return getIOResource(tag, value);
        } catch (IOException e) {
            throw new GinkgoException("IOResource initialize error. " + value, e);
        } catch (URISyntaxException e) {
            throw new GinkgoException("IOResource initialize error. " + value, e);
        }
    }

    /*
     * private
     */

    /**
     * IOリソースを返却します。
     * 
     * @param tag
     *            タグ
     * @param url
     *            URL文字列
     * @return IOリソース
     * @throws IOException
     *             IOリソースの生成に失敗した場合
     * @throws URISyntaxException
     *             引数のURIまたはプレフィックスが URI として不正の場合
     */
    static private IOResource getIOResource(TagNode tag, String url) throws IOException, URISyntaxException {
        IOResourceLoader ioResourceLoader = getIOResourceLoader(tag);
        return ioResourceLoader.getIOResource(url);
    }

    /**
     * IOリソース生成機能を返却します。
     * 
     * @param tag
     *            タグ
     * @return IOリソース生成機能
     */
    private static IOResourceLoader getIOResourceLoader(TagNode tag) {
        IOResourceLoader ioResourceLoader = tag.getTagContext().getGinkgo().getIOResourceLoader();
        if (ioResourceLoader == null) {
            ioResourceLoader = FeatureFactory.getLoader().getIOResourceLoader();
        }
        ClassLoader classLoader = tag.getTagContext().getClassLoader();
        ioResourceLoader.setClassLoader(classLoader);
        return ioResourceLoader;
    }

    /**
     * OGDL式のコンテキストを返却します。
     * 
     * @param doc
     *            ドキュメント
     * @return 式コンテキスト
     */
    private static OgdlContext getOgdlContext(Document doc) {
        return new OgdlContextImpl(doc);
    }

    /**
     * 有効な Boolean の文字列表記か検証します。
     * 
     * @param value
     *            検証する文字列
     * @return 有効な Boolean の文字列表記の場合は true
     */
    private static boolean isBoolean(String value) {
        return BOOL_PATTERN.matcher(value).matches();
    }

    /**
     * 指定された数値文字列をJavaで解析可能な数値文字列に変換して返却します。
     * 
     * @param value
     *            解析する文字列
     * @return Javaで解析可能な数値文字列
     */
    private static String getNumberString(String value) {
        if (NUMBER_FORMAT_PATTERN.matcher(value).matches()) {
            return NUMBER_REMOVE_PATTERN.matcher(value).replaceAll("");
        } else {
            throw new NumberFormatException("value:" + value);
        }
    }

    /**
     * 指定された文字列からクラスをロードし返却します。
     * 
     * @param context
     *            タグのコンテキスト情報
     * @param className
     *            クラスを示す文字列
     * @return ロードされたクラス
     */
    private static Class loadClass(TagContext context, String className) {
        try {
            final DocumentContext _context = context.getDocumentContext();
            return HClass.load(className, context.getClassLoader(), new Closure() {
                public Object evaluate(Object o) {
                    return _context.forClass((String) o);
                }
            });
        } catch (NoSuchResourceException e) {
            throw new GinkgoException("Class couldn't be created. name:" + className, e);
        }
    }

    /**
     * 参照名に対応するタグの値が存在するか検証します。
     * 
     * @param tag
     *            タグ
     * @param id
     *            参照名
     * @return 値が存在する場合は true
     */
    private static boolean isContainsReferenceId(TagNode tag, String id) {
        return (!Eval.isBlank(id) && tag.getTagContext().getDocument().containsId(id));
    }

}
