/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ginkgo.rule;

import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import shohaku.core.lang.Eval;
import shohaku.core.lang.feature.FeatureFactory;
import shohaku.core.resource.IOResource;
import shohaku.core.resource.IOResourceLoader;
import shohaku.ginkgo.Ginkgo;
import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.NodeCompositeRule;
import shohaku.ginkgo.TagAttributesRule;
import shohaku.ginkgo.TagAttributesRuleSet;
import shohaku.ginkgo.TagNode;
import shohaku.ginkgo.TagRule;
import shohaku.ginkgo.TagRuleSet;
import shohaku.ginkgo.ValueNode;

/**
 * XML形式で定義されたデータから NodeCompositeRule を生成する機能を提供します。
 */
public class XMLNodeCompositeRuleFactory {

    /* 構成情報を読み取る構成ルール */
    private static final NodeCompositeRule nodeCompositeRule;
    static {
        List tagRules = new ArrayList(50);
        // root
        addContainTag(tagRules, "/ginkgo-composite-rule");
        addContainTag(tagRules, "/ginkgo-tag-library");
        // /rules
        addContainTag(tagRules, "/rules");
        addCoreTag(tagRules, "/rules/node-composite-rule", "beans.ObjectTag");
        // /library
        addContainTag(tagRules, "/library");
        // /tag-mapping
        addContainTag(tagRules, "/tag-mapping");
        addContainTag(tagRules, "/tag-mapping/tag");
        addContainTag(tagRules, "/tag-mapping/tag/attributes");
        addContainTag(tagRules, "/tag-mapping/tag/attributes/attribute");
        // /pattern-mapping
        addContainTag(tagRules, "/pattern-mapping");
        addContainTag(tagRules, "/pattern-mapping/pattern");

        // 値ノード
        addCoreTag(tagRules, "/string", "base.StringTag");
        addCoreTag(tagRules, "/string/append", "base.ValueTag");
        addCoreTag(tagRules, "/char", "base.CharacterTag");
        addCoreTag(tagRules, "/byte", "base.ByteTag");
        addCoreTag(tagRules, "/short", "base.ShortTag");
        addCoreTag(tagRules, "/int", "base.IntgerTag");
        addCoreTag(tagRules, "/long", "base.LongTag");
        addCoreTag(tagRules, "/double", "base.DoubleTag");
        addCoreTag(tagRules, "/float", "base.FloatTag");
        addCoreTag(tagRules, "/boolean", "base.BooleanTag");
        addCoreTag(tagRules, "/type", "base.ClassTag");
        addCoreTag(tagRules, "/ref", "base.ReferenceTag");
        addCoreTag(tagRules, "/bigDecimal", "math.BigDecimalTag");
        addCoreTag(tagRules, "/bigInteger", "math.BigIntegerTag");
        addCoreTag(tagRules, "/date", "util.DateTimeTag");
        addCoreTag(tagRules, "/value", "util.ExpressionTag");
        addCoreTag(tagRules, "/array", "collections.ArrayTag");
        addCoreTag(tagRules, "/list", "collections.ListTag");
        addCoreTag(tagRules, "/set", "collections.SetTag");
        addCoreTag(tagRules, "/map", "collections.MapTag");
        addCoreTag(tagRules, "/map/entry", "collections.SingletonMapTag");
        addCoreTag(tagRules, "/object", "beans.ObjectTag");
        addCoreTag(tagRules, "/init", "beans.FactoryMethodDescTag");
        addCoreTag(tagRules, "/init/arg", "beans.ArgumentDescTag");
        addCoreTag(tagRules, "/factory", "beans.FactoryMethodDescTag");
        addCoreTag(tagRules, "/factory/arg", "beans.ArgumentDescTag");
        addCoreTag(tagRules, "/property", "beans.SetPropertyDescTag");
        addCoreTag(tagRules, "/method", "beans.MethodDescTag");
        addCoreTag(tagRules, "/method/arg", "beans.ArgumentDescTag");
        // other
        // addContainTag(tagRules, "*");

        NodeCompositeRule rule = new DefaultNodeCompositeRule();
        rule.addTagRule(null, new TagRuleSet(tagRules));
        nodeCompositeRule = rule;
    }

    private static void addContainTag(List tagRules, String pattern) {
        TagRule tagRule = new TagRule();
        tagRule.setPattern(pattern);
        tagRule.setTagClass("shohaku.ginkgo.tags.PublicContainTag");
        tagRules.add(tagRule);
    }

    private static void addCoreTag(List tagRules, String pattern, String className) {
        TagRule tagRule = new TagRule();
        tagRule.setPattern(pattern);
        tagRule.setTagClass("shohaku.ginkgo.tags.core." + className);
        tagRules.add(tagRule);
    }

    /* タグライブラリファイルのIOリソース生成機能 */
    private IOResourceLoader libraryIOResourceLoader;

    /* ノード構成ルールを定義するIOリソース */
    private IOResource ioResource;

    /* 読み取りに使用するクラスローダ */
    private ClassLoader classLoader;

    /**
     * クラスの生成に使用するクラスローダを返却します。
     * 
     * @return クラスローダ
     */
    public ClassLoader getClassLoader() {
        return classLoader;
    }

    /**
     * クラスの生成に使用するクラスローダを格納します。
     * 
     * @param loader
     *            クラスローダ
     */
    public void setClassLoader(ClassLoader loader) {
        this.classLoader = loader;
    }

    /**
     * ノード構成ルールを定義するIOリソースを返却します。
     * 
     * @return IOリソース
     */
    public IOResource getIOResource() {
        return ioResource;
    }

    /**
     * ノード構成ルールを定義するIOリソースを格納します。
     * 
     * @param ioResource
     *            IOリソース
     */
    public void setIOResource(IOResource ioResource) {
        this.ioResource = ioResource;
    }

    /**
     * タグライブラリファイルのIOリソース生成機能を返却します。<br>
     * null の場合は FeatureFactory.getLoader().getSystemIOResourceLoader() が使用されます。
     * 
     * @return IOリソース生成機能
     */
    public IOResourceLoader getLibraryIOResourceLoader() {
        return libraryIOResourceLoader;
    }

    /**
     * タグライブラリファイルのIOリソース生成機能を格納します。
     * 
     * @param ioResourceLoader
     *            IOリソース生成機能
     */
    public void setLibraryIOResourceLoader(IOResourceLoader ioResourceLoader) {
        this.libraryIOResourceLoader = ioResourceLoader;
    }

    /**
     * ノード構成ルールを生成して返却します。
     * 
     * @return 生成されたノード構成ルール
     */
    public NodeCompositeRule create() {
        IOResource resource = getIOResource();
        ClassLoader loader = getClassLoader();
        if (resource == null) {
            throw new GinkgoException("IOResource is null.");
        }
        if (loader == null) {
            loader = this.getClass().getClassLoader();
        }

        InputStream is = null;
        try {

            is = resource.getInputStream();
            return build(is, loader);

        } catch (IOException e) {
            throw new GinkgoException("NodeCompositeRule create err.", e);
        } finally {
            try {
                if (is != null) {
                    is.close();
                }
            } catch (Exception e) {
                // no op
            }
        }
    }

    /* 引数の入力ストリームから NodeCompositeRule を生成して返却します。 */
    private NodeCompositeRule build(InputStream is, ClassLoader loader) {
        try {

            TagNode rootTag = load(is, loader);
            return build(rootTag, loader);

        } catch (Exception e) {
            throw new GinkgoException("NodeCompositeRule create err.", e);
        }
    }

    /* 入力ストリームからノードリストを読み込む。 */
    private TagNode load(InputStream is, ClassLoader loader) {

        // XML構成ルールファイルの解析
        Ginkgo ginkgo = new Ginkgo();
        if (loader != null) {
            ginkgo.setClassLoader(loader);
        }
        ginkgo.setNodeCompositeRule(nodeCompositeRule);
        ginkgo.parse(is);// XMLの解析
        return ginkgo.getDocument().getContext().getRoot();// ルートタグを返却

    }

    /* XML構成ルールを構築します。 */
    private NodeCompositeRule build(TagNode rootTag, ClassLoader loader) {

        Map tagGroupMapping = new HashMap();
        Map uriGroupMapping = new HashMap();

        NodeCompositeRule rule = null;
        for (Iterator i = rootTag.getTagContext().elementIterator(); i.hasNext();) {
            TagNode tag = (TagNode) i.next();
            String name = tag.getTagContext().getTagName();
            if ("rules".equals(name)) {
                rule = buildNodeCompositeRule(tag);
            } else if ("library".equals(name)) {
                buildLibrary(tag, tagGroupMapping, uriGroupMapping, loader);
            }
        }
        if (rule == null) {
            rule = new DefaultNodeCompositeRule();
        }
        for (Iterator i = uriGroupMapping.entrySet().iterator(); i.hasNext();) {
            Map.Entry e = (Map.Entry) i.next();
            rule.addTagRule((String) e.getKey(), new TagRuleSet((List) e.getValue()));
        }
        return rule;
    }

    private void buildLibrary(TagNode libraryTag, Map tagGroupMapping, Map uriGroupMapping, ClassLoader loader) {

        Map tagMapping = new HashMap();
        Map patternMapping = new HashMap();

        String libraryId = libraryTag.getTagContext().getAttribute("id");
        String nsuri = libraryTag.getTagContext().getAttribute("nsuri", "");
        String classPath = libraryTag.getTagContext().getAttribute("class-path");

        if (!Eval.isBlank(classPath)) {
            loadSubLibrary(classPath, tagMapping, patternMapping, loader);
        }

        buildLibraryTag(libraryTag, tagMapping, patternMapping);
        putTagGroupMapping(libraryId, tagGroupMapping, tagMapping);
        List tagRules = buildTagRuleList(tagGroupMapping, tagMapping, patternMapping);
        List l = (List) uriGroupMapping.get(nsuri);
        if (l == null) {
            uriGroupMapping.put(nsuri, tagRules);
        } else {
            l.addAll(tagRules);
        }
    }

    private void loadSubLibrary(String path, Map tagMapping, Map patternMapping, ClassLoader loader) {

        IOResourceLoader irl = getLibraryIOResourceLoader();
        if (irl == null) {
            irl = FeatureFactory.getLoader().getIOResourceLoader();
        }
        irl.setClassLoader(loader);

        InputStream is = null;
        try {

            IOResource ir = irl.getIOResource(path);
            is = ir.getInputStream();
            TagNode libraryTag = load(is, loader);
            buildSubLibraryTag(libraryTag, tagMapping, patternMapping, loader);

        } catch (IOException e) {
            throw new GinkgoException("NodeCompositeRule create err.", e);
        } catch (URISyntaxException e) {
            throw new GinkgoException("NodeCompositeRule create err.", e);
        } finally {
            try {
                if (is != null) {
                    is.close();
                }
            } catch (Exception e) {
                // no op
            }
        }

    }

    private void buildLibraryTag(TagNode libraryTag, Map tagMapping, Map patternMapping) {
        for (Iterator i = libraryTag.getTagContext().elementIterator(); i.hasNext();) {
            TagNode tag = (TagNode) i.next();
            String name = tag.getTagContext().getTagName();
            if ("tag-mapping".equals(name)) {
                buildTagMapping(tag, tagMapping);
            } else if ("pattern-mapping".equals(name)) {
                buildPatternMapping(tag, patternMapping);
            }
        }
    }

    private void buildSubLibraryTag(TagNode libraryTag, Map tagMapping, Map patternMapping, ClassLoader loader) {
        for (Iterator i = libraryTag.getTagContext().elementIterator(); i.hasNext();) {
            TagNode tag = (TagNode) i.next();
            String name = tag.getTagContext().getTagName();
            if ("tag-mapping".equals(name)) {
                buildTagMapping(tag, tagMapping);
            } else if ("pattern-mapping".equals(name)) {
                buildPatternMapping(tag, patternMapping);
            } else if ("library".equals(name)) {
                String classPath = libraryTag.getTagContext().getAttribute("class-path");
                loadSubLibrary(classPath, tagMapping, patternMapping, loader);
            }
        }
    }

    private void putTagGroupMapping(String libraryId, Map tagGroupMapping, Map tagMapping) {
        for (Iterator i = tagMapping.entrySet().iterator(); i.hasNext();) {
            Map.Entry e = (Map.Entry) i.next();
            tagGroupMapping.put(libraryId + ":" + e.getKey(), e.getValue());
        }
    }

    private List buildTagRuleList(Map tagGroupMapping, Map tagMapping, Map patternMapping) {
        List tagRuleSet = new ArrayList(patternMapping.size());
        for (Iterator i = patternMapping.entrySet().iterator(); i.hasNext();) {
            Map.Entry e = (Map.Entry) i.next();
            TagRule tagRule = new TagRule();
            String pattern = (String) e.getKey();
            String tagId = (String) e.getValue();

            Map tagDesc = (Map) tagMapping.get(tagId);
            if (tagGroupMapping != null && tagDesc == null) {
                tagDesc = (Map) tagGroupMapping.get(tagId);
            }
            if (tagDesc == null) {
                throw new NullPointerException("tag doesn't exist. tagId=" + tagId);
            }
            String className = (String) tagDesc.get("class");
            TagAttributesRuleSet attributes = (TagAttributesRuleSet) tagDesc.get("attributes");

            tagRule.setPattern(pattern);
            tagRule.setTagClass(className);
            tagRule.setTagAttributesRuleSet(attributes);

            tagRuleSet.add(tagRule);
        }
        return tagRuleSet;
    }

    /* NodeCompositeRule を生成して返却します。 */
    private NodeCompositeRule buildNodeCompositeRule(TagNode rulesTag) {
        Iterator i = rulesTag.getTagContext().valueIterator("node-composite-rule");
        if (i.hasNext()) {
            ValueNode vnode = (ValueNode) i.next();
            return (NodeCompositeRule) vnode.getNodeValue();
        }
        return new DefaultNodeCompositeRule();
    }

    /* NODESタグの情報を格納します。 */
    private void buildTagMapping(TagNode tagMappingTag, Map tagMapping) {
        for (Iterator i = tagMappingTag.getTagContext().elementIterator("tag"); i.hasNext();) {
            TagNode tag = (TagNode) i.next();
            String tagId = tag.getTagContext().getAttribute("id");

            Map tagDesc = new HashMap(2);
            tagDesc.put("class", tag.getTagContext().getAttribute("class"));
            for (Iterator j = tag.getTagContext().elementIterator(); j.hasNext();) {
                TagNode t = (TagNode) j.next();
                String nm = t.getTagContext().getTagName();
                if ("attributes".equals(nm)) {
                    tagDesc.put(nm, buildTagAttributesRule(t));
                }
            }
            tagMapping.put(tagId, tagDesc);
        }
    }

    private TagAttributesRuleSet buildTagAttributesRule(TagNode attributesTag) {
        List rules = new ArrayList();
        for (Iterator i = attributesTag.getTagContext().elementIterator("attribute"); i.hasNext();) {
            TagNode att = (TagNode) i.next();
            String name = att.getTagContext().getAttribute("name");
            String alias = att.getTagContext().getAttribute("alias");
            String defaultValue = att.getTagContext().getAttribute("defaultValue");
            rules.add(new TagAttributesRule(name, alias, defaultValue));
        }
        return new TagAttributesRuleSet(rules);
    }

    /* TAGSタグの情報を格納します。 */
    private void buildPatternMapping(TagNode patternMappingTag, Map patternMapping) {
        for (Iterator i = patternMappingTag.getTagContext().elementIterator(); i.hasNext();) {
            TagNode tag = (TagNode) i.next();
            String name = tag.getTagContext().getTagName();
            if ("pattern".equals(name)) {
                patternMapping.put(tag.getTagContext().getAttribute("path"), tag.getTagContext().getAttribute("tag"));
            } else if ("remove".equals(name)) {
                patternMapping.remove(tag.getTagContext().getAttribute("path"));
            }
        }
    }
}
