/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ginkgo.tags.core.beans;

import shohaku.core.beans.dynamic.ArgumentDesc;
import shohaku.core.beans.dynamic.ArgumentsDesc;
import shohaku.core.beans.dynamic.MethodDesc;
import shohaku.core.lang.Eval;
import shohaku.core.lang.ObjectCreationProxy;
import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.tags.AbstractValueTag;
import shohaku.ginkgo.type.ExpressionValue;

/**
 * セッタープロパティのメソッドを表現する値タグを提供します。MethodDesc.SET_PROPERTY を持つ shohaku.core.beans.MethodDesc 型のデータを生成します。<br>
 * 子のエレメントが定義されない場合は、テキストを組み込み式として解析します、デフォルトではOGDL式が使用されます。
 */
public class SetPropertyDescTag extends AbstractValueTag {

    /* null と未設定を分別する為の参照識別子 */
    private static final Object NULL = new Object();

    /* 単一の要素データを格納する */
    private Object singleton = NULL;

    /* テキストから自動変換されたデータを格納する */
    private ExpressionValue textValue = null;

    /* 引数のデータ型 */
    private Class type = null;

    /**
     * 値を生成して返却します。
     * 
     * @return 生成された値
     */
    protected Object generateValue() {

        // property name
        String propertyName = getName();
        if (Eval.isBlank(propertyName)) {
            throw new GinkgoException("name is blank." + propertyName);
        }

        // 双方未設定は不正
        if (singleton == NULL && textValue == null) {
            throw new GinkgoException("element and text is null.");
        }

        // テキストと要素は相互排他の為、必ず一方のみ
        Object propertyValue = (textValue != null) ? textValue.getResultValue() : singleton;

        // property type
        Class propertyType = type;
        if (propertyType == null) {
            if (propertyValue == null) {
                propertyType = Object.class;
            } else if (propertyValue instanceof ObjectCreationProxy) {
                propertyType = ((ObjectCreationProxy) propertyValue).getInstanceType();
            } else {
                propertyType = propertyValue.getClass();
            }
        }

        // generate MethodDesc
        ArgumentDesc[] args = new ArgumentDesc[] { new ArgumentDesc(propertyType, propertyValue) };
        return new MethodDesc(propertyName, new ArgumentsDesc(args), MethodDesc.SET_PROPERTY);
    }

    /**
     * type 属性を格納します。
     * 
     * @param type
     *            type属性
     */
    public void setType(Class type) {
        this.type = type;
    }

    /**
     * type 属性を返却します。
     * 
     * @return type 属性
     */
    public Class getType() {
        return this.type;
    }

    /**
     * 組み込み式としてテキストを変換して格納します。
     * 
     * @param textValue
     *            変換されたテキスト
     */
    public void setTextTransferValue(ExpressionValue textValue) {
        this.textValue = textValue;
    }

    /**
     * エレメントを追加します。
     * 
     * @param element
     *            エレメント
     */
    public void addElementTransferValue(Object element) {
        if (singleton == NULL) {
            this.singleton = element;
        } else {
            throw new GinkgoException("element size is one.");
        }
    }

}
