/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ginkgo.tags.core.collections;

import java.util.Collections;

import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.tags.AbstractValueTag;
import shohaku.ginkgo.type.ReferenceValue;

/**
 * キーと値が一つのみ持つマップを生成するノードを提供します。
 */
public class SingletonMapTag extends AbstractValueTag {

    private static final Object NULL = new Object();

    private Object key = NULL;

    private Object value = NULL;

    /**
     * 値を生成して返却します。
     * 
     * @return 生成された値
     */
    protected Object generateValue() {

        if (key == NULL || value == NULL) {
            throw new GinkgoException("entry size is two.");
        }

        return Collections.singletonMap(key, value);
    }

    /**
     * キーを格納します。
     * 
     * @param key
     *            キー
     */
    public void setKey(Object key) {
        if (key != NULL) {
            throw new GinkgoException("key.");
        }
        this.key = key;
    }

    /**
     * キーとなるタグ参照を格納します。
     * 
     * @param ref
     *            タグ参照
     */
    public void setKeyRef(ReferenceValue ref) {
        if (key != NULL) {
            throw new GinkgoException("key.");
        }
        this.key = ref.getResultValue();
    }

    /**
     * 値を格納します。
     * 
     * @param value
     *            値
     */
    public void setValue(Object value) {
        if (value != NULL) {
            throw new GinkgoException("value.");
        }
        this.value = value;
    }

    /**
     * 値となるタグ参照を格納します。
     * 
     * @param ref
     *            タグ参照
     */
    public void setValueRef(ReferenceValue ref) {
        if (value != NULL) {
            throw new GinkgoException("value.");
        }
        this.value = ref.getResultValue();
    }

    /**
     * type 属性を返却します。
     * 
     * @return type 属性
     */
    public Object getKey() {
        return this.key;
    }

    /**
     * value 属性を返却します。
     * 
     * @return value 属性
     */
    public Object getValue() {
        return this.value;
    }

    /**
     * エレメントを追加します。
     * 
     * @param element
     *            エレメント
     */
    public void addElementTransferValue(Object element) {
        if (key == NULL) {
            this.key = element;
        } else {
            if (value != NULL) {
                throw new GinkgoException("entry size is two.");
            }
            this.value = element;
        }
    }

}
