/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ginkgo.tags.core.logic;

import java.util.Iterator;

import shohaku.core.collections.IteratorUtils;
import shohaku.core.lang.Boxing;
import shohaku.core.lang.Eval;
import shohaku.ginkgo.EvaluationTag;
import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.tags.AbstractLogicTag;
import shohaku.ginkgo.type.EvaluationValue;
import shohaku.ginkgo.type.IterateValue;

/**
 * 属性が指定する回数または指定のコレクションの要素数分の反復処理を行いコンテンツを再評価するタグを提供します。
 */
public class ForEachTag extends AbstractLogicTag {

    /*
     * 属性
     */

    /* 反復処理の開始インデックス。 */
    private int begin = 0;

    /* 反復処理の終了インデックス。 */
    private int end = Integer.MAX_VALUE;

    /* 要素のインデックスへの参照識別子（デフォルト：index）。 */
    private String indexVar = "index";

    /* 反復処理のカウント数への参照識別子（デフォルト：count）。 */
    private String countVar = "count";

    /* コレクションの要素への参照識別子（デフォルト：item）。 */
    private String itemVar = "item";

    /* 反復対象のコレクション。 */
    private IterateValue items = null;

    /* 反復の有効性を検証する機能 */
    private EvaluationValue test = null;

    /*
     * 変数
     */

    /* 反復子。 */
    private Iterator itemsIterator;

    /* 要素のインデックス。 */
    private int index;

    /* 反復のカウント数。 */
    private int count;

    /**
     * コンテンツ情報を初期化します。
     * 
     * @return EvaluationTag.INIT_BODY
     */
    public int doInitBody() {

        if (this.items != null) {
            this.itemsIterator = this.items.iterator();
            // 位置の移動（初期値=０回）
            IteratorUtils.shift(this.itemsIterator, this.begin);
        } else {
            // 無限反復を行う反復子（最大回数で停止する）
            this.itemsIterator = IteratorUtils.infiniteLoopIterator();
        }
        this.index = this.begin;
        this.count = 0;
        return EvaluationTag.INIT_BODY;
    }

    /**
     * 指定された属性に応じて再評価を繰り返します。
     * 
     * @return 評価を終了する場合は EvaluationTag.END_BODY 最評価を要求する場合は EvaluationTag.EVAL_BODY
     */
    public int doEvalBody() {
        // 再評価が有効か検証する
        if (hasNext()) {
            getTagContext().getDocumentContext().setAttribute(this.indexVar, Boxing.box(this.index));
            getTagContext().getDocumentContext().setAttribute(this.countVar, Boxing.box(this.count));
            getTagContext().getDocumentContext().setAttribute(this.itemVar, this.itemsIterator.next());
            this.index++;
            this.count++;
            return EvaluationTag.EVAL_BODY;
        }
        // 再評価の終了
        else {
            getTagContext().getDocumentContext().removeAttribute(this.indexVar);
            getTagContext().getDocumentContext().removeAttribute(this.countVar);
            getTagContext().getDocumentContext().removeAttribute(this.itemVar);
            return EvaluationTag.END_BODY;
        }
    }

    /* 次の要素を評価可能かを返却します。 */
    private boolean hasNext() {
        // 検証がTrueでありインデックスか反復子が終端でない場合は再評価する
        return index < end && itemsIterator.hasNext() && (test == null || test.evaluate().booleanValue());
    }

    /**
     * 要素のインデックスの変数名を格納します、デフォルトは index です。
     * 
     * @param indexVar
     *            要素のインデックスの変数名
     */
    public void setIndexVar(String indexVar) {
        if (Eval.isEmpty(indexVar)) {
            throw new GinkgoException("indexVar is empty." + indexVar);
        }
        this.indexVar = indexVar;
    }

    /**
     * 反復のカウント数の変数名を格納します、デフォルトは count です。
     * 
     * @param countVar
     *            反復のカウント数の変数名
     */
    public void setCountVar(String countVar) {
        if (Eval.isEmpty(countVar)) {
            throw new GinkgoException("countVar is empty." + countVar);
        }
        this.countVar = countVar;
    }

    /**
     * 要素の値の変数名を格納します、デフォルトは item です。
     * 
     * @param itemVar
     *            要素の値の変数名
     */
    public void setItemVar(String itemVar) {
        if (Eval.isEmpty(itemVar)) {
            throw new GinkgoException("itemVar is empty." + itemVar);
        }
        this.itemVar = itemVar;
    }

    /**
     * 反復する最大回数を格納します、デフォルトは Integer.MAX_VALUE です。
     * 
     * @param end
     *            反復する最大回数
     */
    public void setEnd(int end) {
        if (0 > end) {
            throw new GinkgoException("end is negative number." + end);
        }
        this.end = end;
    }

    /**
     * 処理以前に反復子の位置を移動する回数を格納します、デフォルトは 0 です。
     * 
     * @param begin
     *            位置を移動する回数
     */
    public void setBegin(int begin) {
        if (0 > begin) {
            throw new GinkgoException("begin is negative number." + begin);
        }
        this.begin = begin;
    }

    /**
     * 反復する反復子情報を格納します、デフォルトは最大回数まで反復を行います。
     * 
     * @param items
     *            反復子情報
     */
    public void setItems(IterateValue items) {
        this.items = items;
    }

    /**
     * 検証機能を格納します。
     * 
     * @param test
     *            検証機能
     */
    protected void setTest(EvaluationValue test) {
        this.test = test;
    }

}
