<?php
/**
 * CSSJの転送プロトコル(CTIP)の低レベルの部分を扱います。
 * 
 * 通常、プログラマがこのパッケージを直接使う必要はありません。
 * 
 * @package cssj
 * @subpackage ctip
 */

/**
 * ユーティリティです。
 */
require_once ('cssj_helpers.php');

/**
 * リクエストパケットタイプ:プロパティ
 * @access public
 */
define ('CSSJ_CTIP_REQ_PROPERTY', 1);
/**
 * リクエストパケットタイプ:リソース
 * @access public
 */
define ('CSSJ_CTIP_REQ_RESOURCE', 2);
/**
 * リクエストパケットタイプ:本体
 * @access public
 */
define ('CSSJ_CTIP_REQ_MAIN', 3);
/**
 * リクエストパケットタイプ:データ
 * @access public
 */
define ('CSSJ_CTIP_REQ_DATA', 4);
/**
 * リクエストパケットタイプ:終了
 * @access public
 */
define ('CSSJ_CTIP_REQ_END', 5);

/**
 * レスポンスパケットタイプ:フラグメント追加
 * @access public
 */
define ('CSSJ_CTIP_RES_ADD', 1);
/**
 * レスポンスパケットタイプ:フラグメント挿入
 * @access public
 */
define ('CSSJ_CTIP_RES_INSERT', 2);
/**
 * レスポンスパケットタイプ:エラーメッセージ
 * @access public
 */
define ('CSSJ_CTIP_RES_ERROR', 3);
/**
 * レスポンスパケットタイプ:データ
 * @access public
 */
define ('CSSJ_CTIP_RES_DATA', 4);

/**
 * メモリ上のフラグメントの最大サイズです。
 * 
 * フラグメントがこの大きさを超えるとディスクに書き込みます。
 * 
 * @access private
 */
define ('CSSJ_CTIP_FRG_MEM_SIZE', 256);

/**
 * メモリ上に置かれるデータの最大サイズです。
 * 
 * メモリ上のデータがこのサイズを超えると、
 * CSSJ_CTIP_FRG_MEM_SIZEとは無関係にディスクに書き込まれます。
 * 
 * @access private
 */
define ('CSSJ_CTIP_ON_MEMORY', 1024 * 1024);

/**
 * 一時ファイルのセグメントサイズです。
 *
 * @access private
 */
define ('CSSJ_CTIP_SEGMENT_SIZE', 8192);

/**
 * セッションを開始します。
 * 
 * @param $fp resource ストリーム
 * @param $encoding string 通信に用いるエンコーディング
 * @return boolean 成功ならtrue,失敗ならfalse
 * @access public
 */
function cssj_ctip_connect(&$fp, $encoding) {
  $err = fwrite($fp, "CTIP/1.0 $encoding\n");
  return ($err === false) ? false : true; 
}

/**
 * プロパティを送ります。
 * 
 * @param $fp resource ストリーム
 * @param $name string 名前
 * @param $value string 値
 * @return boolean 成功ならtrue,失敗ならfalse
 * @access public
 */
function cssj_ctip_req_property(&$fp, $name, $value) {
  $payload = strlen($name) + strlen($value) + 5;
  if (cssj_utils_write_int($fp, $payload) === false) {
    return false;
  }
  if (cssj_utils_write_byte($fp, CSSJ_CTIP_REQ_PROPERTY) === false) {
    return false;
  }
  if (cssj_utils_write_bytes($fp, $name) === false) {
    return false;
  }
  if (cssj_utils_write_bytes($fp, $value) === false) {
    return false;
  }
  return fflush($fp);
}

/**
 * リソースの開始を通知します。
 * 
 * @param $fp resource ストリーム
 * @param $uri string URI
 * @param $mimeType string MIME型
 * @param $encoding string エンコーディング
 * @return boolean 成功ならtrue,失敗ならfalse
 * @access public
 */
function cssj_ctip_req_resource(&$fp, $uri, $mimeType = 'text/css', $encoding = '') {
  $payload = strlen($uri) + strlen($mimeType) + strlen($encoding) + 5;
  if (cssj_utils_write_int($fp, $payload) === false) {
    return false;
  }
  if (cssj_utils_write_byte($fp, CSSJ_CTIP_REQ_RESOURCE) === false) {
    return false;
  }
  if (cssj_utils_write_bytes($fp, $uri) === false) {
    return false;
  }
  if (cssj_utils_write_bytes($fp, $mimeType) === false) {
    return false;
  }
  if (cssj_utils_write_bytes($fp, $encoding) === false) {
    return false;
  }
  return fflush($fp);
}

/**
 * 本体の開始を通知します。
 * 
 * @param $fp resource ストリーム
 * @param $uri string URI
 * @param $mimeType string MIME型
 * @param $encoding string エンコーディング
 * @return boolean 成功ならtrue,失敗ならfalse
 * @access public
 */
function cssj_ctip_req_main(&$fp, $uri, $mimeType = 'text/html', $encoding = '') {
  $payload = strlen($uri) + strlen($mimeType) + strlen($encoding) + 5;
  if (cssj_utils_write_int($fp, $payload) === false) {
    return false;
  }
  if (cssj_utils_write_byte($fp, CSSJ_CTIP_REQ_MAIN) === false) {
    return false;
  }
  if (cssj_utils_write_bytes($fp, $uri) === false) {
    return false;
  }
  if (cssj_utils_write_bytes($fp, $mimeType) === false) {
    return false;
  }
  if (cssj_utils_write_bytes($fp, $encoding) === false) {
    return false;
  }
  return fflush($fp);
}

/**
 * データを送ります。
 * 
 * @param $fp resource ストリーム
 * @param $b string データ
 * @param $len int データの長さ
 * @return boolean 成功ならtrue,失敗ならfalse
 * @access public
 */
function cssj_ctip_req_write(&$fp, &$b, $len = -1) {
  if ($len == -1) {
    $len = strlen($b);
  }
  else {
    $len = min(strlen($b), $len);
  }
  $payload = $len + 1;
  if (cssj_utils_write_int($fp, $payload) === false) {
    return false;
  }
  if (cssj_utils_write_byte($fp, CSSJ_CTIP_REQ_DATA) === false) {
    return false;
  }
  if (fwrite($fp, $b, $len) === false) {
    return false;
  }
  return fflush($fp);
}

/**
 * 終了を通知します。
 * 
 * @param $fp resource ストリーム
 * @return boolean 成功ならtrue,失敗ならfalse
 * @access public
 */
function cssj_ctip_req_end(&$fp) {
  if (cssj_utils_write_int($fp, 0) === false) {
    return false;
  }
  return fflush($fp);
}

/**
 * 次のレスポンスを取得します。
 * 
 * レスポンス(array)には次のデータが含まれます。
 * 
 * - 'type' レスポンスタイプ
 * - 'anchorId' 挿入する場所の直後のフラグメントID
 * - 'level' エラーレベル
 * - 'error' エラーメッセージ
 * - 'id' 断片ID
 * - 'progress' 処理済バイト数
 * - 'bytes' データのバイト列
 * 
 * @param $fp resource ストリーム
 * @return mixed レスポンス,レスポンスの終了ならnull,失敗ならfalse
 * @access public
 */
function cssj_ctip_res_next(&$fp) {
  if (($payload = cssj_utils_read_int($fp)) === false) {
    return false;
  }
  if ($payload === 0) {
    return null;
  }
  if (($type = cssj_utils_read_byte($fp)) === false) {
    return false;
  }
  
  switch ($type) {
      case CSSJ_CTIP_RES_ADD:
      return array(
        'type' => $type,
      );
      
      case CSSJ_CTIP_RES_INSERT:
      if (($anchorId = cssj_utils_read_int($fp)) === false) {
        return false;
      }
      return array(
        'type' => $type,
        'anchorId' => $anchorId
      );
      
      case CSSJ_CTIP_RES_ERROR:
      if (($level = cssj_utils_read_byte($fp)) === false) {
        return false;
      }
      if (($error = cssj_utils_read_bytes($fp)) === false) {
        return false;
      }
      return array(
        'type' => $type,
        'level' => $level,
        'error' => $error
      );
      
      case CSSJ_CTIP_RES_DATA:
      if (($id = cssj_utils_read_int($fp)) === false) {
        return false;
      }
      if (($progress = cssj_utils_read_int($fp)) === false) {
        return false;
      }
      if (($bytes = _cssj_read($fp, $payload - 9)) === false) {
        return false;
      }
      return array(
        'type' => $type,
        'id' => $id,
        'progress' => $progress,
        'bytes' => &$bytes
      );
      
      default:
      trigger_error ("Bad response type:$type", E_USER_ERROR);
      return false;
  }
}

/**
 * レスポンスからデータを構築します。
 * 
 * @param $fp resource ストリーム
 * @param $out mixed 出力先ストリーム(resource),出力先変数(string),または標準出力であればnull。
 * @param $errorFunc エラーコールバック関数
 * @param $progressFunc 進行状況コールバック関数
 * @return boolean 成功ならtrue,失敗ならfalse
 * @access public
 */
function cssj_ctip_res_build(&$fp, &$out, &$errorFunc, &$progressFunc) {
  $builder = cssj_ctip_res_create_builder($fp, $out, $errorFunc, $progressFunc);
  while (($err = cssj_ctip_res_build_next($builder)) === true)
    ;
  return ($err === null) ? true : false;
}

/**
 * ビルダーを構築します。
 * 
 * @param $fp resource ストリーム
 * @param $out mixed 出力先ストリーム(resource),出力先変数(string),または標準出力であればnull。
 * @param $errorFunc エラーコールバック関数
 * @param $progressFunc 進行状況コールバック関数
 * @return mixed ビルダー,エラーの場合はfalse
 * @access public
 */
function &cssj_ctip_res_create_builder(&$fp, &$out, &$errorFunc, &$progressFunc) {
  if (($tempFile = tmpfile()) === false) {
    return false;
  }
  $builder = array(
    'fp' => &$fp,
    'out' => &$out,
    'errorFunc' => &$errorFunc,
    'progressFunc' => &$progressFunc,
    'tempFile' => &$tempFile,
    'frgs' => array(),
    'first' => null,
    'last' => null,
    'onMemory' => 0,
    'length' => 0,
    'segment' => 0
  );
  return $builder;
}

/**
 * 次のビルドタスクを実行します。
 * 
 * @param $builder array ビルダー
 * @return mixed 次がある場合はtrue,終わった場合はnull,エラーの場合はfalse
 */
function cssj_ctip_res_build_next(&$builder) {
  $fp =& $builder['fp'];
  $out =& $builder['out'];
  $errorFunc =& $builder['errorFunc'];
  $progressFunc =& $builder['progressFunc'];
  $tempFile =& $builder['tempFile'];
  $frgs =& $builder['frgs'];
  $first =& $builder['first'];
  $last =& $builder['last'];
  $onMemory =& $builder['onMemory'];
  $length =& $builder['length'];
  $segment =& $builder['segment'];
  
  if (($next = cssj_ctip_res_next($fp)) !== null) {
    if ($next === false) {
      fclose($tempFile);
      return false;
    }
    $type = $next['type'];
    switch ($type) {
      case CSSJ_CTIP_RES_ADD:
      $id = count($frgs);
      $frg =& _cssj_new_frg($id);
      $frgs[$id] =& $frg;
      if (is_null($first)) {
        $first =& $frg;
      }
      else {
        $last['next'] =& $frg;
        $frg['prev'] =& $last;
      }
      $last =& $frg;
      break;
      
      case CSSJ_CTIP_RES_INSERT:
      $anchorId = $next['anchorId'];
      $id = count($frgs);
      $anchor =& $frgs[$anchorId];
      $frg =& _cssj_new_frg($id);
      $frgs[$id] =& $frg;
      $frg['prev'] =& $anchor['prev'];
      $frg['next'] =& $anchor;
      $anchor['prev']['next'] =& $frg;
      $anchor['prev'] =& $frg;
      if ($first['id'] === $anchor['id']) {
        $first =& $frg;
      }
      break;
      
      case CSSJ_CTIP_RES_ERROR:
      if (isset($errorFunc)) {
        $errorFunc($next['level'], $next['error']);
      }
      break;
      
      case CSSJ_CTIP_RES_DATA:
      if (isset($progressFunc)) {
        $progressFunc($next['progress']);
      }
      $frg =& $frgs[$next['id']];
      if (($length += _cssj_write_frg($tempFile, $onMemory, $segment, $frg, $next['bytes'])) === false) {
        fclose($tempFile);
        return false;
      }
      break;
    }
    
    $builder['first'] =& $first;
    $builder['last'] =& $last;
    return true;
  }
  else {
    //構築を完了してデータを送り出します
    if ($out === null) {
      header("Content-Length: $length");
    }
    
    $frg =& $first;
    while (!is_null($frg)) {
      if (_cssj_flush_frg($tempFile, $frg, $out) === false) {
        fclose($tempFile);
        return false;
      }
      $frg =& $frg['next'];
    }
    
    fclose($tempFile);
    return null;
  }
}

/**
 * フラグメントを作成します。
 * 
 * @param $id int フラグメントID
 * @return array フラグメント
 * @access private
 */
function &_cssj_new_frg($id) {
  $frg = array(
    'id' => $id,
    'prev' => null,
    'next' => null,
    'length' => 0,
    'buffer' => ''
  );
  return $frg;
}

/**
 * フラグメントにデータを書き込みます。
 * 
 * @param $tempFile resource 一時ファイル
 * @param $onMemory int メモリ上に置かれたデータの合計サイズ
 * @param $segment int セグメント番号シーケンス
 * @param $frg array フラグメント
 * @param $bytes string データ
 * @return mixed 成功なら書き込んだバイト数,失敗ならfalse
 * @access private
 */
function _cssj_write_frg(&$tempFile, &$onMemory, &$segment, &$frg, &$bytes) {
  $len = strlen($bytes);
  if (!isset($frg['segments']) &&
           ($frg['length'] + $len) <= CSSJ_CTIP_FRG_MEM_SIZE &&
           ($onMemory + $len) <= CSSJ_CTIP_ON_MEMORY) {
    $frg['buffer'] .= $bytes;
    $onMemory += $len;
  }
  else {
    if (isset($frg['buffer'])) {
      if (($wlen = _cssj_raf_write($tempFile, $segment, $frg, $frg['buffer'])) === false) {
          return false;
      }
      $onMemory -= $wlen;
      unset($frg['buffer']);
    }
    if (($len = _cssj_raf_write($tempFile, $segment, $frg, $bytes)) === false) {
      return false;
    }
  }
  $frg['length'] += $len;
  return $len;
}

/**
 * 一時ファイルに書き込みます。
 *
 * @param $tempFile resource 一時ファイル
 * @param $segment int セグメント番号シーケンス
 * @param $frg array フラグメント
 * @param $bytes string データ
 * @return mixed 書き込んだバイト数。失敗した場合はfalse。
 * @access private
 */
function _cssj_raf_write(&$tempFile, &$segment, &$frg, $bytes) {
  if (!isset($frg['segments'])) {
    $frg['segments'] = array($segment++);
    $frg['segLen'] = 0;
  }
  $segments =& $frg['segments'];
  $segLen =& $frg['segLen'];
  $written = 0;
  while (($len = strlen($bytes)) > 0) {
	if ($segLen == CSSJ_CTIP_SEGMENT_SIZE) {
		$segments[] = $segment++;
		$segLen = 0;
	}
	$seg = $segments[count($segments) - 1];
	$wlen = min($len, CSSJ_CTIP_SEGMENT_SIZE - $segLen);
	$wpos = $seg * CSSJ_CTIP_SEGMENT_SIZE + $segLen;
	if (fseek($tempFile, $wpos) === -1) {
	  return false;
	}
	if (($wlen = fwrite($tempFile, $bytes, $wlen)) === false) {
	  return false;
	}
	$segLen += $wlen;
	$written += $wlen;
	$bytes = substr($bytes, $wlen);
  }
  return $written;
}

/**
 * フラグメントの内容を吐き出して、フラグメントを破棄します。
 * 
 * @param $tempFile resource 一時ファイル
 * @param $frg array フラグメント
 * @param $out mixed 出力先ストリーム(resource),出力先変数(string),または標準出力であればnull。
 * @return boolean 成功ならtrue,失敗ならfalse
 * @access private
 */
function _cssj_flush_frg(&$tempFile, &$frg, &$out) {
  if (!isset($frg['segments'])) {
    if (_cssj_output($out, $frg['buffer']) === false) {
      return false;
    }
    unset($frg['buffer']);
  }
  else {
    $segments =& $frg['segments'];
    $segcount = count($segments);
    for ($i = 0; $i < $segcount - 1; ++$i) {
      $seg = $segments[$i];
      $rpos = $seg * CSSJ_CTIP_SEGMENT_SIZE;
      if (fseek($tempFile, $rpos) == -1) {
        return false;
      }
      if (($buff = _cssj_read($tempFile, CSSJ_CTIP_SEGMENT_SIZE)) === false) {
        return false;
      }
      if (_cssj_output($out, $buff) === false) {
        return false;
      }
    }
    $seg = $segments[$segcount - 1];
    $rpos = $seg * CSSJ_CTIP_SEGMENT_SIZE;
    if (fseek($tempFile, $rpos) == -1) {
      return false;
    }
    if (($buff = _cssj_read($tempFile, $frg['segLen'])) === false) {
      return false;
    }
    if (_cssj_output($out, $buff) === false) {
      return false;
    }
  }
  return true;
}

/**
 * 標準出力、ストリーム、変数のいずれかに出力します。
 *
 * @param $out mixed 出力先
 * @param $buff string データ
 * @return boolean 成功ならtrue,失敗ならfalse
 * @access private
 */
function _cssj_output(&$out, &$buff) {
  if ($out === null) {
    //標準出力
    echo $buff;
  }
  elseif (is_resource($out)) {
    //ストリーム
    if (fwrite($out, $buff) === false) {
      return false;
    }
  }
  else {
    //変数
    $out .= $buff;
  }
  return true;
}

?>