package com.ozacc.mail.impl;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.util.Iterator;
import java.util.Map;

import javax.activation.DataHandler;
import javax.activation.DataSource;
import javax.activation.FileDataSource;
import javax.mail.MessagingException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.mail.internet.MimePart;
import javax.mail.internet.MimeUtility;

import com.ozacc.mail.Mail;
import com.ozacc.mail.MultipartMail;

/**
 * MimeMessage󥹥󥹤륯饹
 * 
 * @since 1.0
 * @author Tomohiro Otsuka
 * @version $Id: MimeMessageBuilder.java,v 1.8 2004/09/15 09:03:36 otsuka Exp $
 */
public class MimeMessageBuilder {

	private MimeMessage mimeMessage;

	private String charset = Mail.JIS_CHARSET;

	private boolean hasRecipient = false;

	private MimeMultipart mimeMultipart;

	/**
	 * 󥹥ȥ饯
	 * ǥեȤʸ ISO-2022-JP 󥳡ǥ󥰤˻Ѥޤ
	 * 
	 * @param mimeMessage
	 */
	public MimeMessageBuilder(MimeMessage mimeMessage) {
		this.mimeMessage = mimeMessage;
	}

	/**
	 * 󥹥ȥ饯
	 * ʸ̾Υ󥳡ǥ󥰤˻Ѥʸɤꤷޤ
	 * 
	 * @param mimeMessage
	 * @param charset 󥳡ǥ󥰤˻Ѥʸ
	 */
	public MimeMessageBuilder(MimeMessage mimeMessage, String charset) {
		this.mimeMessage = mimeMessage;
		this.charset = charset;
	}

	/**
	 * 󥹥ȥ饯ΰϤ줿MimeMessage򤽤Τޤ֤ޤ
	 * 
	 * @return MimeMessage
	 */
	public MimeMessage getMimeMessage() {
		return this.mimeMessage;
	}

	/**
	 * ꤵ줿᡼뤫MimeMessageޤ
	 * 
	 * @param mail MimeMessageΥȤʤMail
	 * @throws MessagingException
	 * @throws UnsupportedEncodingException
	 */
	public void buildMimeMessage(Mail mail) throws UnsupportedEncodingException, MessagingException {

		setTo(mail);

		setCc(mail);

		setBcc(mail);

		// λ꤬ʤ票顼
		if (!hasRecipient) {
			throw new MessagingException("λ꤬ޤToCcBccΤ줫Ĥϻꤹɬפޤ");
		}

		setFrom(mail);

		setSubject(mail);

		setReplyTo(mail);

		setXHeaders(mail);

		setImportance(mail);

		if (mail instanceof MultipartMail) {
			MultipartMail mm = (MultipartMail)mail;

			if (mm.getAttachmentFiles().length == 0 && mm.isHtmlMail()) { // Plain, HTML

				MimeMultipart textAndHtmlMultipart = new MimeMultipart("alternative");
				setPlainText(mm, textAndHtmlMultipart);
				setHtmlText(mm, textAndHtmlMultipart);
				mimeMessage.setContent(textAndHtmlMultipart);

			} else if (mm.getAttachmentFiles().length > 0 && mm.isHtmlMail()) { // Plain, HMTL, File

				MimeMultipart textAndHtmlMultipart = new MimeMultipart("alternative");
				setPlainText(mm, textAndHtmlMultipart);
				setHtmlText(mm, textAndHtmlMultipart);

				MimeMultipart containingMultipart = new MimeMultipart();
				MimeBodyPart textBodyPart = createMimeBodyPart(containingMultipart);
				textBodyPart.setContent(textAndHtmlMultipart);
				setAttachmentFiles(mm, containingMultipart);

				mimeMessage.setContent(containingMultipart);

			} else { // Plain, File

				MimeMultipart mimeMultipart = new MimeMultipart();
				setPlainText(mm, mimeMultipart);
				setAttachmentFiles(mm, mimeMultipart);
				mimeMessage.setContent(mimeMultipart);

			}

		} else {
			setText(mail.getText(), this.mimeMessage);
		}

	}

	/**
	 * 
	 * @since 1.1
	 * 
	 * @param mm
	 * @param mimeMultipart
	 * @throws MessagingException
	 * @throws UnsupportedEncodingException 
	 */
	private void setAttachmentFiles(MultipartMail mm, MimeMultipart mimeMultipart)
																					throws MessagingException,
																					UnsupportedEncodingException {
		MultipartMail.AttachmentFile[] files = mm.getAttachmentFiles();
		for (int i = 0; i < files.length; i++) {
			MimeBodyPart bodyPart = createMimeBodyPart(mimeMultipart);
			MultipartMail.AttachmentFile attachmentFile = files[i];
			addAttachment(attachmentFile.getName(), new FileDataSource(attachmentFile.getFile()),
					bodyPart);
		}
	}

	/**
	 * 
	 * @since 1.1
	 * 
	 * @param mm
	 * @param mimeMultipart
	 * @throws MessagingException 
	 */
	private void setHtmlText(MultipartMail mm, MimeMultipart mimeMultipart)
																			throws MessagingException {
		if (mm.isHtmlMail()) {
			MimeBodyPart bodyPart = createMimeBodyPart(mimeMultipart);
			setHtmlText(mm.getHtmlText(), bodyPart);
		}
	}

	/**
	 * 
	 * @since 1.1
	 * 
	 * @param mm
	 * @param mimeMultipart
	 * @throws MessagingException 
	 */
	private void setPlainText(MultipartMail mm, MimeMultipart mimeMultipart)
																			throws MessagingException {
		if (mm.getText() != null && mm.getText().length() > 0) {
			MimeBodyPart bodyPart = createMimeBodyPart(mimeMultipart);
			setText(mm.getText(), bodyPart);
		}
	}

	/**
	 * MimeBodyPart󥹥󥹤ꤵ줿MimeMultipartϿޤ
	 * 
	 * Υ᥽åɤϥޥѡȥ᡼ˤΤ߸ƤӽФȤǤޤ
	 * ץ졼ƥȥ᡼ˤϡmimeMulipartnullʤΤǡ
	 * NullPointerExceptionޤ
	 * 
	 * @since 1.1
	 * 
	 * @param mm
	 * @return 줿MimeBodyPart
	 * @throws MessagingException 
	 */
	private MimeBodyPart createMimeBodyPart(MimeMultipart mm) throws MessagingException {
		MimeBodyPart bodyPart = new MimeBodyPart();
		mm.addBodyPart(bodyPart);
		return bodyPart;
	}

	/**
	 * @since 1.1
	 * 
	 * @param htmlText
	 * @param bodyPart 
	 * @throws MessagingException
	 */
	private void setHtmlText(final String htmlText, MimeBodyPart bodyPart)
																			throws MessagingException {

		// ʬ Spring 1.0.x 饳ԡ
		bodyPart.setDataHandler(new DataHandler(new DataSource() {

			public InputStream getInputStream() throws IOException {
				return new ByteArrayInputStream(charset != null ? htmlText.getBytes(charset)
						: htmlText.getBytes());
			}

			public OutputStream getOutputStream() throws IOException {
				throw new UnsupportedOperationException("Read-only javax.activation.DataSource");
			}

			public String getContentType() {
				return charset == null ? "text/html" : "text/html; charset=" + charset;
			}

			public String getName() {
				return "text";
			}
		}));

		bodyPart.setHeader("Content-Transfer-Encoding", "7bit");
	}

	/**
	 * @param mail 
	 * @throws MessagingException
	 */
	private void setXHeaders(Mail mail) throws MessagingException {
		Map headers = mail.getXHeaders();
		if (headers == null) {
			return;
		}

		Iterator itr = headers.keySet().iterator();
		while (itr.hasNext()) {
			String key = (String)itr.next();
			String value = (String)headers.get(key);
			mimeMessage.setHeader(key, value);
		}
	}

	/**
	 * @param mail 
	 * @throws MessagingException
	 */
	private void setImportance(Mail mail) throws MessagingException {
		if (mail.getImportance() != null) {
			mimeMessage.setHeader("Importance", mail.getImportance());

			int level = 3;
			if (Mail.Importance.HIGH.equals(mail.getImportance())) {
				level = 1;
			} else if (Mail.Importance.LOW.equals(mail.getImportance())) {
				level = 5;
			}
			mimeMessage.setHeader("X-Priority", String.valueOf(level));
		}
	}

	/**
	 * @param mail 
	 * @throws MessagingException
	 */
	private void setReplyTo(Mail mail) throws MessagingException {
		if (mail.getReplyTo() != null) {
			mimeMessage.setReplyTo(new InternetAddress[] { mail.getReplyTo() });
		}
	}

	/**
	 * @param mail 
	 * @throws MessagingException
	 */
	private void setBcc(Mail mail) throws MessagingException {
		if (mail.getBcc().length > 0) {
			mimeMessage.setRecipients(MimeMessage.RecipientType.BCC, mail.getBcc());
			hasRecipient = true;
		}
	}

	/**
	 * @param mail 
	 * @throws MessagingException
	 */
	private void setCc(Mail mail) throws MessagingException {
		if (mail.getCc().length > 0) {
			mimeMessage.setRecipients(MimeMessage.RecipientType.CC, mail.getCc());
			hasRecipient = true;
		}
	}

	/**
	 * @param mail 
	 * @throws MessagingException
	 */
	private void setTo(Mail mail) throws MessagingException {
		if (mail.getTo().length > 0) {
			mimeMessage.setRecipients(MimeMessage.RecipientType.TO, mail.getTo());
			hasRecipient = true;
		}
	}

	/**
	 * ʸ򥻥åȡ
	 * <p>
	 * NOTE: ʸκǸ˲ԤʤMozillaϤΥ᡼顼ǺǽԤܸ줬ʸƤޤ١
	 * message.setTextΰǺǸ\nɲäƤ롣
	 * 
	 * @since 1.1
	 * 
	 * @param text ʸ
	 * @param mimePart ʸ򥻥åȤMimePart 
	 * @throws MessagingException
	 */
	private void setText(String text, MimePart mimePart) throws MessagingException {
		if (charset != null) {
			if (charset.equalsIgnoreCase(Mail.JIS_CHARSET)) {
				// Cp932饹ѤơŪJISѴ
				mimePart.setText(Cp932.toJIS(text) + "\n", charset);
			} else {
				mimePart.setText(text + "\n", charset);
			}
		} else {
			mimePart.setText(text);
		}
		mimePart.setHeader("Content-Transfer-Encoding", "7bit");
	}

	/**
	 * @param mail
	 * @throws MessagingException
	 * @throws UnsupportedEncodingException
	 */
	private void setSubject(Mail mail) throws UnsupportedEncodingException, MessagingException {
		if (charset != null) {
			if (charset.equalsIgnoreCase(Mail.JIS_CHARSET)) {
				String subject = Cp932.toJIS(mail.getSubject());
				mimeMessage.setSubject(MimeUtility.encodeText(subject, charset, "B"));
			} else {
				mimeMessage.setSubject(mail.getSubject(), charset);
			}
		} else {
			mimeMessage.setSubject(mail.getSubject());
		}
	}

	/**
	 * @param mail
	 * @throws MessagingException 
	 */
	private void setFrom(Mail mail) throws MessagingException {
		mimeMessage.setFrom(mail.getFrom());
	}

	/**
	 * źեեǡꤵ줿MimeBodyPart˥åȤޤ
	 * 
	 * @since 1.1
	 * 
	 * @param fileName
	 * @param dataSource
	 * @param mimeBodyPart եǡ򥻥åȤMimeBodyPart
	 * @throws UnsupportedEncodingException
	 * @throws MessagingException
	 */
	private void addAttachment(String fileName, DataSource dataSource, MimeBodyPart mimeBodyPart)
																									throws UnsupportedEncodingException,
																									MessagingException {
		if (charset != null) {
			// ե̾Υ󥳡
			mimeBodyPart.setFileName(MimeUtility.encodeText(fileName, charset, "B"));
		} else {
			mimeBodyPart.setFileName(fileName);
		}

		mimeBodyPart.setDataHandler(new DataHandler(dataSource));
	}
}