//
// Informa -- RSS Library for Java
// Copyright (c) 2002 by Niko Schmuck
//
// Niko Schmuck
// http://sourceforge.net/projects/informa
// mailto:niko_schmuck@users.sourceforge.net
//
// This library is free software.
//
// You may redistribute it and/or modify it under the terms of the GNU
// Lesser General Public License as published by the Free Software Foundation.
//
// Version 2.1 of the license should be included with this distribution in
// the file LICENSE. If the license is not included with this distribution,
// you may find a copy at the FSF web site at 'www.gnu.org' or 'www.fsf.org',
// or you may write to the Free Software Foundation, 675 Mass Ave, Cambridge,
// MA 02139 USA.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied waranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.
//

// $Id: RSS_2_0_Exporter.java,v 1.3.2.2 2004/12/05 05:35:38 otsuka Exp $

package com.ozacc.blog.rss.informa;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Iterator;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.Namespace;
import org.jdom.output.DOMOutputter;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

import com.ozacc.blog.util.DateFormat;

import de.nava.informa.core.ChannelExporterIF;
import de.nava.informa.core.ChannelIF;
import de.nava.informa.core.ItemIF;
import de.nava.informa.utils.ParserUtils;

/**
 * InformaRSS_2_0_Exporterozacc-blogѤ˲¤饹
 * <p>
 * A channel exporter that can write channel objects out into the interchange
 * syntax defined by RSS 2.0.
 * 
 * @author Sami Alireza
 * 
 * @since 1.0
 */
public class RSS_2_0_Exporter implements ChannelExporterIF {

	private static final String RSS_VERSION = "2.0";

	/** RSS 1.0 Dublin Core namespace */
	private static final String NS_DC = "http://purl.org/dc/elements/1.1/";

	/** RSS 1.0 Syndication Module namespace */
	private static final String NS_SY = "http://purl.org/rss/1.0/modules/syndication/";

	private static final String NS_ADMIN = "http://webns.net/mvcb/";

	private static final String NS_RDF = "http://www.w3.org/1999/02/22-rdf-syntax-ns#";

	private static SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ");

	private Writer writer;

	private String encoding;

	/**
	 * Creates a channel exporter bound to the file given in the argument. The
	 * channel will be written out in the UTF-8 encoding.
	 * 
	 * @param filename - The name of the file to which the channel object is to be
	 *                   written.
	 */
	public RSS_2_0_Exporter(String filename) throws IOException {
		this(new File(filename), "utf-8");
	}

	/**
	 * Creates a channel exporter bound to the file given in the argument. The
	 * channel will be written out in the UTF-8 encoding.
	 * 
	 * @param file -  The file object to which the channel object is to be
	 *                written.
	 */
	public RSS_2_0_Exporter(File file) throws IOException {
		this(file, "utf-8");
	}

	/**
	 * Creates a channel exporter bound to the file given in the arguments.
	 * 
	 * @param file -  The file object to which the channel object is to be
	 *                written.
	 * @param encoding -
	 *                The character encoding to write the channel object in.
	 */
	public RSS_2_0_Exporter(File file, String encoding) throws IOException {
		this.writer = new OutputStreamWriter(new FileOutputStream(file), encoding);
		this.encoding = encoding;
	}

	/**
	 * Creates a channel exporter bound to the Writer given in the arguments.
	 * 
	 * @param writer -
	 *                The Writer to which the channel object is to be written.
	 * @param encoding -
	 *                The character encoding the Writer writes in.
	 */
	public RSS_2_0_Exporter(Writer writer, String encoding) {
		this.writer = writer;
		this.encoding = encoding;
	}

	// ------------------------------------------------------------
	// implementation of ChannelExporterIF interface
	// ------------------------------------------------------------

	public void write(ChannelIF channel) throws IOException {
		if (writer == null) {
			throw new RuntimeException("No writer has been initialized.");
		}

		// create XML outputter with indent: 2 spaces, print new lines.
		XMLOutputter outputter = new XMLOutputter(Format.getPrettyFormat());
		Document doc = getJdomDocument(channel);
		outputter.output(doc, writer);
		// ---
		writer.close();
	}

	public static Document getJdomDocument(ChannelIF channel) {
		Namespace dcNs = Namespace.getNamespace("dc", NS_DC);
		Namespace syNs = Namespace.getNamespace("sy", NS_SY);
		Namespace adminNs = Namespace.getNamespace("admin", NS_ADMIN);
		Namespace rdfNs = Namespace.getNamespace("rdf", NS_RDF);

		Element rootElem = new Element("rss");
		rootElem.addNamespaceDeclaration(dcNs);
		rootElem.addNamespaceDeclaration(syNs);
		rootElem.addNamespaceDeclaration(adminNs);
		rootElem.setAttribute("version", RSS_VERSION);

		Element channelElem = new Element("channel");
		// rootElem.setAttribute("version");
		channelElem.addContent(new Element("title").setText(channel.getTitle()));
		if (channel.getSite() != null) {
			channelElem.addContent(new Element("link").setText(channel.getSite().toString()));
		}

		channelElem.addContent(new Element("description").setText(channel.getDescription()));
		if (channel.getLanguage() != null) {
			channelElem.addContent(new Element("language", dcNs).setText(channel.getLanguage()));
		}
		if (channel.getCopyright() != null) {
			channelElem.addContent(new Element("copyright", dcNs).setText(channel.getCopyright()));
		}
		if (channel.getCreator() != null) {
			channelElem.addContent(new Element("creator", dcNs).setText(channel.getCreator()));
		}
		if (channel.getLastUpdated() != null) {
			channelElem.addContent(new Element("date", dcNs).setText(DateFormat.format(channel
					.getLastUpdated())));
		}

		if (channel.getUpdateBase() != null) {
			channelElem.addContent(new Element("updateBase", syNs).setText(df.format(channel
					.getUpdateBase())));
		}
		if (channel.getUpdatePeriod() != null) {
			// don't put out frequency without specifying period
			channelElem.addContent(new Element("updateFrequency", syNs).setText((new Integer(
					channel.getUpdateFrequency())).toString()));
			channelElem.addContent(new Element("updatePeriod", syNs).setText(channel
					.getUpdatePeriod()));
		}

		Collection items = channel.getItems();
		Iterator it = items.iterator();
		while (it.hasNext()) {
			ItemIF item = (ItemIF)it.next();
			Element itemElem = new Element("item");
			itemElem.addContent(new Element("title").setText(item.getTitle()));
			if (item.getLink() != null) {
				itemElem.addContent(new Element("link").setText(item.getLink().toString()));
			}
			if (item.getDescription() != null) {
				itemElem.addContent(new Element("description").setText(item.getDescription()));
			}
			if (item.getDate() != null) {
				itemElem.addContent(new Element("pubDate").setText(ParserUtils.formatDate(item
						.getDate())));
			}
			if (item.getSubject() != null) {
				itemElem.addContent(new Element("subject", dcNs).setText(item.getSubject()));
			}
			if (item.getLink() != null) {
				Element guid = new Element("guid").setText(item.getLink().toString());
				guid.setAttribute("isPermaLink", "true");
				itemElem.addContent(guid);
			}
			if (item.getComments() != null) {
				itemElem.addContent(new Element("comments").setText(item.getComments().toString()));
			}
			channelElem.addContent(itemElem);
		}

		rootElem.addContent(channelElem);

		return new Document(rootElem);
	}

	public static org.w3c.dom.Document getDocument(ChannelIF channel) throws JDOMException {
		Document doc = getJdomDocument(channel);
		DOMOutputter outputter = new DOMOutputter();
		return outputter.output(doc);
	}
}