/*

 RasterSource.java
 
 Copyright 2004 KUBO Hiroya (hiroya@sfc.keio.ac.jp).
 
 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at
 
 http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 
 Created on 2004/07/03

 */
package net.sf.sqs_xml.reader.logic;

/**
 * @author hiroya
 *  
 */
import java.awt.Rectangle;
import java.util.LinkedList;
import java.awt.Point;
import java.awt.geom.Point2D;

import net.sf.sqs_xml.image.RasterSource;

public class ReaderSource extends RasterSource {
    static final float WIDTH = (float)595.0;
    static final float HEIGHT = (float)841.0;
    static final float TOP_GUIDEBLOCK_WIDTH = (float) (20/ WIDTH);
    static final float BOTTOM_GUIDEBLOCK_WIDTH = (float) (10/ HEIGHT);
    static final int SCAN_UNIT_WIDTH_DIV = 2;
    final int blockSearchLengthMax;
    final int sideMargin;
    Point2D[] scanFrameCorner = null;
    public ReaderSource(java.awt.image.Raster raster) {
        super(raster);
        blockSearchLengthMax = getGuideBlockWidth(true)*3;
        sideMargin  = getGuideBlockWidth(true)*2;
    }
    
    int getGuideBlockWidth(boolean isPageTop) {
        if (isPageTop) {
            return (int) (getWidth() * TOP_GUIDEBLOCK_WIDTH);
        } else {
            return (int) (getWidth() * BOTTOM_GUIDEBLOCK_WIDTH);
        }
    }
    
    LinkedList scanBlocksHorizontal(int y, boolean isTop) {
        LinkedList candidate = new LinkedList();
        int blockWidth = getGuideBlockWidth(isTop);
        boolean[] buf = new boolean[blockWidth/SCAN_UNIT_WIDTH_DIV];
        int offset = 0;
        int numberOfWhitePixels = 0;
        int numberOfFlips = 0;
        Rectangle currentRectangle = null;

        for (int x = sideMargin; x < getWidth()-sideMargin; x++) {
            boolean previousIsWhite = buf[offset];
            boolean currentIsWhite = !isBlack(x, y);
            buf[offset] = currentIsWhite;
            offset++;
            if (offset == blockWidth/SCAN_UNIT_WIDTH_DIV) {
                offset = 0;
            }
            if (previousIsWhite) {
                numberOfWhitePixels--;
            }
            if (currentIsWhite) {
                numberOfWhitePixels++;
            }

            double whiteRatio = SCAN_UNIT_WIDTH_DIV * ((double) numberOfWhitePixels) / blockWidth;
            if (numberOfFlips % 2 == 0) {
                if (0.9 < whiteRatio) {
                    numberOfFlips++;
                    if (currentRectangle != null) {
                        try{
                            int w = (x - currentRectangle.x - buf.length) / 2;
                            Point center = new Point(currentRectangle.x + w, currentRectangle.y);
                            int upperBlackLength = scanBlackLineTop(center.x, center.y, true, blockSearchLengthMax);
                            int lowerBlackLength = scanBlackLineBottom(center.x, center.y, true, blockSearchLengthMax);
                            currentRectangle.y = y - upperBlackLength;
                            currentRectangle.height = upperBlackLength + lowerBlackLength;
                            currentRectangle.width = scanBlackLineLeft(center.x, center.y,  true, blockSearchLengthMax)+
                            						 scanBlackLineRight(center.x, center.y, true, blockSearchLengthMax);
                        }catch(ReaderSourceException ex){
                            candidate.removeLast();
                            currentRectangle = null;
                        }
                    }
                }else{
                }
            } else {
                if (whiteRatio < 0.1) {
                    numberOfFlips++;
                    currentRectangle = new Rectangle(x, y, 0, 0);
                    candidate.add(currentRectangle);
                } else {
                }
            }
        }
        //System.err.println("["+y+"] "+candidate.size());
        LinkedList ret = createValidBlockList(isTop, candidate);
        candidate.clear();
        return ret;
    }

    /**
     * @param isTop
     * @param candidate
     * @param ret
     */
    private LinkedList createValidBlockList(boolean isTop, LinkedList candidate) {
        LinkedList ret = new LinkedList();
        for(int i=0; i < candidate.size(); i++){
            Rectangle rect = (Rectangle)candidate.get(i);
            if (isValidBlock(rect, isTop)) {
                ret.add(rect);
            }else{/*
                System.err.println(" NG:"+rect+" "+
                        (Math.min(rect.width, rect.height)+":"+
                        (getWidth() * TOP_GUIDEBLOCK_WIDTH * 0.2 < Math.min(rect.width, rect.height))
                         +":"+
                        (Math.max(rect.width, rect.height)+":"+
                        (Math.max(rect.width, rect.height) < getWidth() * TOP_GUIDEBLOCK_WIDTH * 1.3))));
                        */
                
            }
        }
        return ret;
    }

    private boolean isValidBlock(Rectangle rect, boolean isTop){
        int modifiedWidth;
        if(isTop){
            modifiedWidth = rect.width;
        }else{
            modifiedWidth = 2 * rect.width;
        }
        if(rect.height == 0){
            return false;
        }
        double widthHeightRatio = 1.0 * modifiedWidth / rect.height;
        int topWidth = getGuideBlockWidth(true);
        return (( topWidth * 0.6)  < Math.min(modifiedWidth, rect.height)) &&
                (Math.max(modifiedWidth, rect.height) < (topWidth * 1.3)) &&
                0.8 < widthHeightRatio && widthHeightRatio < 1.3;
    }
    
    private int[] scanGuideBorder(int topHint, int bottomHint, boolean isTop) throws ReaderSourceException{
        int startYValue = 0, endYValue = 0;
        LinkedList maxRectangleList = new LinkedList();
        for (int y = topHint; y < bottomHint; y += 1) {
            LinkedList rectangleList = scanBlocksHorizontal(y, isTop);
            //System.err.println("y="+y+" "+rectangleListOfBlocks.size());
            if (maxRectangleList.size() < rectangleList.size()) {
                startYValue = y;
                endYValue = y;
                maxRectangleList = rectangleList;
            } else if (maxRectangleList.size() == rectangleList.size()) {
                endYValue = y;
            }
            //rectangleListOfBlocks.clear();
        }
        //System.err.println(topHint+":"+bottomHint+":"+maxRectangleListOfBlocks.size());
        if(11 <= maxRectangleList.size()){
            Rectangle r1 = (Rectangle) (maxRectangleList.getFirst());
            Rectangle r2 = (Rectangle) (maxRectangleList.getLast());
            maxRectangleList.clear();
            return new int[] { startYValue, endYValue, r1.x, r2.x };
        }else{
            //System.err.println(blockWidth);
            for(int i=0; i<maxRectangleList.size(); i++){
                System.err.println("*** "+maxRectangleList.get(i));
            }
            throw new ReaderSourceException("(found "+maxRectangleList.size()+" in "+startYValue+"-"+endYValue+")");
        }
    }

    Point2D getGuideBlockCenter(Rectangle r) throws ReaderSourceException{
        return getGuideBlockCenter(r.x + r.width / 2, r.y + r.height / 2);
    }

    Point2D getGuideBlockCenter(int ox, int oy) throws ReaderSourceException{
        int max = getWidth()/20;
        //System.err.print("* "+ox+","+oy+ " ");
        java.util.Set center = new java.util.HashSet();
        while (true) {
            int tx1 = scanBlackLineLeft(ox, oy, true, max);
            int tx2 = scanBlackLineRight(ox, oy, true, max);
            int ty1 = scanBlackLineTop(ox, oy, true, max);
            int ty2 = scanBlackLineBottom(ox, oy, true, max);
            oy =  oy - (ty1-ty2)/2;
            ox =  ox - (tx1-tx2)/2;
            //System.err.print("->("+tx1+","+tx2+"/"+ty1+","+ty2+")- "+ox+","+oy+" ");
            Point2D current = new Point2D.Double(ox, oy);
            if (center.contains(current)) {
                //System.err.println();
                return current;
            }
            center.add(current);
        }
    }

    int scanBlackLineTop(int x, int y, boolean isBlack, int length)
            throws ReaderSourceException {
        Point p = new Point();
        for (int i = 0; i < length; i++) {
            if (isBlack != isBlack(this.getRGBColor(x, y - i))) {
                return i;
            }
        }
        throw new ReaderSourceException("cannot reach edge");
    }

    int scanBlackLineBottom(int x, int y, boolean isBlack, int length)
            throws ReaderSourceException {
        for (int i = 0; i < length; i++) {
            if (isBlack != isBlack(this.getRGBColor(x, y + i))) {
                return i;
            }
        }
        throw new ReaderSourceException("cannot reach edge");
    }

    int scanBlackLineLeft(int x, int y, boolean isBlack, int length)
            throws ReaderSourceException {
        for (int i = 0; i < length; i++) {
            if (isBlack != isBlack(this.getRGBColor(x - i, y))) {
                return i;
            }
        }
        throw new ReaderSourceException("cannot reach edge");
    }

    int scanBlackLineRight(int x, int y, boolean isBlack, int length)
            throws ReaderSourceException {
        for (int i = 0; i < length; i++) {
            if (isBlack != isBlack(this.getRGBColor(x + i, y))) {
                return i;
            }
        }
        throw new ReaderSourceException("cannot reach edge");
    }

    Point2D[] createScanFrameCorner() throws ReaderSourceException{
        int[] top, bottom;
        try{
            top = scanGuideBorder(0, getHeight() / 8, true);
        }catch(ReaderSourceException ex){
            throw new ReaderSourceException("上側の「■」の列を正常に認識できません");//+ex.getMessage()
        }
        try{
            bottom = scanGuideBorder(getHeight() - getHeight() / 8,
                    getHeight() - 1, false);
        }catch(ReaderSourceException ex){
            //System.out.println("width:"+getGuideBlockWidth(true));
            throw new ReaderSourceException("下側の「■」の列を正常に認識できません");//+ex.getMessage()
        }
       
        Point2D[] p = new Point2D[4];
        //System.err.println("TOP: "+top[0]+", "+top[1]+", "+top[2]);
        //System.err.println("BOT: "+bottom[0]+", "+bottom[1]+", "+bottom[2]);
        p[0] = getGuideBlockCenter(top[2], (top[0] + top[1]) / 2);
        p[1] = getGuideBlockCenter(top[3], (top[0] + top[1]) / 2);
        p[2] = getGuideBlockCenter(bottom[2], (bottom[0] + bottom[1]) / 2);
        p[3] = getGuideBlockCenter(bottom[3], (bottom[0] + bottom[1]) / 2);
        this.scanFrameCorner = p; 
        return p;
    }

    boolean checkGuideSquareShape(Point2D[] guide){
        double gx1 = (guide[0].getX() - guide[1].getX());
        double gx2 = (guide[0].getX() - guide[2].getX());
        double gy1 = (guide[0].getY() - guide[1].getY());
        double gy2 = (guide[0].getY() - guide[2].getY());
        double gl1 = Math.sqrt(gx1*gx1 + gy1*gy1);
        double gl2 = Math.sqrt(gx2*gx2 + gy2*gy2);
        
        double sx1 = (scanFrameCorner[0].getX() - scanFrameCorner[1].getX());
        double sx2 = (scanFrameCorner[0].getX() - scanFrameCorner[2].getX());
        double sy1 = (scanFrameCorner[0].getY() - scanFrameCorner[1].getY());
        double sy2 = (scanFrameCorner[0].getY() - scanFrameCorner[2].getY());
        double sl1 = Math.sqrt(sx1*sx1 + sy1*sy1);
        double sl2 = Math.sqrt(sx2*sx2 + sy2*sy2);

        return (Math.abs(sl1/gl1 - sl2/gl2) < 0.02 * (sy2/gy2) && 
                Math.abs(sl1/sl2 - gl1/gl2) < 0.02 * (sy2/gy2));
    }
}
