/*
 * 
 * MarkReader.java
 * 
 * Copyright 2004 KUBO Hiroya (hiroya@sfc.keio.ac.jp).
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy
 * of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package net.sf.sqs_xml.reader.swing;

import java.awt.Cursor;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.io.File;
import java.io.IOException;

import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.UIManager;
import javax.swing.WindowConstants;
import javax.xml.parsers.ParserConfigurationException;

import net.sf.sqs_xml.editor.sqs.swing.SQSFileChooser;
import net.sf.sqs_xml.httpd.MarkReaderHttpd;
import net.sf.sqs_xml.reader.logic.MarkReaderTranslator;
import net.sf.sqs_xml.swing.IconFactory;
import net.sf.sqs_xml.swing.LoggerConsoleFrame;
import net.sf.sqs_xml.swing.SwingUtil;
import net.sf.sqs_xml.translator.logic.Translator;
import net.sf.sqs_xml.translator.logic.TranslatorException;
import net.sf.sqs_xml.translator.logic.TranslatorResource;
import net.sf.sqs_xml.translator.swing.SQSFileTranslator;

import org.xml.sax.SAXException;

public class MarkReader extends SQSFileTranslator {
    public static final String BASEURI = MarkReaderHttpd.getSingleton().getBase();
	public static final String TITLE = "SQS MarkReader(2004/11/10)";
	
	public static void main(final String args[]) throws Exception {
	    System.setProperty("swing.plaf.metal.controlFont","Monospaced-12");
        UIManager.setLookAndFeel(UIManager.getCrossPlatformLookAndFeelClassName());
		//UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		SwingUtil.Factory factory = new SwingUtil.Factory() {
			public Object create() throws Exception {
				return createReaderLogic();
			}
		};
		MarkReaderTranslator translator = (MarkReaderTranslator) SwingUtil.newInstanceWithWaitPromptFrame(factory, TITLE, "Initializing...");
		File dir = null;
		if(0 < args.length){
		  dir = new File(args[0]);
		  if(! dir.isDirectory()){
		      dir = null;
		  }
		}
		MarkReader self = new MarkReader(dir, translator);
		if(dir != null){
		    self.setSlectedFile(dir);
		    self.fileSaveButton.setEnabled(true);
		    self.fileSaveButton.setFocusable(true);
		}
	}

	/**
     * @param args
     * @return
     * @throws TranslatorException
     * @throws ParserConfigurationException
     * @throws SAXException
     * @throws IOException
     */
    static MarkReaderTranslator createReaderLogic() throws TranslatorException, ParserConfigurationException, SAXException, IOException {
        return new MarkReaderTranslator(){
            public void initializeEnvironment(){
                LoggerConsoleFrame logger = ((LoggerConsoleFrame) getLogger());                    
                logger.setDebugEnabled(false);
                JButton cancelButton = logger.getCancelButton();
                cancelButton.addActionListener(new ActionListener(){
                    public void actionPerformed(ActionEvent ev){
                        setSuspending();       
                    }
                });
                logger.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
                logger.addWindowListener(new WindowAdapter(){
                    public void windowClosed(java.awt.event.WindowEvent ev){
                        setSuspending();                    
                    }
                });
            }
        };
    }
    static{
        IconFactory.setBaseURI(MarkReaderHttpd.getSingleton().getBase());
    }
    
	public MarkReader(File dir, Translator translator) throws Exception {
		super(
				IconFactory.create("reader.gif"),
				TITLE,
				translator,
				SQSFileChooser.createOpenSQSFolderChooser(dir),
				"フォルダ(スキャン画像と印刷原稿PDFファイルを含むもの):",	
				null,
				"集計結果データ");
		setSize(400, 215);
		this.inputFileChooserPanel.getFileDropAcceptableTextField().setToolTipText("「スキャンした画像ファイル」と「調査票印刷原稿のPDFファイル」を含むフォルダをdrag&dropしてください");
		this.fileSaveButton.setToolTipText("読み取り処理を実行します");
		this.setVisible(true);
	}
	
	public JFileChooser createInputFileChooser(){
	    return SQSFileChooser.createOpenSQSSourceChooser(null); 
	}
	
	public JFileChooser createOutputFileChooser(File file){
	    return SQSFileChooser.createSaveReaderResultFolderChooser(file);
	}
	
    public LoggerConsoleFrame createLoggerConsoleFrame() {
        final LoggerConsoleFrame logger = new ProgressLoggerConsoleFrame(TITLE+" 進捗状況",  "キャンセル",  "完了");
        logger.getCancelButton().addActionListener(new ActionListener(){
        	public void actionPerformed(ActionEvent ev){
        		logger.setVisible(false);
        		logger.setFinished(true);
        		setFinished(true);
        	}
        });
        logger.setFinished(false);
        return logger;
    }

	public void save(final File sourceFile){
        if(! sourceFile.isDirectory()){
            JOptionPane.showMessageDialog(this, "フォルダを選択してください");
            return;
        }
        super.save(sourceFile);
	}
	
	public void translate(final File sourceFile, File dummy){	    
	    final File targetFile = selectTargetFile(sourceFile);
	    if(targetFile == null){
	        setFinished(true);
	        return;
	    }
	    final TranslatorResource resource = createResource(); 
	    new Thread(){
	        public void run(){
	            try {
	                ((MarkReaderTranslator)translator).translate(sourceFile, resource, targetFile);
	                ((LoggerConsoleFrame)resource.getLogger()).setFinished(true);
	            } catch (TranslatorException e) {
	                e.printStackTrace();
	                showError(((LoggerConsoleFrame)resource.getLogger()), e);
	                ((LoggerConsoleFrame)resource.getLogger()).setFinished(false);
	            }finally{
	                inputFileChooserPanel.setEnabled(true);
	                fileSaveButton.setEnabled(true);
	                setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
	            }
	        }
	        
	    	public void showError(JFrame logger, Exception e){
	    	    JOptionPane.showMessageDialog(logger, e.getLocalizedMessage(), "ERROR-TRANSLATE", JOptionPane.ERROR_MESSAGE);
	    	}

	    }.start();
	}

    private File selectTargetFile(final File sourceFile) {
        File parentdir = sourceFile.getParentFile();
        File tgtdir = new File(parentdir.getAbsoluteFile()+File.separator+sourceFile.getName()+"-RESULT");
        while(true){
            if(tgtdir.exists() && tgtdir.isDirectory()){
                int option = JOptionPane.showConfirmDialog(this, new String[]{"保存先フォルダが既に存在しています．",
                        "保存先:"+tgtdir,                        
                        "このフォルダの内容を上書きしますか?"}, 
                        "確認",JOptionPane.YES_NO_CANCEL_OPTION);
                if(JOptionPane.OK_OPTION == option){
                    tgtdir.delete();
                    tgtdir.mkdir();
                    break;
                }else if(JOptionPane.CANCEL_OPTION == option){
                    return null;
                }
            }else{
                int option = JOptionPane.showConfirmDialog(this, new String[]{"保存先フォルダが存在しません．",
                        "保存先:"+tgtdir,
                        "このフォルダを作成しますか?"}, "確認", JOptionPane.YES_NO_CANCEL_OPTION);
                if(JOptionPane.OK_OPTION == option){
                    tgtdir.mkdir();
                    break;
                }else if(JOptionPane.CANCEL_OPTION == option){
                    return null;
                }
            }
            String tgtName = JOptionPane.showInputDialog(this, new String[]{"保存先フォルダ名を指定してください．", "保存先:"+parentdir.getAbsoluteFile()+File.separator}, tgtdir.getName());
            if(tgtName != null){
                tgtdir = new File(parentdir.getAbsoluteFile()+File.separator+tgtName);
            }
        }
        return tgtdir;
    }
}

