/*
 * 

 RasterSource.java

 Copyright 2004 KUBO Hiroya (hiroya@sfc.keio.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */
package net.sf.sqs_xml.image;

import java.awt.Graphics;
import java.awt.geom.Point2D;
import java.awt.image.BufferedImage;
import java.awt.image.DataBuffer;
import java.awt.image.IndexColorModel;
import java.awt.image.Raster;
import java.awt.image.WritableRaster;

public class RasterSource {
	final RasterSource source;
	final Raster raster;

	int width = 0;
	int height = 0;

	int threshold = 200;
	
	public static final int BLACK = 0;
	public static final int[] BLACK_COLOR = {0, 0, 0};
	public static final int WHITE = ((0xff << 16) | 0xff << 8) | 0xff;//16777216;
	public static final int[] WHITE_COLOR = {255, 255, 255};
	
	public RasterSource() {
			this.source = null;
			this.raster = null;
	}

	public RasterSource(BufferedImage image) {
		this.width = image.getWidth();
		this.height = image.getHeight();
		this.source = null;
		this.raster = image.getData();
	}

	public RasterSource(Raster raster) {
		this.width = raster.getWidth();
		this.height = raster.getHeight();
		this.source = null;
		this.raster = raster;
	}

	public RasterSource(RasterSource source) {
		this.width = source.getWidth();
		this.height = source.getHeight();
		this.source = source;
		this.raster = null;
	}

	public void setThreshold(int threshold){
	    this.threshold = threshold;
	}
	
	public int getThreshold(){
	    return this.threshold;
	}
	
    public final int getWidth() {
		return width;
	}

	public final int getHeight() {
		return height;
	}

	/*
	int getAntialiasedPixel(double x, double y) {
		double xfloor = Math.floor(x);
		double yfloor = Math.floor(y);
		double xceil = Math.ceil(x);
		double yceil = Math.ceil(y);

		int p1 = getPixel((int) xfloor, (int) yfloor);
		int p2 = getPixel((int) xceil, (int) yfloor);
		int p3 = getPixel((int) xfloor, (int) yceil);
		int p4 = getPixel((int) xceil, (int) yceil);

		return (int)
				((p1 & 0xff) * (xceil - x) * (yceil - y)
					+ (p2 & 0xff) * (x - xfloor) * (yceil - y)
					+ (p3 & 0xff) * (xceil - x) * (y - yfloor)
					+ (p4 & 0xff) * (x - xfloor) * (y - yfloor));

		switch (getMode()) {
			case GRAYSCALE_MODE :
				return (int)
					((p1 & 0xff) * (xceil - x) * (yceil - y)
						+ (p2 & 0xff) * (x - xfloor) * (yceil - y)
						+ (p3 & 0xff) * (xceil - x) * (y - yfloor)
						+ (p4 & 0xff) * (x - xfloor) * (y - yfloor));
			case RGB_MODE :
			default :
				int ret = 0;
				for (int bitshift = 0; bitshift < 32; bitshift += 8) {
					ret
						|= (int) (((p1 << bitshift) & 0xff)
							* (xceil - x)
							* (yceil - y)
							+ ((p2 << bitshift) & 0xff)
								* (x - xfloor)
								* (yceil - y)
							+ ((p3 << bitshift) & 0xff)
								* (xceil - x)
								* (y - yfloor)
							+ ((p4 << bitshift) & 0xff)
								* (x - xfloor)
								* (y - yfloor))
						>> bitshift;

				}
				return ret;
		}

	}
		*/
		
	public final boolean isBlack(int src) {
	    if(raster != null){
	        if(getNumBands() == 1){
	            return src != BLACK;
	        }else{
	            int grayscale = (src&0xffff)>>8;
	            return grayscale < threshold;
	        }
	    }else{
	        return source.isBlack(src);
	    }
	}

/*
	public int getPixel(double x, double y) {
		if(raster != null){
			//return getAntialiasedPixel(x, y);
			return getPixel((int)x, (int)y);
		}else{
			return source.getPixel(x, y);
		}
	}
*/
	public boolean isBlack(int x, int y) {
		return isBlack(getRGBColor(x, y));
	}
	
	public int getRGBColor(Point2D p) {
	    return getRGBColor((int)p.getX(), (int)p.getY());
	}
	
	public int getRGBColor(int x, int y) {
		if (raster != null) {
			if(x < 0 || width <= x){
				return BLACK;
			}else{
				//System.out.println("("+x+","+y+")");
				try{
					//return raster.getDataBuffer().getElem(x + y * width);
					int[] argb = new int[4];
					raster.getPixel(x, y, argb);
					return argb[0]<<24 | argb[1]<<16 | argb[2]<<8 | argb[3];
				}catch(IndexOutOfBoundsException ex){
					return BLACK;
				}
			}
		}else{
			return source.getRGBColor(x, y);
		}
	}
	
	public Raster getRaster(){
	    if(raster != null){
	        return raster;
	    }else{
	        return source.getRaster();
	    }
	}

	public WritableRaster createCompatibleWritableRaster(int w, int h){
		if(raster != null){
		    if(getNumBands() == 1){
		        return raster.createCompatibleWritableRaster(w, h);
		    }else{
		        return WritableRaster.createPackedRaster(DataBuffer.TYPE_INT,
		                w, h,
		                new int[] {0x00ff0000, 0x0000ff00, 0x000000ff }, null);
		    }
		}else{
			return source.createCompatibleWritableRaster(w, h);
		}
	}
	
	public int getNumBands(){
	    if(source == null){
	        return raster.getNumBands();
	    }else{
	        return source.getNumBands();
	    }
	}
	
	public RasterSource getSource(){
	    if(source == null){
	        return this;
	    }else{
	        return source;
	    }
	}
	
	public static BufferedImage createImage(Raster src) {
	    return createImage(src, true);
	}
	
	public static BufferedImage createImage(Raster src, boolean isRGBMode) {
	    BufferedImage image = null;
	    if(src.getNumBands() == 1){
	        if(isRGBMode){
	            int[] _argb = new int[4];
	            int[] argb = new int[4];
	            image = new BufferedImage(src.getWidth(), src.getHeight(), BufferedImage.TYPE_INT_RGB);
	            WritableRaster tgt = WritableRaster.createPackedRaster(DataBuffer.TYPE_INT,
	                    src.getWidth(), src.getHeight(),
	                    new int[] {0x00ff0000, 0x0000ff00, 0x000000ff }, null);
	            Graphics g = image.getGraphics();
	            for(int j=0; j<src.getHeight(); j++){
	                for(int i=0; i<src.getWidth(); i++){
	                    src.getPixel(i, j, argb);
	                    _argb[0]=255-argb[0]*255;
	                    _argb[1]=255-argb[0]*255;
	                    _argb[2]=255-argb[0]*255;
	                    _argb[3]=255-argb[0]*255;
	                    tgt.setPixel(i, j, _argb);
	                }
	            }
	            image.setData(tgt);
	            return image;
	        }else{
	            byte[] indexColorMap = null;
	            indexColorMap = new byte[]{0x00, (byte)0xff};
	            final int bitParPixel = 1;  
	            image = new BufferedImage(src.getWidth(), src.getHeight(), BufferedImage.TYPE_BYTE_INDEXED,
	                    new IndexColorModel(bitParPixel, indexColorMap.length, indexColorMap, indexColorMap, indexColorMap));
	        }
	    }else{
	        image = new BufferedImage(src.getWidth(), src.getHeight(), BufferedImage.TYPE_INT_RGB);
	    }
		image.setData(src);
		return image;
	}
}
