/*

 ExportFileLogic.java
 
 Copyright 2004 KUBO Hiroya (hiroya@sfc.keio.ac.jp).
 
 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at
 
 http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 
 Created on 2004/08/18

 */
package net.sf.sqs_xml.editor.sqs.swing;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;

import javax.swing.JOptionPane;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactoryConfigurationError;

import net.sf.sqs_xml.editor.base.swing.SourceEditorMediator;
import net.sf.sqs_xml.editor.sqs.source.SQSSource;
import net.sf.sqs_xml.httpd.SourceEditorHttpd;
import net.sf.sqs_xml.swing.LoggerConsoleFrame;
import net.sf.sqs_xml.translator.logic.SQStoHTMLTranslator;
import net.sf.sqs_xml.translator.logic.SQStoPDFTranslator;
import net.sf.sqs_xml.translator.logic.SQStoPreviewTranslator;
import net.sf.sqs_xml.translator.logic.Translator;
import net.sf.sqs_xml.translator.logic.TranslatorException;
import net.sf.sqs_xml.translator.logic.TranslatorResource;
import net.sf.sqs_xml.util.BrowserLauncher;


public class ExportFileLogic{
    private SourceEditorMediator mediator;
    LoggerConsoleFrame logger;
    
    public ExportFileLogic(SourceEditorMediator mediator){
        this.mediator = mediator;
    }
    
    private LoggerConsoleFrame createLogger(){
        final LoggerConsoleFrame logger = new LoggerConsoleFrame(mediator.getTitle()+" Log",  "Cancel",  "OK");
        logger.setFinished(false);
		logger.getCancelButton().addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ev){
				logger.setVisible(false);
				logger.setFinished(true);
			}
		});	
        logger.info("start translation.");
        logger.setSize(400, 300);
        logger.setVisible(true);
		return logger;
    }
    
    public static final int USER_SELECTION_MODE = -1;
    public static final int HTML_MODE = 0;
    public static final int HTML_PREVIEW_MODE = 1;
    public static final int PDF_MODE = 2;
    
    public void export(final BrowserLauncher launcher, int mode)throws TranslatorException, IOException{
        File sourceFile = mediator.getCurrentTreePane().getSource().getFile();
        File tgtFile = null;
        String suffix, desc;
        if(mode == USER_SELECTION_MODE){
            tgtFile = createSelectedFile(sourceFile, ".pdf");
        }else{
            suffix = getSuffix(mode);
            String tmpdir = System.getProperty("java.io.tmpdir");
            if(tmpdir != null && isBrowserAccessibleDirectory(tmpdir)){
                tgtFile = createTemporaryFile(suffix);
            }else{
                tgtFile = createSelectedFile(sourceFile, suffix);
            }
        }
        if(tgtFile != null){
            mode = getMode(tgtFile);
            suffix = getSuffix(mode);
            desc = getDescription(mode);
            export(launcher, createTranslator(mode), tgtFile, suffix, desc);
        }
    }
    
    private Translator createTranslator(int mode)throws TranslatorException{
        if(mode == HTML_PREVIEW_MODE){
            return new SQStoPreviewTranslator();                
        }else if(mode == HTML_MODE){
            return new SQStoHTMLTranslator();
        }else if(mode == PDF_MODE){
            return new SQStoPDFTranslator();
        }else{
            throw new RuntimeException("export mode error");
        }
    }
    
    private int getMode(File file){
        if(file.getName().endsWith("-preview.html")){
            return HTML_PREVIEW_MODE;
        }else if(file.getName().endsWith(".html")){
            return HTML_MODE;
        }else if(file.getName().endsWith(".pdf")){
            return PDF_MODE;
        }else{
            throw new RuntimeException("export mode error");
        }
    }

    private String getSuffix(int mode){
        if(mode == HTML_MODE){
            return ".html";
        }else if(mode == HTML_PREVIEW_MODE ){
            return "-preview.html";
        }else if(mode == PDF_MODE){
            return ".pdf";
        }else{
            throw new RuntimeException("export mode error");
        }
    }
    
    private String getDescription(int mode){
        if(mode == HTML_MODE){
            return "HTML File";
        }else if(mode == HTML_PREVIEW_MODE ){
            return "Preview HTML File";
        }else if(mode == PDF_MODE){
            return "PDF File";
        }else{
            throw new RuntimeException("export mode error");
        }
    }
    
    private boolean isBrowserAccessibleDirectory(String tmpdir) throws UnsupportedEncodingException {
        return URLEncoder.encode(tmpdir, "UTF-8").equals(URLEncoder.encode(tmpdir, "MS932"));
    }

    public synchronized void export(final BrowserLauncher launcher, final Translator translator, final File tgtFile, final String suffix, final String desc) throws TransformerFactoryConfigurationError {
        new Thread() {
            public void run() {
                try {
                    InputStream sqsInputStream = createSQSInputStream();
                    OutputStream outputStream = new BufferedOutputStream(new FileOutputStream(tgtFile));
                    translate(translator, outputStream, sqsInputStream, SQSSourceEditorMediator.BASEURI);
                    if(launcher != null && isBrowserAccessibleDirectory(tgtFile.getAbsolutePath())){
                        launcher.showDocument(SourceEditorHttpd.getSingleton().getURL(tgtFile));
                    }
                } catch (IOException ex) {
                    ex.printStackTrace();
                    mediator.createSourceEditorMenuBarMediator().showError(ex, "エラー");
                } catch (TranslatorException ex) {
                    ex.printStackTrace();
                    mediator.createSourceEditorMenuBarMediator().showError(ex, "エラー");
                } catch (Exception ex) {
                    ex.printStackTrace();
                    mediator.createSourceEditorMenuBarMediator().showError(ex, "エラー");
                }
            }
        }.start();
    }
    
    private File createTemporaryFile(String suffix)throws IOException{       
        File tgtFile = File.createTempFile("sqs-tmp-", suffix);
        tgtFile.deleteOnExit();
        return tgtFile;
    }
    
    private File createSelectedFile(File sourceFile, String suffix){
        return mediator.getMenuBarMediator().exportFile(sourceFile, suffix);
    }

    private InputStream createSQSInputStream() throws IOException {
        return ((SQSSource)mediator.getSourceEditorTabbedPane().getCurrentEditingSource()).createInputStream();
    }

    private synchronized void translate(final Translator translator, final OutputStream tmpOutputStream, final InputStream sqsInputStream, String baseURI) throws TranslatorException {
        mediator.getToolBar().setEnabled(false);
        final LoggerConsoleFrame logger = createLogger();
        try {
            translator.translate(sqsInputStream,
                    new TranslatorResource(logger), tmpOutputStream, baseURI);
        } catch (TranslatorException ex) {
            if(ex.getCause() instanceof TransformerException){
                showTransformerException(logger, ex);
            }
            throw ex;
        } finally {
            tearOffTranslation(tmpOutputStream, sqsInputStream, logger);
        }
    }

    private void tearOffTranslation(final OutputStream tmpOutputStream, final InputStream sqsInputStream, final LoggerConsoleFrame logger) {
        mediator.getToolBar().setEnabled(true);
        logger.setFinished(true);
        if(! logger.hasError()){
            logger.setVisible(false);
        }
        try {
            sqsInputStream.close();
            tmpOutputStream.close();
        } catch (IOException ignore) {
        }
    }

    private void showTransformerException(final LoggerConsoleFrame logger, TranslatorException ex) {
        TransformerException cause = (TransformerException)ex.getCause();
        logger.error(cause.getMessageAndLocation());
        JOptionPane.showMessageDialog(mediator.getFrame(),
                new Object[]{"ファイル形式が正しくないため，変換処理に失敗しました:",
                "位置:"+cause.getLocationAsString(),
                "内容:"+cause.getLocalizedMessage()
        		},
                "File Error", JOptionPane.ERROR_MESSAGE);
    }

}