/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew.ui.window;

import static java.awt.event.InputEvent.SHIFT_DOWN_MASK;
import static java.awt.event.KeyEvent.*;
import static java.awt.event.MouseEvent.*;
import static javax.swing.KeyStroke.getKeyStroke;

import java.awt.*;
import java.awt.event.*;
import java.beans.*;
import java.util.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import javax.swing.text.*;

/**
 * ʃZbge[uB
 */
final class ResultSetTable extends JTable {

    private static final TableCellRenderer nullRenderer = new NullValueRenderer();

    private ColumnHeaderCellRenderer columnHeaderRenderer;
    private RowHeader rowHeader;

    /*
     * ZҏWJneditCellAt(int, int, EventObject)Ă΂ꂽꍇA
     * KeyEventߑłȂ߁AOKeyEventۑĎgB
     */
    volatile KeyEvent lastKeyEvent;

    /**
     * RXgN^B
     */
    ResultSetTable() {
        JTableHeader columnHeader = getTableHeader();
        TableCellRenderer columnHeaderDefaultRenderer = columnHeader.getDefaultRenderer();
        final RowHeader rowHeader = new RowHeader(this);
        this.columnHeaderRenderer = new ColumnHeaderCellRenderer(columnHeaderDefaultRenderer);
        this.rowHeader = rowHeader;
        setColumnSelectionAllowed(true);
        setAutoResizeMode(AUTO_RESIZE_OFF);
        columnHeader.setDefaultRenderer(columnHeaderRenderer);
        columnHeader.setReorderingAllowed(false);
        // [Cxg̐ݒ]
        // I
        MouseInputListener columnHeaderMouseInputListener = new MouseInputAdapter() {

            private int dragStartColumn;

            @Override
            public void mousePressed(MouseEvent e) {
                if (SwingUtilities.isLeftMouseButton(e)) {
                    changeSelection(e);
                }
            }

            @Override
            public void mouseDragged(MouseEvent e) {
                if (SwingUtilities.isLeftMouseButton(e)) {
                    changeSelection(e);
                }
            }

            private void changeSelection(MouseEvent e) {
                final Point p = e.getPoint();
                int id = e.getID();
                boolean isMousePressed = (id == MOUSE_PRESSED);
                boolean isMouseDragged = (id == MOUSE_DRAGGED);
                if (isMousePressed || isMouseDragged) {
                    if (!e.isControlDown() && !e.isShiftDown()) {
                        clearSelection();
                    }
                    int columnIndex = columnAtPoint(p);
                    if (columnIndex < 0 || getColumnCount() <= columnIndex) {
                        return;
                    }
                    final int index0;
                    final int index1;
                    if (isMousePressed) {
                        if (e.isShiftDown()) {
                            index0 = dragStartColumn;
                            index1 = columnIndex;
                        } else {
                            dragStartColumn = columnIndex;
                            index0 = columnIndex;
                            index1 = columnIndex;
                        }
                    } else if (isMouseDragged) {
                        index0 = dragStartColumn;
                        index1 = columnIndex;
                    } else {
                        return;
                    }
                    selectColumn(index0, index1);
                    requestFocus();
                }
            }

        };
        columnHeader.addMouseListener(columnHeaderMouseInputListener);
        columnHeader.addMouseMotionListener(columnHeaderMouseInputListener);
        // sI
        MouseInputListener rowHeaderMouseInputListener = new MouseInputAdapter() {

            private int dragStartRow;

            @Override
            public void mousePressed(MouseEvent e) {
                changeSelection(e);
            }

            @Override
            public void mouseDragged(MouseEvent e) {
                changeSelection(e);
            }

            private void changeSelection(MouseEvent e) {
                Point p = new Point(e.getX(), e.getY());
                if (SwingUtilities.isLeftMouseButton(e)) {
                    int id = e.getID();
                    boolean isMousePressed = (id == MOUSE_PRESSED);
                    boolean isMouseDragged = (id == MOUSE_DRAGGED);
                    if (isMousePressed || isMouseDragged) {
                        if (p.y >= rowHeader.getBounds().height) {
                            return;
                        }
                        if (!e.isControlDown() && !e.isShiftDown()) {
                            clearSelection();
                        }
                        int rowIndex = rowAtPoint(p);
                        if (rowIndex < 0 || getRowCount() < rowIndex) {
                            return;
                        }
                        final int index0;
                        final int index1;
                        if (isMousePressed) {
                            if (e.isShiftDown()) {
                                index0 = dragStartRow;
                                index1 = rowIndex;
                            } else {
                                dragStartRow = rowIndex;
                                index0 = rowIndex;
                                index1 = rowIndex;
                            }
                        } else if (isMouseDragged) {
                            index0 = dragStartRow;
                            index1 = rowIndex;
                        } else {
                            return;
                        }
                        addRowSelectionInterval(index0, index1);
                        addColumnSelectionInterval(getColumnCount() - 1, 0);
                        requestFocus();
                        // swb_ƃe[üʒu킹
                        JViewport tableView = (JViewport)getParent();
                        Point viewPosition = tableView.getViewPosition();
                        viewPosition.y = ((JViewport)rowHeader.getParent()).getViewPosition().y;
                        tableView.setViewPosition(viewPosition);
                    }
                }
            }

        };
        rowHeader.addMouseListener(rowHeaderMouseInputListener);
        rowHeader.addMouseMotionListener(rowHeaderMouseInputListener);
        // ҏW
        InputMap imap = getInputMap();
        final int shortcutKey = Resource.getMenuShortcutKeyMask();
        imap.put(getKeyStroke(VK_X, shortcutKey), "cut");
        imap.put(getKeyStroke(VK_C, shortcutKey), "copy");
        imap.put(getKeyStroke(VK_V, shortcutKey), "paste");
        imap.put(getKeyStroke(VK_A, shortcutKey), "selectAll");
        // J[\ړ
        bindJumpTo("jump-to-home", VK_HOME, false);
        bindJumpTo("jump-to-end", VK_END, false);
        bindJumpTo("jump-to-top", VK_UP, false);
        bindJumpTo("jump-to-bottom", VK_DOWN, false);
        bindJumpTo("jump-to-leftmost", VK_LEFT, false);
        bindJumpTo("jump-to-rightmost", VK_RIGHT, false);
        bindJumpTo("select-to-home", VK_HOME, true);
        bindJumpTo("select-to-end", VK_END, true);
        bindJumpTo("select-to-top", VK_UP, true);
        bindJumpTo("select-to-bottom", VK_DOWN, true);
        bindJumpTo("select-to-leftmost", VK_LEFT, true);
        bindJumpTo("select-to-rightmost", VK_RIGHT, true);
        // L[Cxg
        addKeyListener(new KeyListener() {

            public void keyTyped(KeyEvent e) {
                lastKeyEvent = e;
            }

            public void keyReleased(KeyEvent e) {
                lastKeyEvent = e;
            }

            public void keyPressed(KeyEvent e) {
                lastKeyEvent = e;
            }

        });
        ActionUtility.getInstance(this).bindAction(new AbstractAction("doNothing") {

            public void actionPerformed(ActionEvent e) {
                // ignore
            }

        }, KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0));
    }

    @Override
    public void editingStopped(ChangeEvent e) {
        try {
            super.editingStopped(e);
        } catch (Exception ex) {
            WindowOutputProcessor.showErrorDialog(getParent(), ex);
        }
    }

    @Override
    public boolean editCellAt(int row, int column, EventObject e) {
        boolean succeeded = super.editCellAt(row, column, e);
        if (succeeded) {
            if (editorComp instanceof JTextField) {
                // ZҏW[hŒlIԂɂ
                if (lastKeyEvent != null && lastKeyEvent.getKeyCode() != VK_F2) {
                    JTextField editor = (JTextField)editorComp;
                    initializeEditorComponent(editor);
                    editor.requestFocus();
                    editor.selectAll();
                }
            }
        }
        return succeeded;
    }

    @Override
    public TableCellEditor getCellEditor() {
        TableCellEditor editor = super.getCellEditor();
        if (editor instanceof DefaultCellEditor) {
            DefaultCellEditor d = (DefaultCellEditor)editor;
            initializeEditorComponent(d.getComponent());
        }
        return editor;
    }

    /**
     * CellEditorB
     * @param c CellEditor̃R|[lg
     */
    private void initializeEditorComponent(Component c) {
        final Color bgColor = Color.ORANGE;
        if (c != null && c.getBackground() != bgColor) {
            // wiFŏς𔻒f
            if (!c.isEnabled()) {
                c.setEnabled(true);
            }
            c.setFont(getFont());
            c.setBackground(bgColor);
            if (c instanceof JTextComponent) {
                final JTextComponent text = (JTextComponent)c;
                ActionUtility.setUndoAction(text);
                c.addFocusListener(new FocusAdapter() {

                    @Override
                    public void focusLost(FocusEvent e) {
                        editingCanceled(new ChangeEvent(e.getSource()));
                    }

                });
            }
        }
    }

    @Override
    public TableCellRenderer getCellRenderer(int row, int column) {
        final Object v = getValueAt(row, column);
        if (v == null) {
            return nullRenderer;
        }
        return super.getCellRenderer(row, column);
    }

    @Override
    public void updateUI() {
        super.updateUI();
        adjustRowHeight(this);
    }

    /**
     * e[u̍s𒲐B
     * @param table Ώۃe[u
     */
    static void adjustRowHeight(JTable table) {
        Component c = new JLabel("0");
        final int height = c.getPreferredSize().height;
        if (height > 0) {
            table.setRowHeight(height);
        }
    }

    @Override
    protected void configureEnclosingScrollPane() {
        super.configureEnclosingScrollPane();
        Container p = getParent();
        if (p instanceof JViewport) {
            Container gp = p.getParent();
            if (gp instanceof JScrollPane) {
                JScrollPane scrollPane = (JScrollPane)gp;
                JViewport viewport = scrollPane.getViewport();
                if (viewport == null || viewport.getView() != this) {
                    return;
                }
                scrollPane.setRowHeaderView(rowHeader);
            }
        }
    }

    @Override
    public void setModel(TableModel dataModel) {
        dataModel.addTableModelListener(rowHeader);
        super.setModel(dataModel);
    }

    /**
     * IB
     * @param index0 JnCfbNX
     * @param index1 ICfbNX
     */
    void selectColumn(int index0, int index1) {
        if (getRowCount() > 0) {
            addColumnSelectionInterval(index0, index1);
            addRowSelectionInterval(getRowCount() - 1, 0);
        }
    }

    /**
     * w肵փWvB
     * @param name 
     * @param index
     * @return 񂪑݂ǂ
     */
    boolean jumpToColumn(int index) {
        final int columnCount = getColumnCount();
        if (0 <= index && index < columnCount) {
            if (getSelectedRowCount() == 0) {
                changeSelection(-1, index, false, false);
            } else {
                int[] selectedRows = getSelectedRows();
                changeSelection(getSelectedRow(), index, false, false);
                for (final int selectedRow : selectedRows) {
                    changeSelection(selectedRow, index, false, true);
                }
            }
            return true;
        }
        return false;
    }

    /**
     * w肵փWvB
     * @param name 
     * @return 񂪑݂ǂ
     */
    boolean jumpToColumn(String name) {
        TableColumnModel columnModel = getColumnModel();
        for (int i = 0, n = columnModel.getColumnCount(); i < n; i++) {
            if (name.equals(String.valueOf(columnModel.getColumn(i).getHeaderValue()))) {
                jumpToColumn(i);
                return true;
            }
        }
        return false;
    }

    RowHeader getRowHeader() {
        return rowHeader;
    }

    ResultSetTableModel getResultSetTableModel() {
        return (ResultSetTableModel)getModel();
    }

    boolean isShowColumnNumber() {
        return columnHeaderRenderer.showColumnNumber;
    }

    void setShowColumnNumber(boolean showColumnNumber) {
        final boolean oldValue = this.columnHeaderRenderer.showColumnNumber;
        this.columnHeaderRenderer.showColumnNumber = showColumnNumber;
        firePropertyChange("showNumber", oldValue, showColumnNumber);
    }

    void repaintRowHeader(String propName) {
        if (rowHeader != null) {
            rowHeader.propertyChange(new PropertyChangeEvent(this, propName, null, null));
        }
    }

    /**
     * JUMP_TO_...̊蓖āB
     * @param name ANV
     * @param key ăL[ 
     * @param withSelect I[h
     */
    private void bindJumpTo(String name, int key, boolean withSelect) {
        final boolean extend = withSelect;
        final CellCursor c = new CellCursor(this);
        final int shortcutKey = Resource.getMenuShortcutKeyMask();
        final Action action;
        if (name.endsWith("-to-home")) {
            action = new AbstractAction() {

                public void actionPerformed(ActionEvent e) {
                    changeSelection(0, 0, false, extend);
                }

            };
        } else if (name.endsWith("-to-end")) {
            action = new AbstractAction() {

                public void actionPerformed(ActionEvent e) {
                    int rowCount = getRowCount();
                    int columnCount = getColumnCount();
                    changeSelection(rowCount - 1, columnCount - 1, false, extend);
                }

            };
        } else if (name.endsWith("-to-top")) {
            action = new AbstractAction() {

                public void actionPerformed(ActionEvent e) {
                    changeSelection(0, c.getColumnPosition(), false, extend);
                }

            };
        } else if (name.endsWith("-to-bottom")) {
            action = new AbstractAction() {

                public void actionPerformed(ActionEvent e) {
                    int rowCount = getRowCount();
                    changeSelection(rowCount - 1, c.getColumnPosition(), false, extend);
                }

            };
        } else if (name.endsWith("-to-leftmost")) {
            action = new AbstractAction() {

                public void actionPerformed(ActionEvent e) {
                    changeSelection(c.getRowPosition(), 0, false, extend);
                }

            };
        } else if (name.endsWith("-to-rightmost")) {
            action = new AbstractAction() {

                public void actionPerformed(ActionEvent e) {
                    int columnCount = getColumnCount();
                    changeSelection(c.getRowPosition(), columnCount - 1, false, extend);
                }

            };
        } else {
            throw new AssertionError("unknown key: " + name);
        }
        getActionMap().put(name, action);
        getInputMap().put(getKeyStroke(key, shortcutKey | (withSelect ? SHIFT_DOWN_MASK : 0)), name);
    }

    /**
     * I͈͂̍Ōɒu鑾g̃ZïʒujB
     */
    private static final class CellCursor {

        private JTable table;

        CellCursor(JTable table) {
            this.table = table;
        }

        /**
         * ʒu̎擾B
         * @return ʒu
         */
        int getColumnPosition() {
            int[] a = table.getSelectedColumns();
            if (a == null || a.length == 0) {
                return -1;
            }
            ListSelectionModel csm = table.getColumnModel().getSelectionModel();
            return (a[0] == csm.getAnchorSelectionIndex()) ? a[a.length - 1] : a[0];
        }

        /**
         * sʒu̎擾B
         * @return sʒu
         */
        int getRowPosition() {
            int[] a = table.getSelectedRows();
            if (a == null || a.length == 0) {
                return -1;
            }
            ListSelectionModel rsm = table.getSelectionModel();
            return (a[0] == rsm.getAnchorSelectionIndex()) ? a[a.length - 1] : a[0];
        }

    }

    /**
     * NulllCellRendererB
     */
    private static final class NullValueRenderer extends DefaultTableCellRenderer {

        NullValueRenderer() {
            // empty
        }

        @Override
        public Component getTableCellRendererComponent(JTable table,
                                                       Object value,
                                                       boolean isSelected,
                                                       boolean hasFocus,
                                                       int row,
                                                       int column) {
            Component c = super.getTableCellRendererComponent(table,
                                                              "NULL",
                                                              isSelected,
                                                              hasFocus,
                                                              row,
                                                              column);
            c.setForeground(new Color(63, 63, 192, 192));
            Font font = c.getFont();
            c.setFont(font.deriveFont(font.getSize() * 0.8f));
            return c;
        }

    }

    /**
     * wb_CellRendererB
     */
    private static final class ColumnHeaderCellRenderer implements TableCellRenderer {

        TableCellRenderer renderer;
        boolean showColumnNumber;

        ColumnHeaderCellRenderer(TableCellRenderer renderer) {
            this.renderer = renderer;
            this.showColumnNumber = false;
        }

        public Component getTableCellRendererComponent(JTable table,
                                                       Object value,
                                                       boolean isSelected,
                                                       boolean hasFocus,
                                                       int row,
                                                       int column) {
            Object o = showColumnNumber ? String.format("%d %s", column + 1, value) : value;
            return renderer.getTableCellRendererComponent(table,
                                                          o,
                                                          isSelected,
                                                          hasFocus,
                                                          row,
                                                          column);
        }

    }

    /**
     * swb_B
     */
    private static final class RowHeader extends JTable implements PropertyChangeListener {

        private static final int DEFAULT_WIDTH = 40;

        private DefaultTableModel model;
        private TableCellRenderer renderer;

        RowHeader(JTable table) {
            this.model = new DefaultTableModel(0, 1) {

                @Override
                public boolean isCellEditable(int row, int column) {
                    return false;
                }

            };
            final ResultSetTable t = (ResultSetTable)table;
            this.renderer = new DefaultTableCellRenderer() {

                @Override
                public Component getTableCellRendererComponent(JTable table,
                                                               Object value,
                                                               boolean isSelected,
                                                               boolean hasFocus,
                                                               int row,
                                                               int column) {
                    assert t.getModel() instanceof ResultSetTableModel;
                    final boolean rowLinked = t.getResultSetTableModel().isLinkedRow(row);
                    JLabel label = new JLabel(String.format("%s ", rowLinked ? row + 1 : "+"));
                    label.setHorizontalAlignment(RIGHT);
                    label.setFont(table.getFont());
                    label.setOpaque(true);
                    return label;
                }

            };
            setModel(model);
            setWidth(table);
            setFocusable(false);
            table.addPropertyChangeListener(this);
        }

        /* @see java.beans.PropertyChangeListener#propertyChange(java.beans.PropertyChangeEvent) */
        public void propertyChange(PropertyChangeEvent e) {
            JTable table = (JTable)e.getSource();
            if (table == null) {
                return;
            }
            String propertyName = e.getPropertyName();
            if (propertyName.equals("enabled")) {
                boolean isEnabled = table.isEnabled();
                setVisible(isEnabled);
                if (isEnabled) {
                    setWidth(table);
                    resetViewPosition(table);
                }
            } else if (propertyName.equals("font")) {
                setFont(table.getFont());
            } else if (propertyName.equals("rowHeight")) {
                setRowHeight(table.getRowHeight());
            } else if (propertyName.equals("model")) {
                model.setRowCount(table.getRowCount());
            } else if (propertyName.equals("unlinkedRowStatus")) {
                repaint();
            } else if (propertyName.equals("ancestor")) {
                // empty
            } else {
                // empty
            }
            validate();
        }

        @Override
        public void tableChanged(TableModelEvent e) {
            Object src = e.getSource();
            if (model != null && src != null) {
                model.setRowCount(((TableModel)src).getRowCount());
            }
            super.tableChanged(e);
        }

        @Override
        public void updateUI() {
            super.updateUI();
            adjustRowHeight(this);
        }

        /**
         * ̐ݒB
         * @param table 
         */
        void setWidth(JTable table) {
            // XXX s
            final int rowCount = table.getRowCount();
            model.setRowCount(rowCount);
            JLabel label = new JLabel(String.valueOf(rowCount * 1000L));
            Dimension d = getSize();
            d.width = Math.max(label.getPreferredSize().width, DEFAULT_WIDTH);
            setPreferredScrollableViewportSize(d);
        }

        /**
         * r[ʒuZbgB
         * e[u͍ĕ`悵ɃXN[ʒu(r[̈ʒu)ێ邪A
         * ̏Ԃ͍swb_ɂ͔fȂ߁AIɂsB
         * @param table 
         */
        private void resetViewPosition(JTable table) {
            Container p1 = table.getParent();
            Container p2 = getParent();
            if (p1 instanceof JViewport && p2 instanceof JViewport) {
                JViewport v1 = (JViewport)p1;
                JViewport v2 = (JViewport)p2;
                v2.setViewPosition(v1.getViewPosition());
            }
        }

        @Override
        public boolean isCellEditable(int row, int column) {
            return false;
        }

        @Override
        public TableCellRenderer getCellRenderer(int row, int column) {
            return renderer;
        }

    }

}
