/*
 *	Qizx/Open version 0.4
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.dm;

import net.xfra.qizxopen.util.*;

import java.io.*;

/**
 *	Fully-Ordered Node Identification Document. A representation of XML Documents
 *	where nodes are accessed by integer handles whose values are guaranteed to be
 *	in document order (i.e. node N1 before node N2 <=> handle(N1) < handle(N2) )
 *	<p>Such a representation is suitable for documents that are built 
 *	in one operation such as parsing and never modified. 
 *	<p>The interface is very close to the XPath/XQuery/XSLT Data Model.
 *	<p>The handle value 0 is reserved and represents a null or absent node.
 */
public interface FONIDocument
{
    /**
     *	Returns the handle of the document node.
     */
    int/*NodeId*/   getRootNode();

    /**
     *	Returns one of the kinds DOCUMENT, ELEMENT, TEXT, PROCESSING_INSTRUCTION, COMMENT,
     *	ATTRIBUTE, NAMESPACE, ATOM_XX.
     */
    int  getKind( int/*NID*/ nodeId );
    /**
     *	    Gets the name of a Node. 
     */
    QName getName( int/*NID*/ nodeId );
    /**
     *	    Gets the name of a pseudo Node Attribute / Namespace. 
     */
    QName pnGetName( int/*NID*/ nodeId );
    /**
     *	    Gets the name-id of a real Node.
     *	All names are represented by an index in a name table.
     */
    int  getNameId( int/*NID*/ nodeId );
    /**
     *	Gets the name-id of a pseudo-node Attribute / Namespace. 
     */
    int  pnGetNameId( int/*NID*/ nodeId );

    int/*NID*/  getParent( int/*NID*/ nodeId );

    int/*NID*/  getNextSibling( int/*NID*/ nodeId );

    int/*NID*/  getNodeSpan( int/*NID*/ nodeId );

    /**
     *	Gets the node that is next in document order.
     *  (first child if non-atomic, otherwise returns getNodeAfter()).
     */
    int/*NID*/  getNodeNext( int/*NID*/ nodeId );
    /**
     *	Gets the node that is next in document order but not contained.
     *  It is the following sibling if any, else the parent's following sibling, recursively.
     */
    int/*NID*/  getNodeAfter( int/*NID*/ nodeId );

    int/*NID*/  getFirstChild( int/*NID*/ nodeId );

    interface NodeIterator {
	boolean next();
	int/*NID*/   currentId();
	NodeIterator reborn(); // for XQuery, because of multiple variable references
    }

    NodeIterator childrenIterator( int/*NID*/ nodeId );
    /**
     *	Returns an Attribute iterator for a node.
     */
    NodeIterator attrIterator( int/*NID*/ nodeId );
    /**
     *	Returns the number of attributes of an element (not guaranteed to be efficient).
     */
    int   getAttrCount( int/*NID*/ nodeId );
    /**
     *	Gets an attribute node by name.
     */
    int/*NID*/  getAttribute( int/*NID*/ nodeId, int nameId );

    /**
     *	Gets the string value for any node but Attributes and Namespaces.
     */
    String getStringValue( int/*NID*/ nodeId );
    /**
     *	Gets the string value for pseudo-nodes Attributes and Namespaces.
     */
    String pnGetStringValue( int/*NID*/ nodeId );

    /**
     *	Specially meant for indexing: gets the string value of a text node.
     *	Returns the value into a char array, reserving chars (at head if reserve > 0). 
     */
    char[]  getCharValue( int/*NID*/ nodeId, int reserve );
    /**
     * Specially meant for indexing: gets the string value of a "pseudo-node" (attribute
     * or NS). Returns the value into a char array, possibly reserving leading chars.
     */
    char[]  pnGetCharValue( int/*NID*/ nodeId, int reserve );

    /**
     *	Gets an atomic value from a leaf node. 
     */
    Object getValue( int/*NID*/ nodeId );
    /**
     *	Gets an integer value from an atome of type ATOM_INT. 
     */
    long getIntegerValue( int/*NID*/ nodeId );

    /**
     *	Returns the number of NS defined on this node.
     */
    int getDefinedNSCount( int/*NID*/ nodeId );

    /**
     *	Returns the namespace nodes of a node.
     */
    NodeIterator namespaceIterator( int/*NID*/ nodeId, boolean inScope );

    /**
     *	Gets the total number of element names.
     */
    int  getElementNameCount( );

    /**
     *	Gets the QName of an element node by the internal id.
     */
    QName  getElementName( int nameId );

    /**
     *	Gets the internal id of an element name.
     */
    int    internElementName( QName name );

    /**
     *	Gets the total number of non-element node names.
     */
    int  getOtherNameCount( );

    /**
     *	Gets the QName of a non-element node by the internal id.
     */
    QName  getOtherName( int nameId );

    /**
     *	Gets the internal id of a non-element node name.
     */
    int    internOtherName( QName name );

    /**
     *	Returns the URI of the document.
     */
    String getBaseURI();

    /**
     *	Cache management.
     */
    int estimateMemorySize();

    int getDocId();
    void setDocId(int docId);
}
