/*
 *	Qizx/Open version 0.4
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.dm;

import net.xfra.qizxopen.util.QName;
import net.xfra.qizxopen.util.NSPrefixMapping;
import net.xfra.qizxopen.xquery.XQueryException;

/**
 * A SAX-like abstract event handler.
 * <p>Used for serialization and "push" style evaluation of XQuery constructors.
 */
public interface XMLEventReceiver
{
    /**
     * Resets the state prior to use startDocument() or startElement().
     */
    public void reset();

    /**
     * Terminates a document or simple subtree. May perform consistency checks.
     */
    public void terminate() throws DataModelException;

    /**
     * Starts a document.
     * <p>It is not called in the case only a fragment is generated.
     */
    public void startDocument() throws DataModelException;

    /**
     * Ends a document. Must be balanced by a matching startDocument().
     */
    public void endDocument() throws DataModelException;

    /**
     * Starts an element. Must be balanced by a matching endElement().
     */
    public void startElement(QName name) throws DataModelException;

    /**
     * Adds a namespace node on the current element. Must follow startElement and
     * precede any child. May be interleaved with attribute().
     */
    public void namespace(String prefix, String uri) throws DataModelException;

    /**
     * Adds an attribute on the current element. Must follow startElement and precede
     * any child. May be interleaved with namespace().
     */
    public void attribute(QName name, String value) throws DataModelException;

    /**
     * Ends an element. Discards prefix/namespace mappings possibly defined by the
     * element.
     */
    public void endElement(QName name) throws DataModelException;

    /**
     * Text chunk inside an element. No space is generated before or after.
     */
    public void text(String value) throws DataModelException;

    /**
     * Text of an atom. The difference with text() is that a space in requested between
     * two atoms.
     */
    public void atom(String value) throws DataModelException;

    /**
     * A processing-instruction node.
     */
    public void pi(String target, String value) throws DataModelException;

    /**
     * A comment node.
     */
    public void comment(String value) throws DataModelException;

    /**
     * Convenience method: traverses and generates a subtree into this receiver.
     * 
     * @param inScopeNS if true, copy all in-scope namespace nodes (not only those
     * defined in the node itself).
     */
    public void traverse( Node node, boolean inScopeNS ) throws DataModelException ;

    /**
     * Resolves a prefix to an URI in the context of the current node.
     * 
     * @return null if the prefix cannot be resolved.
     */
    String resolvePrefix( String prefix );

    /**
     * Optional: defines a preferred prefix/namespace mapping for serialization. This
     * table is used when no mapping is in scope in the current context.
     */
    void  definePrefixHints( NSPrefixMapping prefixes );
} 
