/*
 *	Qizx/Open version 0.4
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.dm;

import net.xfra.qizxopen.util.QName;
import net.xfra.qizxopen.util.Namespace;
import net.xfra.qizxopen.util.NSPrefixMapping;

import java.io.Writer;
import java.io.BufferedWriter;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;

/**
 * Serializes a DataModel Node and its subtree to an output stream. Supports several
 * output methods: XML, XHTML, HTML, and TEXT.
 * <p>A Serializer is first instantiated, then several options can be defined through
 * {@link #setOption}, including the output method. An output stream can be specified
 * with {@link #setOutput}.
 * <p>To serialize a node, use the {@link #output(Node)} method.
 */
public class XMLSerializer extends XMLEventReceiverBase
{
    public static final String METHOD = "method";
    public static final String ENCODING = "encoding";
    public static final String VERSION = "version";
    public static final String OMIT_XML_DECLARATION = "omit-xml-declaration";
    public static final String STANDALONE = "standalone";
    public static final String DOCTYPE_SYSTEM = "doctype-system";
    public static final String DOCTYPE_PUBLIC = "doctype-public";
    public static final String MEDIA_TYPE = "media-type";
    public static final String ESCAPE_URI_ATTRIBUTES = "escape-uri-attributes";
    public static final String INCLUDE_CONTENT_TYPE = "include-content-type";
    public static final String INDENT = "indent";
    public static final String INDENT_VALUE = "indent-value";

    static Namespace XHTML = Namespace.get("http://www.w3.org/1999/xhtml");

    protected BufferedWriter out = new BufferedWriter(new OutputStreamWriter(System.out));
    protected String outputFile = null;
    protected Method method = new XMLMethod();
    protected boolean omitXmlDecl = false;
    protected boolean escapeUriAttr = true;
    protected boolean includeContentType = true;
    protected boolean standalone = false;
    protected boolean indents = !false;
    protected boolean dummy = false;
    protected int    maxDepth = -1;
    protected int    indentValue = 2;
    protected String encoding = "UTF-8";
    protected String version = "1.0";
    protected String mediaType;
    protected String publicId = null, systemId = null;

    protected boolean enableIndent, firstElement;
    protected String indent = "  ";

    // peculiarities of output methods:
    interface Method {
	boolean onlyText();
	boolean isMinimized( QName elementName );
	void    afterStartTag();
	String  endOfEmptyTag();
	String  endOfPI();
	void    outputAttr( String s, QName attrName );
	void    outputText( String s );
    }

    protected String PI_END = "?>";
    protected String EMPTY_END = "/>";  // end of empty element tag

    /**
     * Constructs a XMLSerializer with default XML output method.
     */
    public XMLSerializer( ) {  }

    /**
     * Constructs a XMLSerializer with specification of an output method.
     * 
     * @param method output method name (case-insensitive): XML, XHTML, HTML, or TEXT.
     * @exception DataModelException when the method name is invalid.
     */
    public XMLSerializer( String method ) throws DataModelException {
	setOption("method", method);
    }

    /**
     * Constructs a XMLSerializer with an output writer.
     * 
     * @param output an open Writer, does not need to be buffered.
     */
    public XMLSerializer( Writer output ) {
	setOutput(output);
    }

    /**
     * Serializes a node and its subtree.
     * 
     * @param node node to serialize. If it is not a document, the XML header is not
     * generated.
     */
    public void output( Node node ) throws DataModelException {
	reset();
	startDocument();
	depth = 0;
	traverse(node, true);
	endDocument();
	terminate();
    }

    /**
     * Defines or redefines the output.
     */
    public void setOutput( OutputStream output, String encoding ) {
	Writer w = encoding == null? new OutputStreamWriter(output)
	    : new OutputStreamWriter( output, Charset.forName(encoding));
	out = new BufferedWriter(w);
    }

    /**
     * Defines or redefines the output.
     */
    public void setOutput( Writer output ) {
	out = new BufferedWriter(output);
    }

    /**
     * Gets the current output as a Bufferedriter.
     */
    public BufferedWriter getOutput() {
	return out;
    }

    /**
     * Sets an option.
     * <p>Supported options:
     * <ul>
     * <li>"method" : value is "XML" (default), "XHTML", "HTML" or "TEXT"
     * <li>"encoding" : an encoding supported by Java.
     * <li>"version" : simply generated in XML declaration, not checked.
     * <li>"omit-xml-declaration" : value is "yes" or "no".
     * <li>"standalone" : value is "yes" or "no".
     * <li>"doctype-system" : system id of DTD.
     * <li>"doctype-public" : public id of DTD.
     * <li>"media-type" :
     * <li>"escape-uri-attributes" : value is "yes" or "no".
     * <li>"include-content-type" :
     * <li>"indent" : value is "yes" or "no".
     * <li>"indent-value" : integer number of spaces used for indenting (non-standard).
     * </ul>
     * 
     * @param option name of the option (see above).
     * @param value option value in string form.
     * @exception DataModelException on bad option name or value.
     */
    public void setOption( String option, String value ) throws DataModelException {
	
	if(option.equalsIgnoreCase(METHOD)) {
	    if(value.equalsIgnoreCase("XML"))
		method = new XMLMethod();
	    else if(value.equalsIgnoreCase("XHTML"))
		method = new HTMLMethod(true);
	    else if(value.equalsIgnoreCase("HTML")) {
		method = new HTMLMethod(false);
		omitXmlDecl = true;
	    }
	    else if(value.equalsIgnoreCase("TEXT"))
		method = new TextMethod();
	    else throw new DataModelException("invalid method: "+value);
	}
	else if(option.equalsIgnoreCase(ENCODING))
	    encoding = value;
	else if(option.equalsIgnoreCase(VERSION))
	    version = value;
	else if(option.equalsIgnoreCase(OMIT_XML_DECLARATION))
	    omitXmlDecl = boolOption(option, value);
	else if(option.equalsIgnoreCase(STANDALONE))
	    standalone = boolOption(option, value);
	else if(option.equalsIgnoreCase(DOCTYPE_SYSTEM))
	    systemId = value;
	else if(option.equalsIgnoreCase(DOCTYPE_PUBLIC))
	    publicId = value;
	else if(option.equalsIgnoreCase(MEDIA_TYPE))
	    mediaType = value;
	else if(option.equalsIgnoreCase(ESCAPE_URI_ATTRIBUTES))
	    escapeUriAttr = boolOption(option, value);
	else if(option.equalsIgnoreCase(INCLUDE_CONTENT_TYPE))
	    includeContentType = boolOption(option, value);
	else if(option.equalsIgnoreCase(INDENT))
	    indents = boolOption(option, value);
	else if(option.equalsIgnoreCase(INDENT_VALUE))
	    indentValue = Integer.parseInt(value);
	else if(option.equalsIgnoreCase("dummy-display"))
	    dummy = boolOption(option, value);
	else 
	    throw new DataModelException("invalid option: "+option);
    }

    protected boolean boolOption( String option, String v ) throws DataModelException {
	if(v.equalsIgnoreCase("yes") || v.equalsIgnoreCase("true") || v.equals("1"))
	    return true;
	if(v.equalsIgnoreCase("no") || v.equalsIgnoreCase("false") || v.equals("0"))
	    return false;
	throw new DataModelException("invalid value of option '"+option+"': "+v);
    }

    /**
     * Extension: defines a maximum tree depth. Subtrees deeper than this value appear
     * as an ellipsis '...'.
     */
    public void setDepth( int maxDepth ) {
	this.maxDepth = maxDepth;
    }
    /**
     * Extension: defines the number of spaces used for one level of indentation.
     */
    public void setIndent( int indent ) {
	indents = indent > 0;
	indentValue = indent;
    }
    /**
     * Returns the current encoding.
     * <p>The encoding can have been defined by setOutput or by setOption.
     */
    public String getEncoding() {
	return encoding;
    }

    /**
     * Lower-level output method. Must be preceded by reset(), and optionally
     * startDocument().
     */
    public void traverse( Node node ) throws DataModelException {
	traverse(node, false);
    }

    // ------------------- interface XMLEventReceiver: ------------------------------

    /**
     * [internal: implementation of interface XMLEventReceiver]
     * <p>Prepares the serialization of another tree.
     */
    public void reset() {
	super.reset();
	enableIndent = false;
	//trace = false;
	firstElement = true;
	indent = indents ? 
	    "                ".substring(0, Math.min(indentValue, 16)) : null;
	if(standalone || ! "UTF-8".equals(encoding))
	    omitXmlDecl = false;
    }

    /**
     * [internal: implementation of interface XMLEventReceiver]
     */
    public void terminate() throws DataModelException {
	println();
        try {
	    out.flush();
        }
        catch (java.io.IOException e) {
            throw new DataModelException("IO error", e);
        }
    }

    public void  startDocument() throws DataModelException {
	super.startDocument();
	if(omitXmlDecl || method.onlyText())
	    return;
	print("<?xml version='"); print(version); print('\'');
	print(" encoding='"+ encoding +'\'');
	print("?>");
	println();
    }

    public void  endDocument() throws DataModelException {
	super.endDocument();
    }

    protected void  flushElement( boolean empty ) {
	if(!elementStarted || method.onlyText())
	    return;
	if(firstElement) {      //TODO problem with leading PI/comments
	    if(!omitXmlDecl && (publicId != null || systemId != null)) {
		print("<!DOCTYPE "); print(tagName.getLocalName());
		if(publicId != null) {
		    print(" PUBLIC \""); print(publicId); print("\"");
		    if(systemId != null) {
			print(" \""); print(systemId); print("\"");
		    }
		}
		else if(systemId != null) {
		    print(" SYSTEM \""); print(systemId); print("\"");
		}
		print(">"); println();
	    }
	    firstElement = false;
	}

	doIndent(); 
	print('<');
	printQName(tagName, /*attr*/ false);     
	for( int a = 0; a < attrCnt; a++ ) {
	    print(' ');
	    printQName(attrNames[a], /*attr*/ true);
	    print("=\""); 
	    method.outputAttr(attrValues[a], attrNames[a]);
	    print('\"'); 
	}

	// generate xmlns: pseudo-attributes for added NS:

	for(int n = nsCnt; n > 0 ; n--) {
	    print(" xmlns");
	    Namespace ns = prefixes.getLastNamespace(n);
	    String prefix = prefixes.getLastPrefix(n);
	    if(prefix.length() > 0) {
		print(':'); print(prefix);
	    }
	    print("=\"");
	    method.outputAttr(ns.toString(), null);
	    print("\"");
	}
	print(empty ? method.endOfEmptyTag() : ">");

	elementStarted = false;
	enableIndent = true; 
	spaceNeeded = false;
	method.afterStartTag();
    }

    public void  endElement(QName name) throws DataModelException {
	if(method.onlyText())
	    return;
	boolean noEndTag = elementStarted;
	if(elementStarted) {    // empty content
	    flushElement( noEndTag = method.isMinimized(name) );
	}
	if(!noEndTag) {
	    doIndent(); 
	    print("</");
	    printQName(name, false);
	    print('>');
	}
	super.endElement(name);
	enableIndent = true; 
	spaceNeeded = false;
     }

    public void  text(String value) {
	if(trace) System.err.println("-- text |"+value+'|');
	if(elementStarted)
	    flushElement(false);
	// detect ignorable WS
	int c = value.length();
	if(indent != null)
	    for(; --c >= 0; )
		if(!Character.isWhitespace(value.charAt(c)))
		    break;
	enableIndent = c < 0;   // may indent after ignorable WS
	method.outputText(value);
	spaceNeeded = false;
    }

    public void  atom(String value) {
	if(trace) System.err.println("-- atom "+value);
	if(elementStarted)
	    flushElement(false);
	if(spaceNeeded)
	    print(" ");
	method.outputText(value);
	spaceNeeded = true;
	enableIndent = false;
    }

    public void  pi(String target, String value) {
	if(method.onlyText())
	    return;
	if(elementStarted)
	    flushElement(false);
	doIndent(); 
	print("<?");
	print(target);
	print(' ');
	print(value);
	print( method.endOfPI() );
    }

    public void  comment(String value) {
	if(method.onlyText())
	    return;
	if(elementStarted)
	    flushElement(false);
	doIndent(); 
	print("<!--");
	print(value);
	print("-->");
    }

    void printQName( QName name, boolean isAttribute ) {

	Namespace ns = name.getNamespace();
	if(ns == Namespace.XML) {       // needs not to be declared
	    print("xml:"); print(name.getLocalName());
	}
	else {
	    String prefix = prefixes.convertToPrefix(ns);
	    if(prefix == null)
		if(ns == Namespace.NONE)
		    prefix = "";
		else {
		    if(prefixHints == null || (prefix = prefixHints.convertToPrefix(ns)) == null)
			// generate a non-used prefix:
			for(int id = 1; ; id++) {
			    prefix = "ns"+id;
			    if(prefixes.convertToNamespace(prefix) == null)
				break;
			}
		    prefixes.addMapping(prefix, ns);
		    ++ nsCnt;
		}
	    if(prefix.length() > 0) {
		print(prefix); print(':');
	    }
	    print(name.getLocalName());
	}
    }

    protected void doIndent() {
	if(enableIndent && indent != null) {
	    println();
	    for(int i = 1; i < depth; i++)
		print(indent);
	    enableIndent = false;
	}
    }

    protected void print( String s ) {
	if(maxVolume > 0 && volume > maxVolume)
	    return;
	try {
	    if(!dummy) out.write(s);
	    volume += s.length();
	}
        catch (Exception e) {
            e.printStackTrace();
        }
    }

    protected void print( char c ) {
	if(maxVolume > 0 && volume > maxVolume)
	    return;
	try {
	    if(!dummy) out.write(c);
	    volume ++;
	}
        catch (Exception e) {
            e.printStackTrace();
        }
    }

    protected void println() {
	if(maxVolume > 0 && volume > maxVolume)
	    return;
	try {
	    if(!dummy) out.newLine();
	    volume ++;
	}
        catch (Exception e) {
            e.printStackTrace();
        }
    }

    

    class XMLMethod implements Method {

	public boolean onlyText() {
	    return false;
	}

	public String endOfEmptyTag() {
	    return "/>";
	}

	public void afterStartTag() {
	}

	public boolean isMinimized( QName element ) {
	    return true;
	}

	public void outputText( String s ) {
	    for(int i = 0, L = s.length(); i < L; i++) {
		char c = s.charAt(i);
		switch(c) {
		    case '\n':
			println(); break;
		    case '\r':
			print("&#13;"); break;
		    case '&':
			print("&amp;"); break;
		    case '<':
			print("&lt;"); break;
		    case  '>':
			print("&gt;"); break;
		    default:
			print(c); break;
		}
	    }
	}

	public void outputAttr( String s, QName attrName ) {
	    for(int i = 0, L = s.length(); i < L; i++) {
		char c = s.charAt(i);
		switch(c) {
		    case '\t':
			print("&#9;"); break;
		    case '\n':
			print("&#10;"); break;
		    case '\r':
			print("&#13;"); break;
		    case '&':
			print("&amp;"); break;
		    case '<':
			print("&lt;");  break;
		    default:
			print(c); break;
		}
	    }
	}

	public String endOfPI() {
	    return "?>";
	}
    }

    class HTMLMethod extends XMLMethod {
	boolean xhtml;
	boolean noEscape = false;

	HTMLMethod(boolean xhtml) {
	    this.xhtml = xhtml;
	}

	boolean isHTML(Namespace ns) {      // recognize xhtml even in HTML
	    return ns == XHTML || (!xhtml && ns == Namespace.NONE);
	}

	public String  endOfEmptyTag() {
	    return xhtml ? " />" : ">";   // required space in XHTML: for user agents
	}

	public boolean isMinimized( QName tag ) {
	    if( !isHTML(tag.getNamespace()) )
		return false;
	    String name = tag.getLocalName();
	    switch( Character.toLowerCase(name.charAt(0)) ) {
		case 'a':
		    return name.equalsIgnoreCase("area");
		case 'b':
		    return name.equalsIgnoreCase("br") ||
			   name.equalsIgnoreCase("base") || name.equalsIgnoreCase("basefont");
		case 'c':
		    return name.equalsIgnoreCase("col");
		case 'f':
		    return name.equalsIgnoreCase("frame");
		case 'h':
		    return name.equalsIgnoreCase("hr");
		case 'i':
		    return name.equalsIgnoreCase("img") ||
			   name.equalsIgnoreCase("input") || name.equalsIgnoreCase("isindex");
		case 'l':
		    return name.equalsIgnoreCase("link");
		case 'm':
		    return name.equalsIgnoreCase("meta");
		case 'p':
		    return name.equalsIgnoreCase("param");
		default:
		    return false;
	    }
	}

	public void afterStartTag() {
	    if( isHTML(tagName.getNamespace()) ) {
		String ncname = tagName.getLocalName();
		// include meta after head ?
		if(includeContentType && ncname.equalsIgnoreCase("head")) {
		    print( "<meta http-equiv='Content-Type' content='text/html; charset="+
			   encoding + "'" + endOfEmptyTag() );
		}
		// do not escape SCRIPT and STYLE
		if( !xhtml && 
		    (ncname.equalsIgnoreCase("script") || ncname.equalsIgnoreCase("style")) )
		    noEscape = true;
	    }           
	}

	public void outputText( String s ) {
	    if(noEscape) {
		print(s);
		noEscape = false;
		return;
	    }
	    for(int i = 0, L = s.length(); i < L; i++) {
		char c = s.charAt(i);
		switch(c) {
		    case '\n':
			println(); break;
		    case '\r':
			print("&#13;"); break;
		    case '&':
			print("&amp;"); break;
		    case '<':
			print("&lt;"); break;
		    case '>':
			print("&gt;"); break;
		    case 160:
			print("&nbsp;"); break;
		    default:
			print(c); break;
		}
	    }
	}

	public void outputAttr( String s, QName attrName ) {
	    boolean prevBlank = false;
	    for(int i = 0, L = s.length(); i < L; i++) {
		char c = s.charAt(i);
		switch(c) {
		    case '\n':
		    case '\r':
		    case ' ':
			if(!prevBlank) print(' '); // collapse spaces
			prevBlank = true;
			break;
		    case '&':
			print("&amp;"); break;
		    case '<':
			if(xhtml) print("&lt;"); else print(c); break;
		    default:
			print(c); prevBlank = false;
			break;
		}
	    }
	}

	public String endOfPI() {
	    return xhtml ? "?>" : ">";
	}
    }

    class TextMethod extends XMLMethod {

	public boolean onlyText() {
	    return true;
	}

	public void outputText( String s ) {
	    print(s);
	}
    }

} // end of class Serializer

