/*
 *	Qizx/Open version 0.4
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery;

import net.xfra.qizxopen.xquery.impl.*;
import java.util.Vector;
import java.io.PrintWriter;

/**
 *  A message log for XQuery parsing, static and dynamic analysis.
 *  <p>In the default implementation, messages are both printed immediately and 
 *     stored for later retrieval.
 *  <p>Message locations are character offsets in an input text sequence.
 *  <p>An overloadable method allows to redefine message display.
 *     Messages can also be retrieved individually.
 */
public class Log
{
    public static final int INFO = 0;
    public static final int TRACE = 1;
    public static final int WARNING = 2;
    public static final int ERROR = 3;

    protected int errorCount = 0;
    protected Vector messages = new Vector();

    // default output
    protected PrintWriter output;
    protected boolean printSource = true;

    /**
     *	Creates a default Log with output on the standard error stream.
     */
    public Log( ) {
	this(new PrintWriter(System.err, true));
    }

    /**
     *	Creates a default Log with output on a stream.
     */
    public Log( PrintWriter errorStream ) {
	output = errorStream;
    }

    /**
     *	Clears the messages and the error count.
     */
    public void reset() {
	flush();
	messages.setSize(0);
	errorCount = 0;
    }

    /**
     *	Returns the total number of messages since the last reset().
     */
    public int getMessageCount() {
	return messages.size();
    }

    /**
     *	Returns the number of errors since the last reset().
     */
    public int getErrorCount() {
	return errorCount;
    }

    /**
     *	Retrieves a message by its rank.
     */
    public Message getMessage( int rank ) {
	return rank < 0 || rank >= messages.size() ?
	    null : (Message) messages.get(rank);
    }

    /**
     *	Flushes the output.
     */
    public void flush() {
	output.flush();
    }

    

    /**
     *	Logs an error.  Called by the parser, the type checking or the runtime.
     *	@param location character offset of the error location.
     *	@param message text of the message. Can contain references to arguments of the
     *	form '%n'.
     */
    public void error( Module module, int location, String message,
		       String[] arguments) {
	++ errorCount;
	Message msg = new Message(module, location, ERROR, message, arguments);
	printMessage(msg);
	messages.add(msg);
    }

    /**
     *	Logs an error.  Called by the parser, the type checking or the runtime.
     */
    public void error( Module module, int location, String message) {
	error(module, location, message, new String[0] );
    }

    /**
     *	Logs an error.  Called by the parser, the type checking or the runtime.
     */
    public void error( Module module, int location, String message,
		       String arg) {
	error(module, location, message, new String[] { arg }  );
    }

    /**
     *	Logs an error.  Called by the parser, the type checking or the runtime.
     */
    public void error( Module module, int location, String message,
		       String arg1, String arg2) {
	error(module, location, message, new String[] { arg1, arg2 }  );
    }

    public void warning( Module module, int location, String message,
			 String[] arguments ) {
	Message msg= new Message(module, location, WARNING, message, arguments);
	printMessage(msg);
	messages.add(msg);
     }

    public void warning( Module module, int location, String message,
			 String arg) {
	warning(module, location, message, new String[] { arg } );
    }

    public void trace( Module module, int location, String message,
		       String[] arguments ) {
	Message msg = new Message(module, location, TRACE, message, arguments);
	printMessage(msg);
	messages.add(msg);
     }

    /**
     *	Supplementary information.
     */
    public void info( String message ) {
	info(null, 0, message);
    }

    public void info( Module module, int location, String message ) {
	Message msg = new Message(module, location, INFO, message, null);
	printMessage(msg);
	messages.add(msg);
    }

    /**
     *	Overloadable display method. 
     */
    public void printMessage( Message msg ) {

	if(msg.severity != INFO) {
	    StringBuffer L = new StringBuffer();
	    L.append(msg.severity == ERROR ? "*** "
		     : msg.severity == WARNING ? "* warning "
		     : "-- trace ");
	    if(msg.module == null)
		L.append("at start of input, ");
	    else if(msg.module.getPhysicalURI() != null)
		L.append("in "+ msg.module.getPhysicalURI() +", ");
	    CharSequence source = msg.module == null ? null : msg.module.getSource();
	    if( source != null ) 
		L.append( "at line " + getLineNumber(source, msg.location) );
	    else
		L.append( "at char offset " + msg.location );
	    L.append(": ");
	    L.append(msg.expand());
	    println(L.toString());
	    if(printSource && msg.severity != TRACE && msg.module != null) {
		println( getLine( source, msg.location) );
		L.setLength(0);
		for(int i = getColumn( source, msg.location); --i >= 0; )
		    L.append('-');	//TODO: pb with tabs
		L.append('^');
		println(L.toString());
	    }
	}
	else {
	    println("  " + msg.message);
	}
	output.flush();
    }

    /**
     *	Overloadable output method. 
     */
    protected void println( String s ) {
	output.println(s);
    }

    /**
     *	Returns the line number, given the location offset and the input text.
     */
    public static int getLineNumber( CharSequence input, int location ) {
	int lino = 1;
	for(int i = location; --i >= 0; )	// not very fast, doesn't matter
	    if(input.charAt(i) == '\n')
		++ lino;
	return lino;
    }

    /**
     *	Returns the column number, given the location offset and the input text.
     */
    public static int getColumn( CharSequence input, int location ) {
	int ls = location;
	for( ; --ls >= 0; )
	    if(input.charAt(ls) == '\n')
		break;
	return location - ls - 1;
    }

    /**
     *	Returns the line text, given the location offset and the input text.
     */
    public static String getLine( CharSequence input, int location ) {
	int ls = location;
	for( ; --ls >= 0; )
	    if(input.charAt(ls) == '\n')
		break;
	++ ls;
	int le = ls, lEnd = input.length();
	for( ; le < lEnd; le++)
	    if(input.charAt(le) == '\n')
		break;
	return input.subSequence(ls, le).toString();
    }

    /**
     *	Storage of a message.
     */
    public static class Message 
    {
	public int    severity;
	public Module module;
	public int    location;
	public String message;
	public String[] arguments;

	Message( Module module, int location,
		 int severity, String message, String[] arguments) {
	    this.module = module;
	    this.severity = severity;
	    this.location = location;
	    this.message = message;
	    this.arguments = arguments;
	}

	public String expand() {
	    StringBuffer buf = new StringBuffer(message.length() );
	    for(int i = 0, L = message.length(); i < L; i++) {
		char c = message.charAt(i);
		if(c != '%')
		    buf.append(c);
		else {
		    c = message.charAt(++ i);
		    if(c < '1' || c> '9')
			buf.append(c);
		    else buf.append(arguments[c - '1']);
		}
	    }
	    return buf.toString();
	}
    }
} // end of class Log

