/*
 *	Qizx/Open version 0.4
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.fn;
import net.xfra.qizxopen.xquery.impl.*;

import net.xfra.qizxopen.util.QName;
import net.xfra.qizxopen.util.Util;
import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.op.Expression;
import net.xfra.qizxopen.xquery.op.GlobalVariable;
import net.xfra.qizxopen.xquery.dm.Node;
import net.xfra.qizxopen.xquery.dt.*;

import java.lang.reflect.*;
import java.util.HashSet;
import java.util.HashMap;
import java.util.Enumeration;
import java.util.Vector;
import java.util.ArrayList;
import java.util.Date;
import java.math.BigDecimal;

/**
 *   Function bound to a Java method.
 *   <p>The binding mechanism is based on the qname of the function: the uri part
 *   has the form java:<fully_qualified_class_name> (e.g. "java:java.lang.Math")
 *   and the local part matches the Java method name. For example the following
 *   code calls java.lang.Math.log :
 *	<pre>declare namespace math = "java:java.lang.Math"
 *	math:log(1)</pre>
 *   <p>The function name can be in XPath style with hyphens: it is converted
 *   into 'camelCase' style automatically. For example "list-files" is converted
 *   to "listFiles".
 *   <p>
 */
public class JavaFunction extends Function
{
    Prototype[] prototypes;
    public static boolean trace = false;
    public static HashMap classToType = new HashMap();
    static {
	classToType.put(String.class, Type.STRING.opt);
	classToType.put(Node.class, Type.NODE);
	classToType.put(boolean.class, Type.BOOLEAN);
	classToType.put(Boolean.class, Type.BOOLEAN);
	classToType.put(double.class, Type.DOUBLE);
	classToType.put(Double.class, Type.DOUBLE);
	classToType.put(float.class, Type.FLOAT);
	classToType.put(Float.class, Type.FLOAT);
	classToType.put(BigDecimal.class, Type.DECIMAL);
	classToType.put(long.class, Type.INTEGER);
	classToType.put(Long.class, Type.INTEGER);
	classToType.put(Integer.class, Type.INT);
	classToType.put(int.class, Type.INT);
	classToType.put(short.class, Type.SHORT);
	classToType.put(Short.class, Type.SHORT);
	classToType.put(byte.class, Type.BYTE);
	classToType.put(Byte.class, Type.BYTE);
	classToType.put(char.class, Type.CHAR);
	classToType.put(Character.class, Type.CHAR);
	classToType.put(void.class, Type.NONE);
	classToType.put(Date.class, Type.DATE_TIME);
	// special:
	classToType.put(Value.class, Type.ANY);

	// arrays:
	classToType.put("[Ljava.lang.String;", Type.STRING.star);
	classToType.put("[Lnet.xfra.qizxopen.xquery.dm.Node;", Type.NODE.star);
	classToType.put("[D", Type.DOUBLE.star);
	classToType.put("[F", Type.FLOAT.star);
	classToType.put("[J", Type.INTEGER.star);
	classToType.put("[I", Type.INT.star);
	classToType.put("[S", Type.SHORT.star);
	classToType.put("[B", Type.BYTE.star);
	classToType.put("[C", Type.INTEGER.star);
	// this is a little hack: final Item[] argument is considered the mark
	// of a vararg function (serious problems if not the last arg !)
	classToType.put("[Lnet.xfra.qizxopen.xquery.Item;", Type.ITEM);

	classToType.put(Enumeration.class, Type.WRAPPED_OBJECT.star);
	classToType.put(Vector.class, Type.WRAPPED_OBJECT.star);
	classToType.put(ArrayList.class, Type.WRAPPED_OBJECT.star);
    }
    static Class[] hookProto = { PredefinedModule.class };

    public static class Prototype extends net.xfra.qizxopen.xquery.fn.Prototype
    {
	Method method;			// method and constructor are exclusive
	Constructor constructor;
	QName  autoArg0;   // name of a global automatically added as first arg.
	boolean isStatic;  // true for static method

	public Prototype( QName qname, Type returnType,
			  Method method, Constructor constructor) {
	    super(qname, returnType, Call.class);
	    this.method = method;
	    this.constructor = constructor;
	}

	public Function.Call newInstance(StaticContext ctx,
					 Expression[] actualArguments) {
	    Call call = (Call) super.newInstance(ctx, actualArguments);

	    if(autoArg0 != null && !Modifier.isStatic(method.getModifiers())) {
		// add a first arg which a reference to a global
		GlobalVariable global = ctx.lookforGlobalVariable(autoArg0);
		if(global == null)
		    throw new RuntimeException("no such auto arg "+autoArg0);
		call.autoArg0 = global;
	    }
	    return call;
	}
    }
    
    public static class Plugger implements PredefinedModule.FunctionPlugger {
	HashSet allowedClasses;

	public void authorizeClass( String className ) {
	    if(allowedClasses == null)
		allowedClasses = new HashSet();
	    allowedClasses.add(className);
	}

	public Function plug( QName qname, PredefinedModule target )
	    throws SecurityException {
	    String uri = qname.getURI();
	    if(!uri.startsWith("java:"))
		return null;
	    String className = uri.substring(5);

	    // detect auto arg0 specification:	java:className?localname=uri
	    QName autoArg0 = null;
	    int mark = className.indexOf('?');
	    if(mark > 0) {
		int eq = className.indexOf('=', mark);
		String argName = className.substring(mark+1, eq);
		String argUri = className.substring(eq + 1);
		autoArg0 = QName.get(argUri, argName);
		className = className.substring(0, mark);
	    }
	    // the function name can be itself a qualified name (eg io.File.new)
	    String functionName = qname.getLocalName();
	    int dot = functionName.lastIndexOf('.');
	    if(dot >= 0) {
		className = className + '.' + functionName.substring(0, dot);
		functionName = functionName.substring(dot + 1);
	    }

	    if(allowedClasses != null && !allowedClasses.contains(className))
		throw new SecurityException(
		    "Java extension: security restriction on class "+className);

	    try {
		Class fclass = Class.forName(className);
		// find a plug hook in the class: can declare extra stuff
		try {
		    Method hook = fclass.getMethod("plugHook", hookProto);
		    hook.invoke(null, new Object[] { target });
		}
		catch(Exception diag) {
		    if(trace) System.err.println("no hook for "+fclass+" "+diag);
		}

		// look for the function or constructor
		Prototype[] protos;
		int kept = 0;
		String name = Util.camelCase(functionName, false);
		if(trace)
		    System.err.println("found Java class "+fclass+
				       " for function "+name);
		if(name.equals("new")) {
		    // look for constructors 
		    int mo = fclass.getModifiers();
		    if( ! Modifier.isPublic(mo) ||
			Modifier.isAbstract(mo) || Modifier.isInterface(mo))
			return null;	// instantiation will fail

		    Constructor[] constructors = fclass.getConstructors();
		    protos = new Prototype[constructors.length];
		    for (int c = 0; c < constructors.length; c++) {
			protos[kept] = convertConstructor(qname, constructors[c]);
			if(protos[kept] != null) {
			    if(trace)
				System.err.println("detected constructor "+
						   protos[kept]);
			    ++ kept;
			}
		    }
		}
		else {
		    Method[] methods = fclass.getMethods();
		    protos = new Prototype[methods.length];
		    for(int m = 0; m < methods.length; m++)
			// match name without conversion:
			if( methods[m].getName().equals(name)) {
			    protos[kept] =
				convertMethod(qname, methods[m], autoArg0);
			    if(protos[kept] != null) {
				if(trace)
				    System.err.println("detected method "+
						       protos[kept]);
				++ kept;
			    }
			}
		}
		if(kept > 0)
		    return new JavaFunction(protos, kept);
	    }
	    catch (ClassNotFoundException e) {
		if(trace) System.err.println("*** class not found "+className);
	    }
	    catch (Exception ie) {
		ie.printStackTrace();	// abnormal: report
	    }
	    return null;
	}
    }

    static Prototype convertMethod( QName qname, Method method, QName autoArg0 ) {
	if( !Modifier.isPublic(method.getModifiers()) )
	    return null;
	Class[] params = method.getParameterTypes();
	Prototype proto = new Prototype(qname, null, method, null);
	proto.autoArg0 = autoArg0;
	proto.isStatic = Modifier.isStatic(method.getModifiers());

	if(autoArg0 == null && !Modifier.isStatic(method.getModifiers()))
	    // add a first argument 'this'
	    proto.arg("this", convertClass(method.getDeclaringClass()));
	for(int p = 0; p < params.length; p++) {
	    Type type = convertClass(params[p]);
	    if(type != Type.ITEM)
		proto.arg("p"+(p+1), type );
	    else { // hack:
		proto.vararg = true;
	    }
	}
	proto.returnType = convertClass( method.getReturnType() );
	return proto;
    }

    static Prototype convertConstructor( QName qname, Constructor constr ) {
	if( !Modifier.isPublic(constr.getModifiers()) )
	    return null;
	Class[] params = constr.getParameterTypes();
	Prototype proto = new Prototype(qname, null, null, constr);
	for(int p = 0; p < params.length; p++) {
	    proto.arg("p"+(p+1), convertClass(params[p]));
	}
	proto.returnType = convertClass(constr.getDeclaringClass());
	return proto;
    }

    public static synchronized Type convertClass(Class javaType) {
	Type type = (Type) classToType.get(javaType);
	if(type != null)
	    return type;
	// happens when an actual type is passed (called from convertObject):
	if(net.xfra.qizxopen.dm.Node.class.isAssignableFrom(javaType))
	    return Type.NODE;
	if(javaType.isArray()) {
	    // apparently no way to get the array item type:
	    String name = javaType.getName();
	    type = (Type) classToType.get(name);
	    if(type == null)
		classToType.put( name,
				 type = (new WrappedObjectType(javaType)).star);
	}
	else {
	    classToType.put(javaType, type = new WrappedObjectType(javaType));
	}
	return type;
    }

    /**
     *	Convert Java object to XQuery item or sequence.
     */
    public static Value convertObject( Object object ) {
	if(object == null)
	    return Value.empty;
	return convertClass(object.getClass()).convertFromObject(object);
    }

    // -------------------------------------------------------------------------

    public JavaFunction( Prototype[] protos, int count ) {
        prototypes = new Prototype[count];
	System.arraycopy(protos, 0, prototypes, 0, count);
    }

    public net.xfra.qizxopen.xquery.fn.Prototype[] getProtos()
	{ return prototypes; } // dummy here


    public static class Call extends Function.Call
    {
	GlobalVariable autoArg0;

	public void dump( ExprDump d ) {
	    d.header( this, getClass().getName() );
	    d.display("prototype", prototype.toString());
	    if(autoArg0 != null) d.display("auto arg", autoArg0);
	    d.display("actual arguments", args);
	}

	public Value eval( Focus focus, EvalContext context )
	    throws XQueryException {
	    context.at(this);
	    Prototype proto = (Prototype) prototype;

	    if(proto.method != null) {
		Object target = null;
		if(autoArg0 != null) {
		    Value v = context.loadGlobal(autoArg0);
		    target = ((SingleWrappedObject) v).getObject();
		}
		else if(!proto.isStatic) 
		    target = proto.argTypes[0].convertToObject(args[0],
							       focus, context);

		int argcnt = proto.argCnt, shift = 0;
		if(!proto.isStatic && autoArg0 == null) {
		    -- argcnt;	// target is first arg
		    shift = 1;
		}
		// if vararg we add a final Object[] arg
		Object[] params = new Object[proto.vararg? (argcnt+1) : argcnt];
		for(int a = 0; a < argcnt; a++)
		    params[a] = proto.argTypes[a + shift]
			        .convertToObject(args[a + shift], focus, context);
		if(proto.vararg) {
		    int vcnt = args.length - argcnt - shift;
		    Item[] vargs = new Item[vcnt];
		    // eval and convert extra args into Item or null (type item?)
		    // like WrappedObjectType but treat also int double boolean
		    for(int va = 0; va < vcnt; va++) 
			vargs[va] =
			    args[argcnt + va].evalAsOptItem(focus, context);
		    params[argcnt] = vargs;
		}
		try {
		    Object result = proto.method.invoke(target, params);
		    if(trace) System.err.println("calling Java method: "
						 +proto.method+"\n for "+ proto);
		    return proto.returnType.convertFromObject(result);
		}
		catch (InvocationTargetException iex) {
		    //ex.printStackTrace();
		    Exception cause = (Exception) iex.getCause();
		    context.error(this,
		     new EvalException("exception in extension function: "+cause,
				       cause));
		}
		catch (Exception ex) {
		    context.error(this, new EvalException(
				      "invocation of extension function "+
				      proto.toString(context.getStaticContext())+
				      ": "+ex, ex));
		}
	    }
	    else {
		Object[] params = new Object[proto.argCnt];
		for(int a = 0; a < proto.argCnt; a++)
		    params[a] = proto.argTypes[a].convertToObject(args[a],
								  focus, context);
		try {
		    if(trace)
			System.err.println("calling Java constructor: "+
					   proto.constructor+
					   "\n for "+ proto);
		    Object result = proto.constructor.newInstance(params);
		    return proto.returnType.convertFromObject(result);
		}
		catch (Exception ex) {
		    //ex.printStackTrace();
		    context.error(this,
				  "invocation of extension constructor: "+ex);
		}
	    }
	    return Value.empty;
	}
    }
}
