/*
 *	Qizx/Open version 0.4
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */
package net.xfra.qizxopen.xquery.fn;

import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.fn.Function;
import net.xfra.qizxopen.xquery.op.Expression;
import net.xfra.qizxopen.util.QName;
import net.xfra.qizxopen.util.Namespace;
import net.xfra.qizxopen.util.NSPrefixMapping;

/**
 *	Description of the prototype of an operator or a function.
  */
public class Prototype 
{
    public QName qname;
    public Type  returnType;
    public Class classe;
    public QName[] argNames = new QName[3];
    public Type[]  argTypes = new Type[3];
    public int     argCnt = 0;
    public boolean vararg = false;	// allows more than argCnt arguments
    public boolean hidden = false;

    public Prototype(QName qname, Type returnType, Class classe, boolean vararg) {
	this.qname = qname;
	this.returnType = returnType;
	this.classe = classe;
	this.vararg = vararg;
    }

    public Prototype( QName qname, Type returnType, Class classe) {
	this(qname, returnType, classe, false);
    }

    public static Prototype op( String name, Type returnType, Class classe) {
	return new Prototype( QName.get(Namespace.OP, name), returnType, classe);
    }

    public static Prototype fn( String name, Type returnType, Class classe) {
	return new Prototype( QName.get(Namespace.FN, name), returnType, classe);
    }

    public static Prototype varfn( String name, Type returnType, Class classe) {
	return new Prototype( QName.get(Namespace.FN, name), returnType, classe,
			      true );
    }

    public static Prototype xs( String name, Type returnType, Class classe) {
	return new Prototype(QName.get(Namespace.XSD, name), returnType, classe);
    }

    public Prototype hidden() {
	this.hidden = true; return this;
    }

    public Prototype arg( String name, Type type) {
	return arg(QName.get(Namespace.NONE, name), type);
    }

    public Prototype arg( QName name, Type type) {
	if(argCnt >= argNames.length) {
	    QName[] oldn = argNames;
	    argNames = new QName[argNames.length + 3];
	    System.arraycopy(oldn, 0, argNames, 0, oldn.length);
	    Type[] oldt = argTypes;
	    argTypes = new Type[argTypes.length + 3];
	    System.arraycopy(oldt, 0, argTypes, 0, oldt.length);
	}
	argNames[argCnt] = name;
	argTypes[argCnt] = type;
	++ argCnt;
	return this;
    }

    // redefined in JavaFunction
    public Function.Call newInstance(StaticContext ctx,
				     Expression[] actualArguments) {
	try {
	    Function.Call fc = (Function.Call) classe.newInstance();
	    fc.prototype = this;
	    fc.args = actualArguments;
	    return fc;
	}
        catch (Exception e) {
            e.printStackTrace(); return null;
        }
    }

    public String getArgName( int arg, StaticContext ctx ) {
	return ctx.prefixedName(argNames[arg]);
    }

    public String displayName( StaticContext ctx ) {
	if(qname.getNamespace() == Namespace.OP)
	    return "operator "+ qname.getLocalName();
	else
	    return ctx.prefixedName(qname);
    }

    public String toString( EvalContext ctx ) {
	return toString(ctx.getStaticContext());
    }

    public String toString( StaticContext ctx ) {
	StringBuffer buf = new StringBuffer();
	buf.append( displayName(ctx) );
	buf.append(" ( ");
	for(int a = 0; a < argCnt; a++) {
	    if(a > 0) buf.append(", ");
	    buf.append('$');
	    buf.append(ctx.prefixedName(argNames[a]));
	    buf.append(" as ").append(argTypes[a].toString(ctx));
	}
	if(vararg)
	    buf.append("...");
	return buf.append(" ) as ").append(returnType.toString(ctx)).toString();
    }

    public String toString() {
	String prefix = NSPrefixMapping.std.convertToPrefix(qname.getNamespace());
	String name = (prefix == null) ? qname.toString()
	       : prefix + ":" + qname.getLocalName();

	StringBuffer buf = new StringBuffer( name + "( " );
	for(int a = 0; a < argCnt; a++) {
	    if(a > 0) buf.append(", ");
	    buf.append('$');
	    buf.append(argNames[a]).append(" as ").append(argTypes[a].toString());
	}
	if(vararg)
	    buf.append("...");
	return buf.append(" ) as ").append(returnType.toString()).toString();
    }

    /**
     * Tests an argument list for the corresponding formal argument types.
     * Raises no error.
     * TODO: fine argument matching returning distance instead of bool.
     */
    public boolean check( Expression[] actualArguments ) {
	if(actualArguments.length != argCnt &&
	    !(vararg && actualArguments.length >= argCnt) )
	    return false;
	// vararg extra arguments not checked
	for(int a = 0; a < argCnt; a++)
	    if(!checkArg( a, actualArguments[a] ))
		return false;
	return true;
    }

    /**
     *	Tests an expression for a formal argument type. Raises no error.
     */
    public boolean checkArg( int rank, Expression actualArgument ) {
	return checkArg( rank, actualArgument.getType() );
    }

    /**
     *	Tests an actual type for a formal argument type. Raises no error.
     */
    public boolean checkArg( int rank, Type actualType ) {
	
	return argTypes[rank].accepts(actualType);
    }

} // end of class Prototype

