/*
 *	Qizx/Open version 0.4
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.op;

import net.xfra.qizxopen.util.*;
import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.dm.*;

/**
 *  Static methods and objects for expression rewriting and transforming.
 *	
 */
public abstract class Expr extends Expression
{
    // extended 'self' can be SelfStep with wildcard
    // or a local VarRef with null vardecl (produced by rewriting)
    public static boolean isDot(Expression e) {
	if(e instanceof SelfStep && ((SelfStep) e).nodeTest == null)
	    return true;
	return (e instanceof VarReference.Local) && 
	       ((VarReference.Local) e).decl == null;
    }

    public static boolean pathLike(Expression e) {
	return (e instanceof PathExpr || e instanceof FilterExpr ||
		e instanceof BasicStep || isDot(e));
    }

    public static boolean positionalPredicate(Expression pred) {
	ItemType predType = pred.getType().getItemType();
	return (pred instanceof FilterExpr.PosTest) ||
	    Type.NUMERIC.isSuperType(predType) &&
	    (pred.findSubExpression( net.xfra.qizxopen.xquery.fn.Position.Exec.class ) != null ||
	     pred.findSubExpression( net.xfra.qizxopen.xquery.fn.Last.Exec.class ) != null ||
	     pred.findSubExpression( SelfStep.class ) != null );
    }

    // Filters a source expr by a predicate.
    // If source is a path, adds predicate to last step.
    //
    static Expression addPredicate( Expression source, Expression predicate ) {
	if(source instanceof NodeSortExpr) {
	    NodeSortExpr sort = (NodeSortExpr) source;
	    sort.expr = addPredicate(sort.expr, predicate);
	}
	else if(source instanceof PathExpr) {
	    PathExpr path = (PathExpr) source;
	    int rankLast = path.steps.length - 1;
	    path.steps[rankLast] = addPredicate(path.steps[rankLast], predicate);
	}
	else {
	    FilterExpr filt = new FilterExpr(source);
	    filt.addPredicate(predicate);
	    source = filt;
	}
	return source;
    }

    // scans an expression looking for local variables that are declared
    // in a 'for' clause. Fails if more than one.
    public static class ForVarCollector extends Visitor {
	LocalVariable found;
	LocalVariable found2;	// detects failure

	public boolean preTest(Expression focus) {
	    if(!(focus instanceof VarReference.Local))
		return true;
	    VarReference.Local local = (VarReference.Local) focus;
	    Expression owner = local.decl.owner;
	    if(!(owner instanceof ForClause))
		return true;
	    
	    if(found == null || found == local.decl) {
		found = local.decl;
		return true;
	    }
	    found2 = local.decl;
	    return false;
	}
    }

    static boolean depends(Expression expr, LocalVariable var) {
	return !new VarDependence(var).visit(expr);
    }

    public static class VarDependence extends Visitor {
	LocalVariable var;

	VarDependence(LocalVariable var) {
	    this.var = var;
	}

	public boolean preTest(Expression focus) {
	    if(!(focus instanceof VarReference.Local))
		return true;
	    VarReference.Local local = (VarReference.Local) focus;
	    return (local.decl != var);	// ie stop if found
	}
    }

    // detects a dependence to a variable such that it can be replaced by '.' if the
    // expression becomes a filter predicate. It means that the var reference cannot
    // itself be inside a predicate.
    public static class DottingVarFinder extends Visitor {
	LocalVariable var;
	boolean OK = false;

	DottingVarFinder(LocalVariable var) {
	    this.var = var;
	}

	public boolean preTest(Expression focus) {
	    if(!(focus instanceof VarReference.Local))
		return true;
	    VarReference.Local local = (VarReference.Local) focus;
	    if(local.decl != var)
		return true;
	    // check the context:
	    for(int p = ctxPtr; --p >= 0; )
		if(context[p] instanceof FilterExpr) {
		    // inside a predicate (bug? we dont know which child we are) TODO
		    return OK = false;
		}
	    return OK = true;
	}
    }

    public static class VarReplacer extends Visitor {
	LocalVariable replaced, replacing;

	VarReplacer(LocalVariable replaced, LocalVariable replacing) {
	    this.replaced = replaced;
	    this.replacing = replacing;
	}

	public boolean preTest(Expression focus) {
	    if(!(focus instanceof VarReference.Local))
		return true;
	    VarReference.Local local = (VarReference.Local) focus;
	    if(local.decl == replaced) {
		local.decl = replacing;
		if(replacing != null)
		    local.name = replacing.name;
	    }
	    return true;
	}
    }
} 
