/*
 *	Qizx/Open version 0.4
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.op;

import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.fn.Function;
import net.xfra.qizxopen.xquery.dt.SingleDecimal;
import net.xfra.qizxopen.xquery.fn.Prototype;
import java.math.BigDecimal;

/**
 *  Implementation of operator mod.
 * 
 */
public class ModOp extends NumericOp {

    public ModOp( Expression expr1, Expression expr2 ) {
        super( expr1, expr2 );
    }

    public void dump( ExprDump d ) {
	d.header( this, "mod" );
        d.display("expr1", operands[0]);
        d.display("expr2", operands[1]);
    }

    static Prototype[] protos = { 
        Prototype.op("mod", Type.INTEGER, ExecI.class)
	  .arg("op1", Type.INTEGER).arg("op2", Type.INTEGER),
        Prototype.op("mod", Type.DECIMAL, ExecDec.class)
	  .arg("op1", Type.DECIMAL).arg("op2", Type.DECIMAL),
        Prototype.op("mod", Type.FLOAT, ExecF.class)
	  .arg("op1", Type.FLOAT).arg("op2", Type.FLOAT),
        Prototype.op("mod", Type.DOUBLE, ExecD.class)
	  .arg("op1", Type.DOUBLE).arg("op2", Type.DOUBLE)
    };
    public Prototype[] getProtos() { return protos; }

    public static class ExecI extends Function.IntegerCall {

	public long evalAsInteger( Focus focus, EvalContext context ) throws XQueryException {
	    long e1 = args[0].evalAsInteger(focus, context);
	    long e2 = args[1].evalAsInteger(focus, context);
	    context.at(this);
	    if(e2 == 0)
		context.error(this, "division by zero");
	    return e1 % e2;
	}
    }

    public static class ExecDec extends Function.Call {

	public Value eval( Focus focus, EvalContext context ) throws XQueryException {
	    Value e1 = args[0].eval(focus, context);
	    Value e2 = args[1].eval(focus, context);
	    context.at(this);
	    BigDecimal a = e1.asDecimal(), b = e2.asDecimal();
	    BigDecimal q = a.divide( b, 0, BigDecimal.ROUND_DOWN );
	    return new SingleDecimal( a.subtract( q.multiply(b) ));
	}
    }

    public static class ExecF extends Function.FloatCall {

	public float evalAsFloat( Focus focus, EvalContext context ) throws XQueryException {
	    float e1 = args[0].evalAsFloat(focus, context);
	    float e2 = args[1].evalAsFloat(focus, context);
	    context.at(this);
	    return e1 % e2;
	}
    }

    public static class ExecD extends Function.DoubleCall {

	public double evalAsDouble( Focus focus, EvalContext context ) throws XQueryException {
	    double e1 = args[0].evalAsDouble(focus, context);
	    double e2 = args[1].evalAsDouble(focus, context);
	    context.at(this);
	    return e1 % e2;
	}
    }
}
