#include "../stdafx.h"
#include <cmath>
#include <vector>
#include <map>
#include <CString>
#include "CommandSerialResistor.h"
#include "dsCommandMessage_c.h"
#include "dsMathLine_c.h"

CommandSerialResistor::CommandSerialResistor()
: CommandBase(Options::instance().application)
, m_Tracker(0)
{    
}

CommandSerialResistor::~CommandSerialResistor()
{
}

ImplementTrackerUpdateNotifyHook(CommandSerialResistor)
bool CommandSerialResistor::ExecuteNotify()
{
    // Create Tracker and register it
    application->CreateTracker(&m_Tracker);
    RegisterTrackerUpdateNotifyHook(m_Tracker);

    // Get the command line interface.
    dsCommandMessage_ptr commandline;
    application->GetCommandMessage( &commandline );

    // Get Active document
    dsDocument_ptr doc;
    application->GetActiveDocument(&doc);

    // Prompt for selection
    dsSelectionManager_ptr selection_manager;
    doc->GetSelectionManager(&selection_manager);
    selection_manager->ClearSelections( dsSelectionSetType_Current );
    selection_manager->ClearSelections( dsSelectionSetType_Previous );

    // Set a selection filter to accept only block instances
    // (all electrical components are block instances)
    dsSelectionFilter_c* filter;
    selection_manager->GetSelectionFilter( &filter );
    filter->Clear();
    filter->AddEntityType( dsBlockInstanceType );
    filter->put_Active( true );

    // Prompt for selection
    dsString keyword;
    dsPromptResultType_e prompt_result;
    DSRESULT dsResult;
    while ( true )
    {
        dsResult = commandline->PromptForSelectionOrKeyword(
            false,
            L"Specify all electrical components",
            L"Invalid selection",
            dsStringArray(),
            dsStringArray(),
            0, &keyword, &prompt_result );
        if ( dsResult == DSRESULT_True && 
            ( prompt_result == dsPromptResultType_Cancel || 
            prompt_result == dsPromptResultType_Value || 
            prompt_result == dsPromptResultType_Keyword ) )
            break;
    }

    filter->put_Active( false );
    filter->Clear();

    dsLongArray types;
    dsObjectPtrArray objects;
    dsResult = selection_manager->GetSelectedObjects( dsSelectionSetType_Previous, &types, &objects );
    if ( 0 == objects.getSize() || dsResult != DSRESULT_True )
        return false;

    // The tracker shows all connections and the resistor
    // with its calculated resistance
    commandline->AddTracker( m_Tracker );
    DoSerialize( doc, objects, true );

    dsStringArray globalKeywords;
    globalKeywords.add(L"_Yes");
    globalKeywords.add(L"_No");
    dsStringArray localKeywords;
    localKeywords.add(L"Yes");
    localKeywords.add(L"No");
    
    // Prompt the user if they want to add the connections to the drawing
    commandline->PromptForKeyword2(
        L"Do you want to serialize the components in a circuit and add the resistor?",
        L"",
        globalKeywords,
        localKeywords,
        0,
        globalKeywords[0],
        &keyword,
        &prompt_result);
    if (prompt_result == dsPromptResultType_Cancel || prompt_result != dsPromptResultType_Keyword )
    {
        dsSimpleNote->put_Erased( true );
        commandline->RemoveTracker( m_Tracker );
        UnRegisterTrackerUpdateNotifyHook(m_Tracker);

        return true;
    }

    if (keyword.CompareNoCase(L"_YES") == 0)
    {
        DoSerialize( doc, objects );
    }
    else
    {
        dsSimpleNote->put_Erased( true );
    }
     
    // Remove tracker and unregister it
    commandline->RemoveTracker( m_Tracker );
    UnRegisterTrackerUpdateNotifyHook(m_Tracker);
    return true; 
}

void CommandSerialResistor::DoSerialize( dsDocument_c *doc, dsObjectPtrArray objects, bool TemporaryCreation )
{
    // Get Model
    dsModel_ptr model;
    doc->GetModel(&model);
    // Get Sketch Manager
    dsSketchManager_ptr sketch_manager;
    model->GetSketchManager(&sketch_manager);

    // Voltage of the power supply in [Volts]
    int iVoltage = 0;

    // The power supply is a block instance that contains
    // an attribute called "Voltage"
    dsObjectPtrArray objPower;

    // The array of LEDs which will be serialized. An LED is a block instance
    // that contains an attribute called "Serial" denoting its position in
    // the circuit
    std::map< int, dsObject*> ledArray;
    for(unsigned int i = 0; i < objects.getSize(); ++i)
    {
        dsBlockInstance_c* blkInst = (dsBlockInstance_c*)objects[i];
        dsObjectPtrArray attribute_instances, attribute_definitions;
        blkInst->GetAttributeInstances(&attribute_instances);
        dsBlockDefinition_ptr block_definition;
        blkInst->GetBlockDefinition( &block_definition );
        block_definition->GetAttributeDefinitions(&attribute_definitions);

        int size = attribute_instances.getSize();
        for (unsigned int j = 0; j < attribute_instances.getSize(); ++j)
        {
            dsAttributeInstance_c *attribute_instance = (dsAttributeInstance_c*)attribute_instances[j];
            dsAttributeDefinition_c *attribute_definition = get_attribute_definition( attribute_instance, attribute_definitions);
            if (!attribute_definition)
            {
                continue;
            }


            // "Voltage" -> assign to objPower
            // "Serial" -> add to ledArray
            dsString caption;
            attribute_definition->get_Caption(&caption);
            if ( caption.CompareNoCase(L"Voltage") == 0 )
            {
                objPower.add(objects[i]);
                dsString sValue;
                attribute_instance->get_Value( &sValue );
                iVoltage = d2i( sValue );
            }
            else if ( caption.CompareNoCase(L"Serial") == 0 )
            {
                dsString sValue;
                attribute_instance->get_Value( &sValue );
                int iKey = d2i( sValue );
                dsObject* obj = (dsObject*)objects[i] ;
                ledArray[iKey] = obj;
                break;
            }
        }
    }

    dsObjectPtrArray ledObjects;
    ledObjects.add((dsObject*)objPower[0]);
    for(auto const & item : ledArray)
    {
        dsObject* obj = item.second;
        if (!obj)
        {
            continue;
        }
        ledObjects.add(obj);
    }

    // Calculate the connection lines to make up the circuit
    dsMathUtility_c* pMathUtil = 0;
    application->GetMathUtility(&pMathUtil);
    bool bRegistorCreated = false;
    int size = ledObjects.getSize();
    for(unsigned int i = 0; i < size; ++i)
    {
        dsObject* obj = ledObjects[i];
        dsMathPoint_c *firstPt = 0, *secPt = 0, *nextFirstPt = 0, *nextSecPt = 0;
        pMathUtil->CreatePoint(0,0,0, &firstPt);
        pMathUtil->CreatePoint(0,0,0, &secPt);
        pMathUtil->CreatePoint(0,0,0, &nextFirstPt);
        pMathUtil->CreatePoint(0,0,0, &nextSecPt);
        getStartAndEndPts( doc, obj, firstPt, secPt );

        dsObject* objNext = 0;
        if( i == size-1 )
        {
            objNext = ledObjects[0];
            getStartAndEndPts( doc, objNext, nextFirstPt, nextSecPt );
        }
        else
        {
            objNext = ledObjects[i+1];
            getStartAndEndPts( doc, objNext, nextFirstPt, nextSecPt );
        }       

        dsBlockInstance_c* blkInst = (dsBlockInstance_c*)obj;
        dsBlockInstance_c* blkInstNext = (dsBlockInstance_c*)objNext;
        if( !bRegistorCreated )
        {
            // Take the current that would flow through the next LED,
            // reading its "Current" attribute. There is one caveat here:
            // In a serial circuit, the current stays the same, so the
            // assumption we make, is that every LED has the same value
            //' written in "Current".
            int iCurrent = getCurrent( objNext );
            // Create the resistor with R = iVoltage/iCurrent
            createResistor( sketch_manager, blkInst, blkInstNext, pMathUtil, secPt, nextFirstPt, iVoltage, iCurrent, TemporaryCreation ); //Resistor creation
        }
        if ( bRegistorCreated )
        {
            double rotation = 0.0, rotationNext = 0.0;
            blkInst->get_Rotation( &rotation );
            blkInstNext->get_Rotation( &rotationNext );
            rotation = rotation * 180 / 3.141592653589793238;
            rotationNext = rotationNext * 180 / 3.141592653589793238;
            double angle = rotationNext - rotation;
            angle = fabs(angle);
            createSerialization( sketch_manager, blkInst, blkInstNext, secPt, nextFirstPt, angle, TemporaryCreation );
        } 
        bRegistorCreated = true;        
    }
}

void CommandSerialResistor::createSerialization( dsSketchManager_ptr sketch_manager, dsBlockInstance_c* blkInst, dsBlockInstance_c* blkInstNext, 
     dsMathPoint_c *stPt, dsMathPoint_c *endPt, double angle, bool TemporaryCreation )
{
    if ( TemporaryCreation )
    {
        application->put_TemporaryEntityMode( true );
    }

    dsLine_ptr dsLine = 0;
    double stX, stY, stZ, endX, endY, endZ;
    stPt->GetPosition( &stX, &stY, &stZ );
    endPt->GetPosition( &endX, &endY, &endZ );
    if( angle == 90 || angle == 270 )
    {
        dsMathUtility_c* pMathUtil = 0;
        application->GetMathUtility(&pMathUtil);
        dsDocument_ptr doc;
        application->GetActiveDocument(&doc);
        dsMathPoint_c *firstPt = 0, *secPt = 0;
        pMathUtil->CreatePoint(0,0,0, &firstPt);
        pMathUtil->CreatePoint(0,0,0, &secPt);
        getStartAndEndPts( doc, blkInst, firstPt, secPt );
        double x1, y1, z1, x2, y2, z2; 
        firstPt->GetPosition( &x1, &y1, &z1 );
        secPt->GetPosition( &x2, &y2, &z2 );               
        dsMathLine_ptr lineForDir;
        pMathUtil->CreateLine( x1, y1, z1, x2, y2, z2, dsMathLineType_Bounded, &lineForDir );
        dsMathVector_ptr dirVec = 0;
        lineForDir->GetDirection( &dirVec );

        dsMathLine_ptr lineForLen = 0;
        pMathUtil->CreateLine( stX, stY, stZ, endX, endY, endZ, dsMathLineType_Bounded, &lineForLen );
        double length;
        lineForLen->GetLength( &length );
        dirVec->ScaleVector(length);
        dsMathPoint_c *firstArbitPt = addPointToVector( stPt, dirVec );

        stPt->GetPosition( &stX, &stY, &stZ ); 
        firstArbitPt->GetPosition( &endX, &endY, &endZ );
        dsMathLine_ptr lineForClosestPt = 0;
        pMathUtil->CreateLine( stX, stY, stZ, endX, endY, endZ, dsMathLineType_Bounded, &lineForClosestPt );
        dsMathPoint_c *intSecPt;
        lineForClosestPt->GetClosestPointTo( endPt, &intSecPt );

        dsLine_ptr dsLine1 = 0, dsLine2 = 0;
        intSecPt->GetPosition( &endX, &endY, &endZ );
        sketch_manager->InsertLine( stX, stY, stZ, endX, endY, endZ, &dsLine1 );
        intSecPt->GetPosition( &stX, &stY, &stZ );
        endPt->GetPosition( &endX, &endY, &endZ );
        sketch_manager->InsertLine( stX, stY, stZ, endX, endY, endZ, &dsLine2 );
        if ( TemporaryCreation )
        {
            m_Tracker->AddTemporaryEntity( dsLine1 );
            m_Tracker->AddTemporaryEntity( dsLine2 );
            dsColor_ptr dsColor;
            dsLine1->get_Color( &dsColor );
            dsColor->SetNamedColor( dsNamedColor_Green );
            dsLine1->put_Color( dsColor );
            dsLine2->put_Color( dsColor );
            application->put_TemporaryEntityMode( false );
        }
    }
    else
    {
        sketch_manager->InsertLine( stX, stY, stZ, endX, endY, endZ, &dsLine );
        if ( TemporaryCreation )
        {
            m_Tracker->AddTemporaryEntity( dsLine );
            dsColor_ptr dsColor;
            dsLine->get_Color( &dsColor );
            dsColor->SetNamedColor( dsNamedColor_Green );
            dsLine->put_Color( dsColor );
            application->put_TemporaryEntityMode( false );
        }
    }    
}

void CommandSerialResistor::createResistor( dsSketchManager_ptr sketch_manager, dsBlockInstance_c* blkInst, dsBlockInstance_c* blkInstNext, dsMathUtility_c* pMathUtil, dsMathPoint_c *oldSecPt, 
                                           dsMathPoint_c*firstPt, int iVoltage, int iCurrent, bool TemporaryCreation )
{
    // This function calculates the resistance and creates a resistor block and
    // a simple note showing the resistance value in [Ohms].
    if ( dsSimpleNote && TemporaryCreation == false )
    {
        dsSimpleNote->put_Erased( true );
    }
    if ( TemporaryCreation )
    {
        application->put_TemporaryEntityMode( true );
    }
    dsMathPoint_c *firstMidPt = getMidPoint( pMathUtil, oldSecPt, firstPt );
    dsMathPoint_c *secMidPt = getMidPoint( pMathUtil, oldSecPt, firstMidPt );
    dsMathPoint_c *thirdMidPt = getMidPoint( pMathUtil, firstMidPt, firstPt );
    double x1, y1, z1, x2, y2, z2;
    secMidPt->GetPosition( &x1, &y1, &z1 );
    thirdMidPt->GetPosition( &x2, &y2, &z2 );
    dsMathLine_c* dsMathLine = 0;
    pMathUtil->CreateLine( x1, y1, z1, x2, y2, z2, dsMathLineType_Bounded, &dsMathLine );
    double length;
    dsMathLine->GetLength( &length );
    length = length / 2;
    dsDoubleArray coordinates;
    coordinates.add( x1 );
    coordinates.add( y1 + length / 2 );
    coordinates.add( x2 );
    coordinates.add( y2 + length / 2 );
    coordinates.add( x2 );
    coordinates.add( y2 - length / 2 );
    coordinates.add( x1 );
    coordinates.add( y1 - length / 2 );
    dsPolyLine_ptr dsPolyLine;
    sketch_manager->InsertPolyline2D( coordinates, true, &dsPolyLine );    
    createSerialization( sketch_manager, blkInst, blkInstNext, oldSecPt, secMidPt, 0, TemporaryCreation );
    createSerialization( sketch_manager, blkInst, blkInstNext, thirdMidPt, firstPt, 0, TemporaryCreation );
    double x, y, z;
    firstMidPt->GetPosition( &x, &y, &z );
    int iResist = iVoltage / ( iCurrent * 0.001 );
    dsString sResist = getResistance( iResist );
    sketch_manager->InsertSimpleNote( x, y, z, length/2, 0, sResist, &dsSimpleNote );
    dsSimpleNote->put_Justify( dsTextJustification_MiddleCenter );

    //dsObjectPtrArray entArr;
    //entArr.add( dsPolyLine );
    //entArr.add( dsSimpleNote );
    //dsHatch_ptr dsHatch;
    //sketch_manager->InsertHatchByEntities( entArr, L"SOLID", 1, 0, &dsHatch );
    //m_Tracker->AddTemporaryEntity( dsHatch );
    //entArr.add( dsHatch );
    //dsDocument_ptr dsDoc;
    //application->GetActiveDocument( &dsDoc );
    //dsLongArray entTypeArr;
    //entTypeArr.add( dsPolyLineType );
    //entTypeArr.add( dsSimpleNoteType );
    //entTypeArr.add( dsHatchType );
    /*dsBlockDefinition_ptr dsBlockDef;
    dsDoc->CreateBlockDefinition( L"Resistor", L"Resistor of Circuit", x2, y2, z2, entTypeArr, entArr, 
        dsBlockDefinitionEntities_ConvertToBlock, &dsBlockDef );
    dsBlockDef->GetBlockInstances(&m_EntityArray);
    dsBlockInstance_c *blkinst = (dsBlockInstance_c*)m_EntityArray[0];
    m_Tracker->AddTemporaryEntity( dsBlockDef );
    m_Tracker->AddTemporaryEntity( blkinst );*/

    if ( TemporaryCreation )
    {
        m_Tracker->AddTemporaryEntity( dsPolyLine );
        m_Tracker->AddTemporaryEntity( dsSimpleNote );
        //m_EntityArray.add(dsPolyLine);
        //m_EntityArray.add(dsSimpleNote);
        dsColor_ptr dsColor;
        dsPolyLine->get_Color( &dsColor );
        dsColor->SetNamedColor( dsNamedColor_Green );
        dsPolyLine->put_Color( dsColor );
        dsSimpleNote->put_Color( dsColor );
        application->put_TemporaryEntityMode( false );
    }
}

void CommandSerialResistor::getStartAndEndPts( dsDocument_c *doc, dsObject* obj, dsMathPoint_c *firstPt, dsMathPoint_c*secPt )
{
    dsBlockInstance_c* blkInst = (dsBlockInstance_c*)obj;
    dsBlockDefinition_c *blkDef;
    blkInst->GetBlockDefinition( &blkDef );
    dsLongArray types;
    dsObjectPtrArray objects;
    blkDef->GetEntities(&types, &objects);
    double x1, y1, z1, x2, y2, z2;
    for(unsigned int i = 0; i < objects.getSize(); ++i)
    {
        dsObjectType_e type;
        dsObject * object = objects[i];
        type = (dsObjectType_e)types[i];
        if ( type == dsLineType )
        {
            dsLine_ptr dsLine( (dsLine_c*)object );
            dsLine->GetFirstPoint( &x1, &y1, &z1 );
            dsLine->GetSecondPoint( &x2, &y2, &z2 );
        }
    }
    dsMathUtility_c* pMathUtil = 0;
    application->GetMathUtility(&pMathUtil);
    dsMathLine_c* dsMathLine = 0;
    dsMathLineType_e Linetype;
    pMathUtil->CreateLine( x1, y1, z1, x2, y2, z2, dsMathLineType_Bounded, &dsMathLine );

    double x,y,z;
    blkInst->GetPosition( &x, &y, &z);
    dsMathVector_c* pPosVector;
    pMathUtil->CreateVector( x, y, z, &pPosVector );

    //Translate 'ptLowerRightInner' to the old-block insertion-point that is 'pPosVector'
    dsMathTransform_c* pPosMat;
    pMathUtil->CreateTransformTranslation( pPosVector, &pPosMat );
    dsMathLine->TransformBy( pPosMat );

    //Now transform the 'ptLowerRightInner' with scale matrix
    dsMathPoint_c *pScalePt;
    pMathUtil->CreatePoint( x, y, z, &pScalePt );
    dsMathTransform_c* pScaleMat;
    pMathUtil->CreateTransformScaling( pScalePt, 1, &pScaleMat );
    dsMathLine->TransformBy( pScaleMat );

    //Now transform the 'ptLowerRightInner' with rotate matrix
    dsMathTransform_c* pRotMat;
    double rotation = 0.0;
    blkInst->get_Rotation( &rotation );
    dsCustomCoordinateSystemManager_c* pCoordinateManger = 0;
    doc->GetCustomCoordinateSystemManager(&pCoordinateManger);
    dsCustomCoordinateSystem_c* pCordinate = NULL;
    pCoordinateManger->GetActiveCustomCoordinateSystem(&pCordinate);
    dsMathVector_c* pZAxisVector = NULL;
    pCordinate->GetZAxisDirection(&x, &y, &z);
    pMathUtil->CreateVector(x,y,z,&pZAxisVector);
    pMathUtil->CreateTransformRotation( pScalePt, pZAxisVector, rotation, &pRotMat );
    dsMathLine->TransformBy( pRotMat );

    dsMathPoint_c *st, *end;
    dsMathLine->get_StartPoint( &st );
    dsMathLine->get_EndPoint( &end );    
    st->GetPosition( &x1, &y1, &z1 );
    end->GetPosition( &x2, &y2, &z2 );
    firstPt->SetPosition( x1, y1, z1 );
    secPt->SetPosition( x2, y2, z2 );
}

int CommandSerialResistor::getCurrent( dsObject* obj )
{
    // Read the attribute "Current" from *obj*, returning its value as integer
    int iCurrent = 0;
    dsBlockInstance_c* blkInst = (dsBlockInstance_c*)obj;
    dsObjectPtrArray attribute_instances, attribute_definitions;
    blkInst->GetAttributeInstances(&attribute_instances);
    dsBlockDefinition_ptr block_definition;
    blkInst->GetBlockDefinition( &block_definition );
    block_definition->GetAttributeDefinitions(&attribute_definitions);
    for (unsigned int j = 0; j < attribute_instances.getSize(); ++j)
    {
        dsAttributeInstance_c *attribute_instance = (dsAttributeInstance_c*)attribute_instances[j];
        dsAttributeDefinition_c *attribute_definition = get_attribute_definition( attribute_instance, attribute_definitions);
        if (!attribute_definition)
        {
            continue;
        }
        dsString caption;
        attribute_definition->get_Caption(&caption);
        if ( caption.CompareNoCase(L"Current") == 0 )
        {
            dsString sValue;
            attribute_instance->get_Value( &sValue );
            iCurrent = d2i( sValue );
            return iCurrent;
        }
    }
    return iCurrent;
}

bool CommandSerialResistor::UpdateNotify(dsMathPoint_c *CursorPosition)
{
    // This function gets called every time the tracker is updated, i.e. the
    // mouse was moved. Since the temporary circuit does not depend on the
    // mouse, this function contains no code.
    return true;
}

dsMathPoint_c* CommandSerialResistor::addPointToVector(dsMathPoint_c* pPoint, dsMathVector_c* pMathVector)
{
    double vX, vY, vZ;
	pMathVector->GetCoordinates(&vX, &vY, &vZ);

	double pX, pY, pZ;
	pPoint->GetPosition(&pX, &pY, &pZ);

	return createPoint(pX + vX, pY + vY, pZ + vZ);
}

dsMathPoint_c* CommandSerialResistor::createPoint(double x, double y, double z)
{
    dsMathUtility_c* pMathUtil = 0;
    application->GetMathUtility(&pMathUtil);
    dsMathPoint_c* pPoint = NULL;
    pMathUtil->CreatePoint(x, y, z, &pPoint);
    return pPoint;
}
