/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */
import * as model from "./model";
import { mqtt } from "aws-crt";
export { model };
/**
 * @module aws-iot-device-sdk
 * @category IotIdentity
 */
export declare class IotIdentityError extends Error {
    readonly payload?: string | ArrayBuffer | ArrayBufferView | Record<string, unknown> | undefined;
    prototype: any;
    constructor(message?: string, payload?: string | ArrayBuffer | ArrayBufferView | Record<string, unknown> | undefined);
}
/**
 * API Documentation: https://docs.aws.amazon.com/iot/latest/developerguide/provision-wo-cert.html
 *
 * @module aws-iot-device-sdk
 * @category IotIdentity
 */
export declare class IotIdentityClient {
    private connection;
    private decoder;
    constructor(connection: mqtt.MqttClientConnection);
    /**
     * API Documentation: https://docs.aws.amazon.com/iot/latest/developerguide/provision-wo-cert.html#fleet-provision-api
     * Publish CreateKeysAndCertificate message
     * If the device is offline, the PUBLISH packet will be sent once the connection resumes.
     *
     * @param request Message to be serialized and sent
     * @param qos Quality of Service for delivering this message
     * @returns Promise which returns a `mqtt.MqttRequest` which will contain the packet id of
     *          the PUBLISH packet.
     *
     * * For QoS 0, completes as soon as the packet is sent.
     * * For QoS 1, completes when PUBACK is received.
     * * QoS 2 is not supported by AWS IoT.
     *
     * @category IotIdentity
     */
    publishCreateKeysAndCertificate(request: model.CreateKeysAndCertificateRequest, qos: mqtt.QoS): Promise<mqtt.MqttRequest>;
    /**
     * API Documentation: https://docs.aws.amazon.com/iot/latest/developerguide/provision-wo-cert.html#fleet-provision-api
     *
     * Subscribe to CreateKeysAndCertificateAccepted messages
     *
     * subscribeToCreateKeysAndCertificateAccepted may be called while the device is offline, though the async
     * operation cannot complete successfully until the connection resumes.
     *
     * Once subscribed, `messageHandler` is invoked each time a message matching
     * the `topic` is received. It is possible for such messages to arrive before
     * the SUBACK is received.
     *
     * @param request Subscription request configuration
     * @param qos Maximum requested QoS that server may use when sending messages to the client.
     *            The server may grant a lower QoS in the SUBACK
     * @param messageHandler Callback invoked when message or error is received from the server.
     * @returns Promise which returns a `mqtt.MqttSubscribeRequest` which will contain the
     *          result of the SUBSCRIBE. The Promise resolves when a SUBACK is returned
     *          from the server or is rejected when an exception occurs.
     *
     * @category IotIdentity
     */
    subscribeToCreateKeysAndCertificateAccepted(request: model.CreateKeysAndCertificateSubscriptionRequest, qos: mqtt.QoS, messageHandler: (error?: IotIdentityError, response?: model.CreateKeysAndCertificateResponse) => void): Promise<mqtt.MqttSubscribeRequest>;
    /**
     * API Documentation: https://docs.aws.amazon.com/iot/latest/developerguide/provision-wo-cert.html#fleet-provision-api
     *
     * Subscribe to CreateKeysAndCertificateRejected messages
     *
     * subscribeToCreateKeysAndCertificateRejected may be called while the device is offline, though the async
     * operation cannot complete successfully until the connection resumes.
     *
     * Once subscribed, `messageHandler` is invoked each time a message matching
     * the `topic` is received. It is possible for such messages to arrive before
     * the SUBACK is received.
     *
     * @param request Subscription request configuration
     * @param qos Maximum requested QoS that server may use when sending messages to the client.
     *            The server may grant a lower QoS in the SUBACK
     * @param messageHandler Callback invoked when message or error is received from the server.
     * @returns Promise which returns a `mqtt.MqttSubscribeRequest` which will contain the
     *          result of the SUBSCRIBE. The Promise resolves when a SUBACK is returned
     *          from the server or is rejected when an exception occurs.
     *
     * @category IotIdentity
     */
    subscribeToCreateKeysAndCertificateRejected(request: model.CreateKeysAndCertificateSubscriptionRequest, qos: mqtt.QoS, messageHandler: (error?: IotIdentityError, response?: model.ErrorResponse) => void): Promise<mqtt.MqttSubscribeRequest>;
    /**
     * API Documentation: https://docs.aws.amazon.com/iot/latest/developerguide/provision-wo-cert.html#fleet-provision-api
     *
     * Subscribe to RegisterThingRejected messages
     *
     * subscribeToRegisterThingRejected may be called while the device is offline, though the async
     * operation cannot complete successfully until the connection resumes.
     *
     * Once subscribed, `messageHandler` is invoked each time a message matching
     * the `topic` is received. It is possible for such messages to arrive before
     * the SUBACK is received.
     *
     * @param request Subscription request configuration
     * @param qos Maximum requested QoS that server may use when sending messages to the client.
     *            The server may grant a lower QoS in the SUBACK
     * @param messageHandler Callback invoked when message or error is received from the server.
     * @returns Promise which returns a `mqtt.MqttSubscribeRequest` which will contain the
     *          result of the SUBSCRIBE. The Promise resolves when a SUBACK is returned
     *          from the server or is rejected when an exception occurs.
     *
     * @category IotIdentity
     */
    subscribeToRegisterThingRejected(request: model.RegisterThingSubscriptionRequest, qos: mqtt.QoS, messageHandler: (error?: IotIdentityError, response?: model.ErrorResponse) => void): Promise<mqtt.MqttSubscribeRequest>;
    /**
     * API Documentation: https://docs.aws.amazon.com/iot/latest/developerguide/provision-wo-cert.html#fleet-provision-api
     *
     * Subscribe to CreateCertificateFromCsrAccepted messages
     *
     * subscribeToCreateCertificateFromCsrAccepted may be called while the device is offline, though the async
     * operation cannot complete successfully until the connection resumes.
     *
     * Once subscribed, `messageHandler` is invoked each time a message matching
     * the `topic` is received. It is possible for such messages to arrive before
     * the SUBACK is received.
     *
     * @param request Subscription request configuration
     * @param qos Maximum requested QoS that server may use when sending messages to the client.
     *            The server may grant a lower QoS in the SUBACK
     * @param messageHandler Callback invoked when message or error is received from the server.
     * @returns Promise which returns a `mqtt.MqttSubscribeRequest` which will contain the
     *          result of the SUBSCRIBE. The Promise resolves when a SUBACK is returned
     *          from the server or is rejected when an exception occurs.
     *
     * @category IotIdentity
     */
    subscribeToCreateCertificateFromCsrAccepted(request: model.CreateCertificateFromCsrSubscriptionRequest, qos: mqtt.QoS, messageHandler: (error?: IotIdentityError, response?: model.CreateCertificateFromCsrResponse) => void): Promise<mqtt.MqttSubscribeRequest>;
    /**
     * API Documentation: https://docs.aws.amazon.com/iot/latest/developerguide/provision-wo-cert.html#fleet-provision-api
     * Publish RegisterThing message
     * If the device is offline, the PUBLISH packet will be sent once the connection resumes.
     *
     * @param request Message to be serialized and sent
     * @param qos Quality of Service for delivering this message
     * @returns Promise which returns a `mqtt.MqttRequest` which will contain the packet id of
     *          the PUBLISH packet.
     *
     * * For QoS 0, completes as soon as the packet is sent.
     * * For QoS 1, completes when PUBACK is received.
     * * QoS 2 is not supported by AWS IoT.
     *
     * @category IotIdentity
     */
    publishRegisterThing(request: model.RegisterThingRequest, qos: mqtt.QoS): Promise<mqtt.MqttRequest>;
    /**
     * API Documentation: https://docs.aws.amazon.com/iot/latest/developerguide/provision-wo-cert.html#fleet-provision-api
     *
     * Subscribe to RegisterThingAccepted messages
     *
     * subscribeToRegisterThingAccepted may be called while the device is offline, though the async
     * operation cannot complete successfully until the connection resumes.
     *
     * Once subscribed, `messageHandler` is invoked each time a message matching
     * the `topic` is received. It is possible for such messages to arrive before
     * the SUBACK is received.
     *
     * @param request Subscription request configuration
     * @param qos Maximum requested QoS that server may use when sending messages to the client.
     *            The server may grant a lower QoS in the SUBACK
     * @param messageHandler Callback invoked when message or error is received from the server.
     * @returns Promise which returns a `mqtt.MqttSubscribeRequest` which will contain the
     *          result of the SUBSCRIBE. The Promise resolves when a SUBACK is returned
     *          from the server or is rejected when an exception occurs.
     *
     * @category IotIdentity
     */
    subscribeToRegisterThingAccepted(request: model.RegisterThingSubscriptionRequest, qos: mqtt.QoS, messageHandler: (error?: IotIdentityError, response?: model.RegisterThingResponse) => void): Promise<mqtt.MqttSubscribeRequest>;
    /**
     * API Documentation: https://docs.aws.amazon.com/iot/latest/developerguide/provision-wo-cert.html#fleet-provision-api
     *
     * Subscribe to CreateCertificateFromCsrRejected messages
     *
     * subscribeToCreateCertificateFromCsrRejected may be called while the device is offline, though the async
     * operation cannot complete successfully until the connection resumes.
     *
     * Once subscribed, `messageHandler` is invoked each time a message matching
     * the `topic` is received. It is possible for such messages to arrive before
     * the SUBACK is received.
     *
     * @param request Subscription request configuration
     * @param qos Maximum requested QoS that server may use when sending messages to the client.
     *            The server may grant a lower QoS in the SUBACK
     * @param messageHandler Callback invoked when message or error is received from the server.
     * @returns Promise which returns a `mqtt.MqttSubscribeRequest` which will contain the
     *          result of the SUBSCRIBE. The Promise resolves when a SUBACK is returned
     *          from the server or is rejected when an exception occurs.
     *
     * @category IotIdentity
     */
    subscribeToCreateCertificateFromCsrRejected(request: model.CreateCertificateFromCsrSubscriptionRequest, qos: mqtt.QoS, messageHandler: (error?: IotIdentityError, response?: model.ErrorResponse) => void): Promise<mqtt.MqttSubscribeRequest>;
    /**
     * API Documentation: https://docs.aws.amazon.com/iot/latest/developerguide/provision-wo-cert.html#fleet-provision-api
     * Publish CreateCertificateFromCsr message
     * If the device is offline, the PUBLISH packet will be sent once the connection resumes.
     *
     * @param request Message to be serialized and sent
     * @param qos Quality of Service for delivering this message
     * @returns Promise which returns a `mqtt.MqttRequest` which will contain the packet id of
     *          the PUBLISH packet.
     *
     * * For QoS 0, completes as soon as the packet is sent.
     * * For QoS 1, completes when PUBACK is received.
     * * QoS 2 is not supported by AWS IoT.
     *
     * @category IotIdentity
     */
    publishCreateCertificateFromCsr(request: model.CreateCertificateFromCsrRequest, qos: mqtt.QoS): Promise<mqtt.MqttRequest>;
}
